package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.coremon.api.task.DeleteMetricsTerminateProgress.CleanUpNonExistingShardProgress;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class CleanUpNonExistingShard implements AutoCloseable {

    private final RetryContext retryCtx;

    private final DeletedMetricsDao deletedMetricsDao;

    private final DeleteMetricsParams params;

    private final AtomicReference<CleanUpNonExistingShardProgress> progress;

    private final AtomicInteger total;
    private final AtomicInteger deleted;

    CleanUpNonExistingShard(
        RetryConfig retry,
        DeletedMetricsDao deletedMetricsDao,
        DeleteMetricsParams params,
        CleanUpNonExistingShardProgress progress)
    {
        this.retryCtx = new RetryContext(retry);
        this.deletedMetricsDao = deletedMetricsDao;
        this.params = params;
        this.progress = new AtomicReference<>(progress);

        this.total = new AtomicInteger(progress.getTotalMetrics());
        this.deleted = new AtomicInteger(progress.getDeletedMetrics());
    }

    public CompletableFuture<Void> start() {
        try {
            return tryStart();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public CleanUpNonExistingShardProgress progress() {
        return progress.get();
    }

    private CompletableFuture<Void> tryStart() {
        if (progress().getComplete()) {
            return completedFuture(null);
        }

        return updateMetricCount()
            .thenCompose(i -> cleanUp())
            .thenCompose(i -> updateMetricCount())
            .whenComplete((i, t) -> updateProgress(t == null));
    }

    private CompletableFuture<Void> updateMetricCount() {
        return getMetricCount().thenAccept(count -> {
            total.set(Math.max(total.get(), Math.toIntExact(count)));
            deleted.set(total.get() - Math.toIntExact(count));
            updateProgress(false);
        });
    }

    private CompletableFuture<?> cleanUp() {
        var doneFuture = new CompletableFuture<>();
        continueCleanUp(doneFuture);
        return doneFuture;
    }

    private void continueCleanUp(CompletableFuture<?> doneFuture) {
        deleteBatch().whenComplete((count, thr) -> {
            if (thr != null) {
                doneFuture.completeExceptionally(thr);
                return;
            }

            try {
                if (count > 0) {
                    deleted.addAndGet(Math.toIntExact(count));
                    updateProgress(false);
                    continueCleanUp(doneFuture);
                } else {
                    doneFuture.complete(null);
                }
            } catch (Throwable t) {
                doneFuture.completeExceptionally(t);
            }
        });
    }

    private CompletableFuture<Long> getMetricCount() {
        return retry(() -> deletedMetricsDao.count(params.getOperationId(), params.getNumId()));
    }

    private CompletableFuture<Long> deleteBatch() {
        return retry(() -> deletedMetricsDao.deleteBatch(params.getOperationId(), params.getNumId()));
    }

    private void updateProgress(boolean complete) {
        CleanUpNonExistingShardProgress prev;
        CleanUpNonExistingShardProgress update;
        do {
            prev = progress.get();
            var total = this.total.get();
            var deleted = this.deleted.get();
            update = CleanUpNonExistingShardProgress.newBuilder()
                .setComplete(complete || prev.getComplete())
                .setTotalMetrics(total)
                .setDeletedMetrics(deleted)
                .setProgress(total > 0 ? deleted / (double) total : 1)
                .build();
        } while (!progress.compareAndSet(prev, update));
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryCtx.retry(supplier);
    }

    @Override
    public void close() {
        retryCtx.close();
    }
}
