package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Message;
import io.grpc.Status;
import io.grpc.protobuf.StatusProto;

import ru.yandex.coremon.api.task.DeleteMetricsCheckProgress;
import ru.yandex.coremon.api.task.DeleteMetricsCheckResult;
import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.solomon.core.conf.ShardConfMaybeWrong;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.tasks.AbstractTask;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.StockpileClient;

import static java.lang.System.currentTimeMillis;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.TimeUnit.MINUTES;
import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class DeleteMetricsCheckTask extends AbstractTask<DeleteMetricsCheckProgress> {

    private final DeleteMetricsTaskMetrics metrics;
    private final SolomonConfHolder confHolder;
    private final DeleteMetricsParams params;

    private final boolean interrupted;
    private final CheckNoRecentWrites checkNoRecentWrites;

    DeleteMetricsCheckTask(
        RetryConfig retryConfig,
        ExecutionContext context,
        DeleteMetricsParams params,
        DeleteMetricsCheckProgress progress,
        Executor executor,
        ScheduledExecutorService timer,
        DeleteMetricsTaskMetrics metrics,
        SolomonConfHolder confHolder,
        StockpileClient stockpileClient,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver)
    {
        super(
            DeleteMetricsCheckTaskHandler.TYPE,
            retryConfig,
            context,
            params,
            progress,
            executor,
            timer);

        var selectors = ShardSelectors.withoutShardKey(Selectors.parse(params.getSelectors()));

        this.metrics = metrics;
        this.confHolder = confHolder;
        this.params = params;

        this.interrupted = progress.getInterrupted();
        this.checkNoRecentWrites = new CheckNoRecentWrites(
            retryConfig(),
            stockpileClient,
            shardResolver,
            executor,
            params,
            selectors,
            progress.getCheckNoRecentWrites());
    }

    @Override
    protected CompletableFuture<Void> onStart() {
        return interrupted
            ? fail(Status.CANCELLED.withDescription("interruption requested"))
            : checkNoRecentWrites();
    }

    private CompletableFuture<Void> checkNoRecentWrites() {
        return checkNoRecentWrites.start()
            .thenCompose(i -> forceSave())
            .thenCompose(i -> onCompleteCheckNoRecentWrites());
    }

    private CompletableFuture<Void> onCompleteCheckNoRecentWrites() {
        var progress = checkNoRecentWrites.progress();
        if (!progress.getComplete()) {
            return checkShardExistence();
        }

        var recentWriteLabels = progress.getRecentWriteLabels();
        if (!recentWriteLabels.isEmpty()) {
            return fail(
                Status.FAILED_PRECONDITION.withDescription("metric has recent writes: " + recentWriteLabels));
        }

        return completedFuture(null);
    }

    private CompletableFuture<Void> checkShardExistence() {
        return shardStillExists()
            ? reschedule(currentTimeMillis() + MINUTES.toMillis(5))
            : completedFuture(null);
    }

    private boolean shardStillExists() {
        return getShardConfOrNull() != null;
    }

    @Nullable
    private ShardConfMaybeWrong getShardConfOrNull() {
        return confHolder.getConfOrThrow().getShardByNumIdOrNull(params.getNumId());
    }

    @Override
    protected DeleteMetricsCheckProgress onUpdateProgress(DeleteMetricsCheckProgress prev, @Nullable Status status) {
        var update = prev.toBuilder()
            .setCheckNoRecentWrites(checkNoRecentWrites.progress());

        if (status != null) {
            if (status.isOk()) {
                update.clearAttempt();
                update.clearStatus();
            } else {
                update.setAttempt(prev.getAttempt() + 1);
                update.setStatus(StatusProto.fromStatusAndTrailers(status, null));
            }
        }

        return update.build();
    }

    @Override
    protected DeleteMetricsCheckResult result(DeleteMetricsCheckProgress progress) {
        return DeleteMetricsCheckResult.newBuilder()
            .setMinLastPointSeconds(progress.getCheckNoRecentWrites().getMinLastPointSeconds())
            .build();
    }

    @Override
    protected void afterComplete(Message result) {
        var shardConf = getShardConfOrNull();
        if (shardConf != null) {
            metrics.checkComplete(
                shardConf.getRaw().getProjectId(),
                shardConf.getId(),
                millisecondsToSeconds(currentTimeMillis() - params.getSubTaskCreatedAt()));
        }
    }

    @Override
    protected int attempt(DeleteMetricsCheckProgress progress) {
        return progress.getAttempt();
    }

    @Override
    protected void onClose() {
        checkNoRecentWrites.close();
    }

}
