package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.Message;
import io.grpc.Status;
import io.grpc.protobuf.StatusProto;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.coremon.api.task.DeleteMetricsMoveProgress;
import ru.yandex.coremon.api.task.DeleteMetricsMoveResult;
import ru.yandex.coremon.api.task.DeleteMetricsMoveResultOrBuilder;
import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.solomon.core.conf.ShardConfMaybeWrong;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.tasks.AbstractTask;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.lang.System.currentTimeMillis;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.TimeUnit.MINUTES;
import static ru.yandex.solomon.util.time.DurationUtils.randomize;
import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class DeleteMetricsMoveTask extends AbstractTask<DeleteMetricsMoveProgress> {

    private static final Logger logger = LoggerFactory.getLogger(DeleteMetricsMoveTask.class);

    private final DeleteMetricsTaskMetrics metrics;
    private final LocksDao locksDao;
    private final SolomonConfHolder confHolder;
    private final DeleteMetricsParams params;

    private final boolean interrupted;
    private final MoveToDeletedMetrics moveToDeletedMetrics;
    private final ReloadShard reloadShard;
    private final RepairDeletedMetrics repairDeletedMetrics;

    private volatile String lockId;

    DeleteMetricsMoveTask(
        RetryConfig retryConfig,
        ExecutionContext context,
        DeleteMetricsParams params,
        DeleteMetricsMoveProgress progress,
        Executor executor,
        ScheduledExecutorService timer,
        DeleteMetricsTaskMetrics metrics,
        LocksDao locksDao,
        MetricsDaoFactory metricsDaoFactory,
        SolomonConfHolder confHolder,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        DeletedMetricsDao deletedMetricsDao)
    {
        super(
            DeleteMetricsMoveTaskHandler.TYPE,
            retryConfig,
            context,
            params,
            progress,
            executor,
            timer);

        var selectors = ShardSelectors.withoutShardKey(Selectors.parse(params.getSelectors()));

        this.metrics = metrics;
        this.locksDao = locksDao;
        this.confHolder = confHolder;
        this.params = params;

        this.interrupted = progress.getInterrupted();
        this.moveToDeletedMetrics = new MoveToDeletedMetrics(
            retryConfig(),
            deletedMetricsDao,
            metricsDaoFactory,
            shardResolver,
            executor,
            params,
            selectors,
            progress.getMoveToDeletedMetrics());
        this.reloadShard = new ReloadShard(
            shardResolver,
            params,
            progress.getReloadShard());
        this.repairDeletedMetrics = new RepairDeletedMetrics(
            retryConfig(),
            deletedMetricsDao,
            shardResolver,
            executor,
            params);
    }

    @Override
    protected CompletableFuture<Void> onStart() {
        return lockShard();
    }

    private CompletableFuture<Void> lockShard() {
        var lockId = toLockId(params.getNumId());
        var operationId = params.getOperationId();
        var expiredAt = Instant.now().plus(Duration.ofDays(30));
        return locksDao.acquireLock(lockId, operationId, expiredAt)
            .thenCompose(lock -> {
                if (!operationId.equals(lock.owner())) {
                    logger.debug("shard lock acquired by other operation: this={} lock={}", operationId, lock);
                    var rescheduleAt = currentTimeMillis() + randomize(MINUTES.toMillis(10));
                    return reschedule(rescheduleAt);
                }

                logger.debug("shard lock acquired by this operation: this={} lock={}", operationId, lock);
                return onLockAcquired(lock.id());
            });
    }

    private CompletableFuture<Void> onLockAcquired(String lockId) {
        this.lockId = lockId;
        return interrupted
            ? reloadShardForInterruption()
            : moveToDeletedMetrics();
    }

    private CompletableFuture<Void> moveToDeletedMetrics() {
        return moveToDeletedMetrics.start()
            .thenCompose(i -> forceSave())
            .thenCompose(i -> onCompleteMoveToDeletedMetrics());
    }

    private CompletableFuture<Void> onCompleteMoveToDeletedMetrics() {
        var progress = moveToDeletedMetrics.progress();
        if (!progress.getComplete()) {
            return checkShardExistence();
        }

        if (progress.getExactTotalMetrics() == 0) {
            // no need to reload when nothing has been moved
            return unlockShard();
        }

        return reloadShard();
    }

    private CompletableFuture<Void> reloadShard() {
        return reloadShard.start()
            .thenCompose(i -> forceSave())
            .thenCompose(i -> onCompleteReloadShard());
    }

    private CompletableFuture<Void> onCompleteReloadShard() {
        var progress = reloadShard.progress();
        if (!progress.getComplete()) {
            return checkShardExistence();
        }

        return unlockShard();
    }

    private CompletableFuture<Void> reloadShardForInterruption() {
        return reloadShard.start()
            .thenCompose(i -> forceSave())
            .thenCompose(i -> onCompleteReloadShardForInterruption());
    }

    private CompletionStage<Void> onCompleteReloadShardForInterruption() {
        var progress = reloadShard.progress();
        if (!progress.getComplete()) {
            return checkShardExistence();
        }

        return repairDeletedMetrics();
    }

    private CompletableFuture<Void> repairDeletedMetrics() {
        return repairDeletedMetrics.repair()
            .thenCompose(this::onCompleteRepairDeletedMetrics);
    }

    private CompletableFuture<Void> onCompleteRepairDeletedMetrics(boolean complete) {
        if (!complete) {
            return checkShardExistence();
        }

        return unlockShard()
            .thenCompose(i -> fail(Status.CANCELLED.withDescription("interruption requested")));
    }

    private CompletableFuture<Void> checkShardExistence() {
        return shardStillExists()
            ? reschedule(currentTimeMillis() + MINUTES.toMillis(5))
            : completedFuture(null);
    }

    private boolean shardStillExists() {
        return getShardConfOrNull() != null;
    }

    @Nullable
    private ShardConfMaybeWrong getShardConfOrNull() {
        return confHolder.getConfOrThrow().getShardByNumIdOrNull(params.getNumId());
    }

    private CompletableFuture<Void> unlockShard() {
        var lockId = this.lockId;
        assert lockId != null;

        return locksDao.releaseLock(lockId, params.getOperationId())
            .thenApply(success -> null);
    }

    @Override
    protected DeleteMetricsMoveProgress onUpdateProgress(DeleteMetricsMoveProgress prev, @Nullable Status status) {
        var update = prev.toBuilder()
            .setMoveToDeletedMetrics(moveToDeletedMetrics.progress())
            .setReloadShard(reloadShard.progress());

        if (status != null) {
            if (status.isOk()) {
                update.clearAttempt();
                update.clearStatus();
            } else {
                update.setAttempt(prev.getAttempt() + 1);
                update.setStatus(StatusProto.fromStatusAndTrailers(status, null));
            }
        }

        return update.build();
    }

    @Override
    protected DeleteMetricsMoveResult result(DeleteMetricsMoveProgress progress) {
        return DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(progress.getMoveToDeletedMetrics().getExactTotalMetrics())
            .build();
    }

    @Override
    protected void afterComplete(Message result) {
        var shardConf = getShardConfOrNull();
        if (shardConf != null) {
            metrics.moveComplete(
                shardConf.getRaw().getProjectId(),
                shardConf.getId(),
                millisecondsToSeconds(currentTimeMillis() - params.getSubTaskCreatedAt()),
                ((DeleteMetricsMoveResultOrBuilder) result).getMovedMetrics());
        }
    }

    @Override
    protected int attempt(DeleteMetricsMoveProgress progress) {
        return progress.getAttempt();
    }

    @Override
    protected void onClose() {
        moveToDeletedMetrics.close();
        reloadShard.close();
        repairDeletedMetrics.close();
    }

    @VisibleForTesting
    static String toLockId(int numId) {
        return DeleteMetricsMoveTaskHandler.TYPE + "_" + Integer.toUnsignedLong(numId);
    }
}
