package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Message;
import io.grpc.Status;
import io.grpc.protobuf.StatusProto;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.coremon.api.task.DeleteMetricsRollbackProgress;
import ru.yandex.coremon.api.task.DeleteMetricsRollbackResult;
import ru.yandex.solomon.core.conf.ShardConfMaybeWrong;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.tasks.AbstractTask;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.lang.System.currentTimeMillis;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.TimeUnit.MINUTES;
import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class DeleteMetricsRollbackTask extends AbstractTask<DeleteMetricsRollbackProgress> {

    private final DeleteMetricsTaskMetrics metrics;
    private final SolomonConfHolder confHolder;
    private final DeleteMetricsParams params;

    private final Status prevProgressStatus;

    private final boolean interrupted;
    private final RollbackDeletedMetrics rollbackDeletedMetrics;

    DeleteMetricsRollbackTask(
        RetryConfig retryConfig,
        ExecutionContext context,
        DeleteMetricsParams params,
        DeleteMetricsRollbackProgress progress,
        Status prevProgressStatus,
        Executor executor,
        ScheduledExecutorService timer,
        DeleteMetricsTaskMetrics metrics,
        SolomonConfHolder confHolder,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        DeletedMetricsDao deletedMetricsDao)
    {
        super(
            DeleteMetricsRollbackTaskHandler.TYPE,
            retryConfig,
            context,
            params,
            progress,
            executor,
            timer);

        this.metrics = metrics;
        this.confHolder = confHolder;
        this.params = params;

        this.prevProgressStatus = prevProgressStatus;

        this.interrupted = progress.getInterrupted();
        this.rollbackDeletedMetrics = new RollbackDeletedMetrics(
            retryConfig(),
            deletedMetricsDao,
            shardResolver,
            executor,
            params,
            progress.getRollbackDeletedMetrics(),
            progress.getInterrupted());
    }

    @Override
    protected CompletableFuture<Void> onStart() {
        return rollbackDeletedMetrics();
    }

    private CompletableFuture<Void> rollbackDeletedMetrics() {
        return rollbackDeletedMetrics.start()
            .thenCompose(i -> forceSave())
            .thenCompose(i -> onCompleteRollbackDeletedMetrics());
    }

    private CompletableFuture<Void> onCompleteRollbackDeletedMetrics() {
        if (interrupted) {
            return fail(Status.CANCELLED.withDescription(
                "interrupted from status=" + prevProgressStatus.toString()));
        }

        var progress = rollbackDeletedMetrics.progress();
        if (!progress.getComplete()) {
            return checkShardExistence();
        }

        return completedFuture(null);
    }

    private CompletableFuture<Void> checkShardExistence() {
        return shardStillExists()
            ? reschedule(currentTimeMillis() + MINUTES.toMillis(5))
            : completedFuture(null);
    }

    private boolean shardStillExists() {
        return getShardConfOrNull() != null;
    }

    @Nullable
    private ShardConfMaybeWrong getShardConfOrNull() {
        return confHolder.getConfOrThrow().getShardByNumIdOrNull(params.getNumId());
    }

    @Override
    protected DeleteMetricsRollbackProgress onUpdateProgress(
        DeleteMetricsRollbackProgress prev,
        @Nullable Status status)
    {
        var update = prev.toBuilder()
            .setRollbackDeletedMetrics(rollbackDeletedMetrics.progress());

        if (status != null) {
            if (status.isOk()) {
                update.clearAttempt();
                update.clearStatus();
            } else {
                update.setAttempt(prev.getAttempt() + 1);
                update.setStatus(StatusProto.fromStatusAndTrailers(status, null));
            }
        }

        return update.build();
    }

    @Override
    protected DeleteMetricsRollbackResult result(DeleteMetricsRollbackProgress progress) {
        return DeleteMetricsRollbackResult.newBuilder()
            .setStillDeletedMetrics(progress.getRollbackDeletedMetrics().getStillDeletedMetrics())
            .build();
    }

    @Override
    protected void afterReschedule(long executeAt, DeleteMetricsRollbackProgress progress, @Nullable Status status) {
        if (status != null && !status.isOk()) {
            var shard = getShardConfOrNull();
            if (shard != null) {
                metrics.rollbackStuck(
                    shard.getRaw().getProjectId(),
                    shard.getId(),
                    status.getCode());
            }
        }
    }

    @Override
    protected void afterComplete(Message result) {
        var shardConf = getShardConfOrNull();
        if (shardConf != null) {
            metrics.rollbackComplete(
                shardConf.getRaw().getProjectId(),
                shardConf.getId(),
                millisecondsToSeconds(currentTimeMillis() - params.getSubTaskCreatedAt()));
        }
    }

    @Override
    protected int attempt(DeleteMetricsRollbackProgress progress) {
        return progress.getAttempt();
    }

    @Override
    protected void onClose() {
        rollbackDeletedMetrics.close();
    }

}
