package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.Descriptors.Descriptor;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.coremon.api.task.DeleteMetricsRollbackProgress;
import ru.yandex.coremon.api.task.DeleteMetricsRollbackResult;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.ProgressOperator.Ok;
import ru.yandex.solomon.scheduler.ProgressOperator.Result;
import ru.yandex.solomon.scheduler.ProgressOperator.Stop;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.grpc.Proto;
import ru.yandex.stockpile.client.StockpileClient;

import static java.lang.System.currentTimeMillis;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public final class DeleteMetricsRollbackTaskHandler extends AbstractDeleteMetricsTaskHandler<DeleteMetricsRollbackTask> {

    static final String TYPE = "delete_metrics_rollback";

    private final DeletedMetricsDao deletedMetricsDao;
    private final Executor executor;
    private final ScheduledExecutorService timer;

    public DeleteMetricsRollbackTaskHandler(
        DeleteMetricsTaskMetrics metrics,
        SolomonConfHolder confHolder,
        StockpileClient stockpileClient,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        DeletedMetricsDao deletedMetricsDao,
        Executor executor,
        ScheduledExecutorService timer)
    {
        super(metrics, confHolder, stockpileClient, shardResolver);
        this.deletedMetricsDao = deletedMetricsDao;
        this.executor = executor;
        this.timer = timer;
    }

    @Override
    public String type() {
        return TYPE;
    }

    @Override
    DeleteMetricsRollbackTask createTask(ExecutionContext context, DeleteMetricsParams params) {
        // explicitly clear previous status at the beginning of next attempt
        // to remove cancellation stuck details from operation status message
        var prevProgress = DeleteMetricsTaskProto.rollbackProgress(context.task().progress());
        var prevProgressStatus = Proto.fromProto(prevProgress.getStatus());
        var progress = prevProgress.toBuilder()
            .clearStatus()
            .build();

        return new DeleteMetricsRollbackTask(
            RETRY_CONFIG,
            context,
            params,
            progress,
            prevProgressStatus,
            executor,
            timer,
            metrics,
            confHolder,
            shardResolver,
            deletedMetricsDao
        );
    }

    @Override
    public List<Descriptor> descriptors() {
        return List.of(
            DeleteMetricsParams.getDescriptor(),
            DeleteMetricsRollbackProgress.getDescriptor(),
            DeleteMetricsRollbackResult.getDescriptor());
    }

    public static Task task(DeleteMetricsParams params) {
        return Task.newBuilder()
            .setId(taskId(params))
            .setType(TYPE)
            .setExecuteAt(currentTimeMillis())
            .setParams(Any.pack(params))
            .build();
    }

    public static String taskId(DeleteMetricsParams params) {
        return params.getOperationId() + "_rollback_" + Integer.toUnsignedLong(params.getNumId());
    }

    public static Result interruptOperator(Any progress) {
        var prev = DeleteMetricsTaskProto.rollbackProgress(progress);
        if (prev.getInterrupted() || prev.getRollbackDeletedMetrics().getComplete()) {
            return new Stop();
        }

        var update = prev.toBuilder()
            .setInterrupted(true)
            .build();

        return new Ok(Any.pack(update));
    }

}
