package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Status;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.coremon.DeleteMetricsConfig;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public final class DeleteMetricsTaskMetrics {

    private final ConcurrentMap<MetricsKey, Metrics> metrics = new ConcurrentHashMap<>();

    private final MetricRegistry registry;
    private final Rate nonExistingShard;
    private final boolean verbose;

    public DeleteMetricsTaskMetrics(MetricRegistry registry, DeleteMetricsConfig config) {
        this.registry = registry;
        this.nonExistingShard = registry.rate("delete_metrics.terminate.non_existing_shard");
        this.verbose = config.getReportVerboseMetrics();

        prepareMetrics("total", "total");
    }

    void touch(String projectId, String shardId) {
        prepareMetrics(projectId, "total");
        if (verbose) {
            prepareMetrics(projectId, shardId);
        }
    }

    void checkComplete(String projectId, String shardId, int elapsedSeconds) {
        recordCheckElapsedSeconds("total", "total", elapsedSeconds);
        recordCheckElapsedSeconds(projectId, "total", elapsedSeconds);
        if (verbose) {
            recordCheckElapsedSeconds(projectId, shardId, elapsedSeconds);
        }
    }

    void moveComplete(String projectId, String shardId, int elapsedSeconds, int movedMetrics) {
        recordMove("total", "total", elapsedSeconds, movedMetrics);
        recordMove(projectId, "total", elapsedSeconds, movedMetrics);
        if (verbose) {
            recordMove(projectId, shardId, elapsedSeconds, movedMetrics);
        }
    }

    void rollbackStuck(String projectId, String shardId, Status.Code statusCode) {
        assert statusCode != Status.Code.OK;
        incRollbackStuck("total", "total", statusCode);
        incRollbackStuck(projectId, "total", statusCode);
        if (verbose) {
            incRollbackStuck(projectId, shardId, statusCode);
        }
    }

    void rollbackComplete(String projectId, String shardId, int elapsedSeconds) {
        recordRollbackElapsedSeconds("total", "total", elapsedSeconds);
        recordRollbackElapsedSeconds(projectId, "total", elapsedSeconds);
        if (verbose) {
            recordRollbackElapsedSeconds(projectId, shardId, elapsedSeconds);
        }
    }

    void terminateComplete(String projectId, String shardId, int elapsedSeconds) {
        recordTerminateElapsedSeconds("total", "total", elapsedSeconds);
        recordTerminateElapsedSeconds(projectId, "total", elapsedSeconds);
        if (verbose) {
            recordTerminateElapsedSeconds(projectId, shardId, elapsedSeconds);
        }
    }

    void nonExistingShard() {
        nonExistingShard.inc();
    }

    private void recordCheckElapsedSeconds(String projectId, String shardId, int elapsedSeconds) {
        prepareMetrics(projectId, shardId).checkElapsedSeconds.record(elapsedSeconds);
    }

    private void recordMove(String projectId, String shardId, int elapsedSeconds, int movedMetrics) {
        var metrics = prepareMetrics(projectId, shardId);
        metrics.moveElapsedSeconds.record(elapsedSeconds);
        metrics.moved.record(movedMetrics);
    }

    private void recordRollbackElapsedSeconds(String projectId, String shardId, int elapsedSeconds) {
        prepareMetrics(projectId, shardId).rollbackElapsedSeconds.record(elapsedSeconds);
    }

    private void recordTerminateElapsedSeconds(String projectId, String shardId, int elapsedSeconds) {
        prepareMetrics(projectId, shardId).terminateElapsedSeconds.record(elapsedSeconds);
    }

    private void incRollbackStuck(String projectId, String shardId, Status.Code statusCode) {
        prepareMetrics(projectId, shardId).prepareStuck(statusCode).inc();
    }

    private Metrics prepareMetrics(String projectId, String shardId) {
        var key = new MetricsKey(projectId, shardId);
        var result = metrics.get(key);
        if (result == null) {
            metrics.putIfAbsent(key, result = new Metrics(key, registry));
        }
        return result;
    }

    private record MetricsKey(String projectId, String shardId) { }

    private static final class Metrics {

        static final List<Status.Code> stuckCodes =
            List.of(
                Status.Code.ALREADY_EXISTS,
                Status.Code.RESOURCE_EXHAUSTED,
                Status.Code.UNKNOWN);

        final Histogram checkElapsedSeconds;
        final Histogram moveElapsedSeconds;
        final Histogram rollbackElapsedSeconds;
        final Histogram terminateElapsedSeconds;

        final Histogram moved;

        final Map<Status.Code, Rate> stuck;
        final Rate stuckDefault;

        Metrics(MetricsKey key, MetricRegistry registry) {
            var labels = Labels.of("projectId", key.projectId(), "shardId", key.shardId());
            var subRegistry = registry.subRegistry(labels);

            this.checkElapsedSeconds = taskElapsedSeconds(subRegistry, DeleteMetricsCheckTaskHandler.TYPE);
            this.moveElapsedSeconds = taskElapsedSeconds(subRegistry, DeleteMetricsMoveTaskHandler.TYPE);
            this.rollbackElapsedSeconds = taskElapsedSeconds(subRegistry, DeleteMetricsRollbackTaskHandler.TYPE);
            this.terminateElapsedSeconds = taskElapsedSeconds(subRegistry, DeleteMetricsTerminateTaskHandler.TYPE);

            this.moved = subRegistry.histogramRate(
                "delete_metrics.move.moved_metrics",
                Histograms.exponential(19, 2, 1000));

            var stuck = new EnumMap<Status.Code, Rate>(Status.Code.class);
            for (var sc : stuckCodes) {
                stuck.put(
                    sc,
                    subRegistry.rate(
                        "delete_metrics.rollback.stuck",
                        Labels.of("status", sc.name())));
            }
            this.stuck = stuck;
            this.stuckDefault = stuck.get(Status.Code.UNKNOWN);
        }

        Rate prepareStuck(Status.Code statusCode) {
            return stuck.getOrDefault(statusCode, stuckDefault);
        }

        static Histogram taskElapsedSeconds(MetricRegistry registry, String type) {
            return registry.histogramRate(
                "delete_metrics.task_elapsed_seconds",
                Labels.of("type", type),
                Histograms.exponential(13, 2, 16));
        }
    }
}
