package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Message;
import io.grpc.Status;
import io.grpc.protobuf.StatusProto;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.coremon.api.task.DeleteMetricsTerminateProgress;
import ru.yandex.coremon.api.task.DeleteMetricsTerminateResult;
import ru.yandex.solomon.core.conf.ShardConfMaybeWrong;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.tasks.AbstractTask;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.StockpileClient;

import static java.lang.System.currentTimeMillis;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.TimeUnit.MINUTES;
import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class DeleteMetricsTerminateTask extends AbstractTask<DeleteMetricsTerminateProgress> {

    private final DeleteMetricsTaskMetrics metrics;
    private final SolomonConfHolder confHolder;
    private final DeleteMetricsParams params;

    private final DeletePermanently deletePermanently;
    private final CleanUpNonExistingShard cleanUpNonExistingShard;

    DeleteMetricsTerminateTask(
        RetryConfig retryConfig,
        ExecutionContext context,
        DeleteMetricsParams params,
        DeleteMetricsTerminateProgress progress,
        Executor executor,
        ScheduledExecutorService timer,
        DeleteMetricsTaskMetrics metrics,
        SolomonConfHolder confHolder,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        DeletedMetricsDao deletedMetricsDao,
        StockpileClient stockpileClient)
    {
        super(
            DeleteMetricsTerminateTaskHandler.TYPE,
            retryConfig,
            context,
            params,
            progress,
            executor,
            timer);

        this.metrics = metrics;
        this.confHolder = confHolder;
        this.params = params;

        this.deletePermanently = new DeletePermanently(
            retryConfig(),
            deletedMetricsDao,
            stockpileClient,
            shardResolver,
            executor,
            params,
            progress.getDeletePermanently());
        this.cleanUpNonExistingShard = new CleanUpNonExistingShard(
            retryConfig,
            deletedMetricsDao,
            params,
            progress.getCleanUpNonExistingShard());
    }

    @Override
    protected CompletableFuture<Void> onStart() {
        return deletePermanently();
    }

    private CompletableFuture<Void> deletePermanently() {
        return deletePermanently.start()
            .thenCompose(i -> forceSave())
            .thenCompose(i -> onCompleteDeletePermanently());
    }

    private CompletableFuture<Void> onCompleteDeletePermanently() {
        var progress = deletePermanently.progress();
        if (!progress.getComplete()) {
            return checkShardExistence();
        }

        return completedFuture(null);
    }

    private CompletableFuture<Void> checkShardExistence() {
        return shardStillExists()
            ? reschedule(currentTimeMillis() + MINUTES.toMillis(5))
            : cleanUpNonExistingShard();
    }

    private CompletableFuture<Void> cleanUpNonExistingShard() {
        return cleanUpNonExistingShard.start()
            .thenCompose(i -> forceSave())
            .thenCompose(i -> onCompleteCleanUpNonExistingShard());
    }

    private CompletableFuture<Void> onCompleteCleanUpNonExistingShard() {
        return completedFuture(null);
    }

    private boolean shardStillExists() {
        return getShardConfOrNull() != null;
    }

    @Nullable
    private ShardConfMaybeWrong getShardConfOrNull() {
        return confHolder.getConfOrThrow().getShardByNumIdOrNull(params.getNumId());
    }

    @Override
    protected DeleteMetricsTerminateProgress onUpdateProgress(DeleteMetricsTerminateProgress prev, @Nullable Status status) {
        var update = prev.toBuilder()
            .setDeletePermanently(deletePermanently.progress())
            .setCleanUpNonExistingShard(cleanUpNonExistingShard.progress());

        if (status != null) {
            if (status.isOk()) {
                update.clearAttempt();
                update.clearStatus();
            } else {
                update.setAttempt(prev.getAttempt() + 1);
                update.setStatus(StatusProto.fromStatusAndTrailers(status, null));
            }
        }

        return update.build();
    }

    @Override
    protected DeleteMetricsTerminateResult result(DeleteMetricsTerminateProgress progress) {
        return DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(progress.getDeletePermanently().getDeletedMetrics())
            .build();
    }

    @Override
    protected void afterComplete(Message result) {
        var shardConf = getShardConfOrNull();
        if (shardConf != null) {
            metrics.terminateComplete(
                shardConf.getRaw().getProjectId(),
                shardConf.getId(),
                millisecondsToSeconds(currentTimeMillis() - params.getSubTaskCreatedAt()));
        } else {
            metrics.nonExistingShard();
        }
    }

    @Override
    protected int attempt(DeleteMetricsTerminateProgress progress) {
        return progress.getAttempt();
    }

    @Override
    protected void onClose() {
        deletePermanently.close();
        cleanUpNonExistingShard.close();
    }

}
