package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.TextFormat;
import com.google.protobuf.UnsafeByteOperations;
import io.grpc.Status;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import it.unimi.dsi.fastutil.longs.LongList;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.coremon.api.task.DeleteMetricsTerminateProgress.DeletePermanentlyProgress;
import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.serializer.OwnerField;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TWriteDataBinaryRequest;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequestBuilder;

import static java.util.concurrent.CompletableFuture.allOf;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class DeletePermanently implements AutoCloseable {

    //TODO: configure maxAsyncOperations? <SOLOMON-990>
    private static final int DEFAULT_MAX_ASYNC_OPERATIONS = 4;
    private static final int DEFAULT_MAX_BATCH_SIZE = 10_000;
    private static final int DEFAULT_FIND_METRICS_LIMIT = 10_000;

    private final RetryContext retryCtx;

    private final DeletedMetricsDao deletedMetricsDao;
    private final StockpileClient stockpileClient;
    private final MetabaseShardResolver<? extends MetabaseShard> shardResolver;
    private final Executor executor;

    private final DeleteMetricsParams params;

    private final AtomicReference<DeletePermanentlyProgress> progress;

    private final int findMetricsLimit;
    private final int maxBatchSize;

    private final AtomicInteger total;
    private final AtomicInteger deleted;

    DeletePermanently(
        RetryConfig retry,
        DeletedMetricsDao deletedMetricsDao,
        StockpileClient stockpileClient,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params,
        DeletePermanentlyProgress progress)
    {
        this(
            retry,
            deletedMetricsDao,
            stockpileClient,
            shardResolver,
            executor,
            params,
            progress,
            DEFAULT_FIND_METRICS_LIMIT,
            DEFAULT_MAX_BATCH_SIZE);
    }

    @VisibleForTesting
    DeletePermanently(
        RetryConfig retry,
        DeletedMetricsDao deletedMetricsDao,
        StockpileClient stockpileClient,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params,
        DeletePermanentlyProgress progress,
        int findMetricsLimit,
        int maxBatchSize)
    {
        this.retryCtx = new RetryContext(retry);
        this.deletedMetricsDao = deletedMetricsDao;
        this.stockpileClient = stockpileClient;
        this.shardResolver = shardResolver;
        this.executor = executor;
        this.params = params;
        this.progress = new AtomicReference<>(progress);
        this.findMetricsLimit = findMetricsLimit;
        this.maxBatchSize = maxBatchSize;

        this.total = new AtomicInteger(progress.getTotalMetrics());
        this.deleted = new AtomicInteger(progress.getDeletedMetrics());
    }

    public CompletableFuture<Void> start() {
        try {
            return tryStart();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public DeletePermanentlyProgress progress() {
        return progress.get();
    }

    private CompletableFuture<Void> tryStart() {
        if (progress().getComplete()) {
            return completedFuture(null);
        }

        var shard = shardResolver.resolveShardOrNull(params.getNumId());
        if (shard == null || !shard.isLoaded()) {
            return completedFuture(null);
        }

        return updateMetricCount()
            .thenCompose(i -> deleteMetrics(shard.getLabelAllocator()))
            .thenCompose(i -> updateMetricCount())
            .whenComplete((i, t) -> updateProgress(t == null));
    }

    private CompletableFuture<Void> updateMetricCount() {
        return getMetricCount().thenAccept(count -> {
            total.set(Math.max(total.get(), Math.toIntExact(count)));
            deleted.set(total.get() - Math.toIntExact(count));
            updateProgress(false);
        });
    }

    private CompletableFuture<?> deleteMetrics(LabelAllocator labelAllocator) {
        return new DeleteActor(labelAllocator).start();
    }

    private CompletableFuture<Long> getMetricCount() {
        return retry(() -> deletedMetricsDao.count(params.getOperationId(), params.getNumId()));
    }

    private void updateProgress(boolean complete) {
        DeletePermanentlyProgress prev;
        DeletePermanentlyProgress update;
        do {
            prev = progress.get();
            var total = this.total.get();
            var deleted = this.deleted.get();
            update = DeletePermanentlyProgress.newBuilder()
                .setComplete(complete || prev.getComplete())
                .setTotalMetrics(total)
                .setDeletedMetrics(deleted)
                .setProgress(total > 0 ? deleted / (double) total : 1)
                .build();
        } while (!progress.compareAndSet(prev, update));
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryCtx.retry(supplier);
    }

    @Override
    public void close() {
        retryCtx.close();
    }

    private static TWriteDataBinaryRequest toWriteDataBinaryRequest(DeleteBatch batch) {
        var builder = new StockpileShardWriteRequestBuilder(EProjectId.UNKNOWN, OwnerField.UNKNOWN_SHARD_ID);
        for (int i = 0; i < batch.size(); i++) {
            builder.addDeleteData(batch.getLocalId(i));
        }

        byte[] content;
        try (var request = builder.build()) {
            content = request.serialize();
        }

        return TWriteDataBinaryRequest.newBuilder()
            .setShardId(batch.getStockpileShardId())
            .setContent(UnsafeByteOperations.unsafeWrap(content))
            .build();
    }

    @ParametersAreNonnullByDefault
    private class DeleteActor {

        private final Int2ObjectMap<DeleteBatch> batchByStockpileShardId = new Int2ObjectOpenHashMap<>();
        private final List<DeleteBatch> readyBatches = new ArrayList<>();

        private final LabelAllocator labelAllocator;
        private final AsyncActorRunner actorRunner;

        @Nullable
        private volatile CoremonMetricArray metrics;

        DeleteActor(LabelAllocator labelAllocator) {
            this.labelAllocator = labelAllocator;
            this.actorRunner = new AsyncActorRunner(this::body, executor, 1);
        }

        CompletableFuture<?> start() {
            return nextMetrics(null)
                .thenCompose(i -> actorRunner.start())
                .whenComplete((i, t) -> {
                    var metrics = this.metrics;
                    if (metrics != null) {
                        metrics.close();
                    }
                });
        }

        private CompletableFuture<?> body() {
            try (var metrics = this.metrics) {
                assert metrics != null;
                this.metrics = null; // takes ownership

                if (metrics.isEmpty()) {
                    return processRemainingBatches();
                }

                return processMetrics(metrics);
            }
        }

        private CompletableFuture<?> processRemainingBatches() {
            var batches = batchByStockpileShardId.values();
            if (batches.isEmpty()) {
                return completedFuture(AsyncActorBody.DONE_MARKER);
            }

            return new DeleteBatchesActor(batches.iterator()).start()
                .thenApply(i -> AsyncActorBody.DONE_MARKER);
        }

        private CompletableFuture<?> processMetrics(CoremonMetricArray metrics) {
            var lastKey = metrics.getLabels(metrics.size() - 1);
            var nextMetricsFuture = nextMetrics(lastKey);

            try {
                return tryProcessMetrics(metrics, nextMetricsFuture);
            } catch (Throwable t) {
                return allOf(failedFuture(t), nextMetricsFuture);
            }
        }

        private CompletableFuture<?> tryProcessMetrics(
            CoremonMetricArray metrics,
            CompletableFuture<?> nextMetricsFuture)
        {
            for (int i = 0; i < metrics.size(); i++) {
                var spShardId = metrics.getShardId(i);
                var localId = metrics.getLocalId(i);
                var labels = metrics.getLabels(i);

                var batch = batchByStockpileShardId.computeIfAbsent(spShardId, DeleteBatch::new);
                batch.add(localId, labels);

                if (batch.size() >= maxBatchSize) {
                    readyBatches.add(batchByStockpileShardId.remove(spShardId));
                }
            }

            if (!readyBatches.isEmpty()) {
                var batches = List.copyOf(readyBatches);
                readyBatches.clear();

                var deleteBatchesFuture = new DeleteBatchesActor(batches.iterator()).start();
                return allOf(nextMetricsFuture, deleteBatchesFuture);
            }

            return nextMetricsFuture;
        }

        private CompletableFuture<?> nextMetrics(@Nullable Labels lastKey) {
            var nextMetrics = new CoremonMetricArray(findMetricsLimit);
            return findMetrics(nextMetrics, lastKey)
                .whenComplete((i, t) -> {
                    if (t == null) {
                        this.metrics = nextMetrics;
                    } else {
                        nextMetrics.close();
                    }
                });
        }

        private CompletableFuture<Void> findMetrics(
            CoremonMetricArray buffer,
            @Nullable Labels lastKey)
        {
            return retry(
                () -> deletedMetricsDao.find(
                    params.getOperationId(),
                    params.getNumId(),
                    findMetricsLimit,
                    lastKey,
                    buffer,
                    labelAllocator));
        }
    }

    @ParametersAreNonnullByDefault
    private class DeleteBatchesActor {

        private final Iterator<DeleteBatch> batchIt;
        private final AsyncActorRunner actorRunner;

        DeleteBatchesActor(Iterator<DeleteBatch> batchIt) {
            this.batchIt = batchIt;
            this.actorRunner = new AsyncActorRunner(this::body, executor, DEFAULT_MAX_ASYNC_OPERATIONS);
        }

        CompletableFuture<Void> start() {
            return actorRunner.start();
        }

        private CompletableFuture<?> body() {
            if (batchIt.hasNext()) {
                var batch = batchIt.next();

                return deleteFromStockpile(batch)
                    .thenCompose(this::deleteFromDeletedMetrics)
                    .thenAccept(count -> {
                        deleted.addAndGet(count);
                        updateProgress(false);
                    });
            }

            return completedFuture(AsyncActorBody.DONE_MARKER);
        }

        private CompletableFuture<Collection<Labels>> deleteFromStockpile(DeleteBatch batch) {
            try {
                var request = toWriteDataBinaryRequest(batch);
                var labels = batch.getLabels();

                return writeDataBinary(request).thenApply(i -> labels);
            } catch (Throwable t) {
                return failedFuture(t);
            }
        }

        private CompletableFuture<?> writeDataBinary(TWriteDataBinaryRequest request) {
            return retry(
                () -> stockpileClient.writeDataBinary(request)
                    .thenAccept(response -> {
                        if (response.getStatus() != EStockpileStatusCode.OK) {
                            var desc = "Unable to write data binary for " + TextFormat.shortDebugString(params) +
                                " from stockpile shard " + request.getShardId() +
                                " caused by " + response.getStatus() +
                                ": " + response.getStatusMessage();
                            throw new StatusRuntimeExceptionNoStackTrace(Status.ABORTED.withDescription(desc));
                        }
                    }));
        }

        private CompletableFuture<Integer> deleteFromDeletedMetrics(Collection<Labels> keys) {
            return retry(() -> deletedMetricsDao.delete(params.getOperationId(), params.getNumId(), keys))
                .thenApply(i -> keys.size());
        }
    }

    @ParametersAreNonnullByDefault
    private static class DeleteBatch {

        private final LongList localIds = new LongArrayList();
        private final List<Labels> labels = new ArrayList<>();

        private final int stockpileShardId;

        DeleteBatch(int stockpileShardId) {
            this.stockpileShardId = stockpileShardId;
        }

        int getStockpileShardId() {
            return stockpileShardId;
        }

        void add(long localId, Labels labels) {
            this.localIds.add(localId);
            this.labels.add(labels);
        }

        long getLocalId(int i) {
            return localIds.get(i);
        }

        Collection<Labels> getLabels() {
            return labels;
        }

        int size() {
            return localIds.size();
        }
    }
}
