package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import io.grpc.Status;

import ru.yandex.coremon.api.task.DeleteMetricsMoveProgress.MoveToDeletedMetricsProgress;
import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.collection.CloseableIterator;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class MoveToDeletedMetrics implements AutoCloseable {

    //TODO: configure maxAsyncOperations? <SOLOMON-990>
    private static final int DEFAULT_MAX_ASYNC_OPERATIONS = 10;
    private static final int DEFAULT_MAX_BATCH_SIZE = 1000;

    private final RetryContext retryCtx;

    private final DeletedMetricsDao deletedMetricsDao;
    private final MetricsDaoFactory metricsDaoFactory;
    private final MetabaseShardResolver<? extends MetabaseShard> shardResolver;
    private final Executor executor;

    private final DeleteMetricsParams params;
    private final Selectors selectors;

    private final AtomicReference<MoveToDeletedMetricsProgress> progress;

    private final int maxBatchSize;

    @Nullable
    private MoveActor mover;

    MoveToDeletedMetrics(
        RetryConfig retry,
        DeletedMetricsDao deletedMetricsDao,
        MetricsDaoFactory metricsDaoFactory,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params,
        Selectors selectors,
        MoveToDeletedMetricsProgress progress)
    {
        this(
            retry,
            deletedMetricsDao,
            metricsDaoFactory,
            shardResolver,
            executor,
            params,
            selectors,
            progress,
            DEFAULT_MAX_BATCH_SIZE);
    }

    @VisibleForTesting
    MoveToDeletedMetrics(
        RetryConfig retry,
        DeletedMetricsDao deletedMetricsDao,
        MetricsDaoFactory metricsDaoFactory,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params,
        Selectors selectors,
        MoveToDeletedMetricsProgress progress,
        int maxBatchSize)
    {
        this.retryCtx = new RetryContext(retry);
        this.deletedMetricsDao = deletedMetricsDao;
        this.metricsDaoFactory = metricsDaoFactory;
        this.shardResolver = shardResolver;
        this.executor = executor;
        this.params = params;
        this.selectors = selectors;
        this.progress = new AtomicReference<>(progress);
        this.maxBatchSize = maxBatchSize;
    }

    public CompletableFuture<Void> start() {
        try {
            return tryStart();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Void> tryStart() {
        if (progress().getComplete()) {
            return completedFuture(null);
        }

        var shard = shardResolver.resolveShardOrNull(params.getNumId());
        if (shard == null || !shard.isLoaded()) {
            return completedFuture(null);
        }

        var metricsDao = metricsDaoFactory.create(params.getNumId(), shard.getLabelAllocator());

        var fileMetrics = shard.getStorage().getFileMetrics();
        var searchResult = fileMetrics.searchIterator(selectors);

        var processed = progress().getProcessed();
        var remaining = searchResult.count();
        var it = searchResult.iterator();

        mover = new MoveActor(
            it,
            metricsDao,
            processed,
            remaining);

        return mover.start();
    }

    public MoveToDeletedMetricsProgress progress() {
        return progress.get();
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryCtx.retry(supplier);
    }

    @Override
    public void close() {
        if (mover != null) {
            mover.close();
        }
        retryCtx.close();
    }

    @ParametersAreNonnullByDefault
    private class MoveActor implements AutoCloseable {

        private final CloseableIterator<? extends CoremonMetric> metricIt;
        private final MetricsDao metricsDao;
        private final AsyncActorRunner actorRunner;

        private final AtomicInteger processed;
        private final int currentTotal;

        private final int createdAtThresholdSeconds;
        private final CompletableFuture<Void> resultFuture = new CompletableFuture<>();

        private volatile boolean closed;

        MoveActor(
            CloseableIterator<? extends CoremonMetric> metricIt,
            MetricsDao metricsDao,
            int processed,
            int remaining)
        {
            this.metricIt = metricIt;
            this.metricsDao = metricsDao;
            this.processed = new AtomicInteger(processed);
            this.currentTotal = processed + remaining;
            this.createdAtThresholdSeconds = millisecondsToSeconds(params.getCreatedAt());
            this.actorRunner = new AsyncActorRunner(this::body, executor, DEFAULT_MAX_ASYNC_OPERATIONS);
        }

        CompletableFuture<Void> start() {
            estimateTotal();
            var actorFuture = actorRunner.start()
                .whenComplete((i, t) -> cleanUp());
            CompletableFutures.whenComplete(actorFuture, resultFuture);
            return resultFuture.thenCompose(i -> complete())
                .whenComplete((i, t) -> {
                    if (t != null) {
                        updateProgress();
                    }
                });
        }

        private CompletableFuture<?> body() {
            if (closed) {
                return failedFuture(Status.CANCELLED.asRuntimeException());
            }

            var moveFuture = moveMetricsFromIterator();
            if (moveFuture != null) {
                return moveFuture;
            }

            return completedFuture(AsyncActorBody.DONE_MARKER);
        }

        @Nullable
        private CompletableFuture<?> moveMetricsFromIterator() {
            if (!metricIt.hasNext()) {
                return null;
            }

            int createdAfter = 0;
            var batch = new CoremonMetricArray(maxBatchSize);
            while (metricIt.hasNext()) {
                try (var metric = metricIt.next()) {
                    if (metric.getCreatedAtSeconds() > createdAtThresholdSeconds) {
                        createdAfter++;
                        continue;
                    }

                    batch.add(metric);
                }

                if (batch.size() == maxBatchSize) {
                    break;
                }
            }

            if (createdAfter > 0) {
                processed.addAndGet(createdAfter);
                updateProgress();
            }

            return moveBatch(batch);
        }

        private CompletableFuture<?> moveBatch(CoremonMetricArray batch) {
            if (batch.isEmpty()) {
                batch.close();
                return completedFuture(null);
            }

            return upsertToDeleted(batch)
                .thenCompose(this::deleteFromMain)
                .handle((i, t) -> {
                    if (t != null) {
                        resultFuture.completeExceptionally(t);
                        return AsyncActorBody.DONE_MARKER;
                    }
                    return null;
                });
        }

        private CompletableFuture<? extends Collection<Labels>> upsertToDeleted(CoremonMetricArray batch) {
            return retry(() -> deletedMetricsDao.bulkUpsert(params.getOperationId(), params.getNumId(), batch))
                .thenApply(ignore -> {
                    var keys = new ArrayList<Labels>(batch.size());
                    for (int i = 0; i < batch.size(); i++) {
                        keys.add(batch.getLabels(i));
                    }
                    return keys;
                })
                .whenComplete((i, t) -> batch.close());
        }

        private CompletableFuture<Void> deleteFromMain(Collection<Labels> keys) {
            return retry(() -> metricsDao.deleteMetrics(keys))
                .thenRun(() -> {
                    processed.addAndGet(keys.size());
                    updateProgress();
                });
        }

        private void estimateTotal() {
            MoveToDeletedMetricsProgress prev;
            MoveToDeletedMetricsProgress update;
            do {
                prev = progress.get();
                if (prev.getEstimatedTotalMetrics() != 0) {
                    return;
                }

                update = prev.toBuilder()
                    .setEstimatedTotalMetrics(currentTotal)
                    .build();
            } while (!progress.compareAndSet(prev, update));
        }

        private void updateProgress() {
            MoveToDeletedMetricsProgress prev;
            MoveToDeletedMetricsProgress update;
            do {
                prev = progress.get();

                var processed = this.processed.get();
                var progress = currentTotal > 0 ? processed / (double) currentTotal : 1;

                update = prev.toBuilder()
                    .setProgress(Math.max(prev.getProgress(), progress))
                    .setProcessed(processed)
                    .build();
            } while (!progress.compareAndSet(prev, update));
        }

        private CompletableFuture<Void> complete() {
            return getMetricCount().thenAccept(count -> {
                MoveToDeletedMetricsProgress prev;
                MoveToDeletedMetricsProgress update;
                do {
                    prev = progress.get();

                    update = prev.toBuilder()
                        .setComplete(true)
                        .setProgress(1)
                        .setExactTotalMetrics(Math.toIntExact(count))
                        .build();
                } while (!progress.compareAndSet(prev, update));
            });
        }

        private CompletableFuture<Long> getMetricCount() {
            return retry(() -> deletedMetricsDao.count(params.getOperationId(), params.getNumId()));
        }

        private void cleanUp() {
            metricIt.close();
        }

        @Override
        public void close() {
            closed = true;
        }
    }
}
