package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;

import static java.util.concurrent.CompletableFuture.allOf;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class RepairDeletedMetrics implements AutoCloseable {

    private static final int DEFAULT_FIND_METRICS_LIMIT = 10_000;

    private final RetryContext retryCtx;

    private final DeletedMetricsDao deletedMetricsDao;
    private final MetabaseShardResolver<? extends MetabaseShard> shardResolver;
    private final Executor executor;

    private final DeleteMetricsParams params;

    private final int findMetricsLimit;

    RepairDeletedMetrics(
        RetryConfig retry,
        DeletedMetricsDao deletedMetricsDao,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params)
    {
        this(
            retry,
            deletedMetricsDao,
            shardResolver,
            executor,
            params,
            DEFAULT_FIND_METRICS_LIMIT);
    }

    @VisibleForTesting
    RepairDeletedMetrics(
        RetryConfig retry,
        DeletedMetricsDao deletedMetricsDao,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params,
        int findMetricsLimit)
    {
        this.retryCtx = new RetryContext(retry);
        this.deletedMetricsDao = deletedMetricsDao;
        this.shardResolver = shardResolver;
        this.executor = executor;
        this.params = params;
        this.findMetricsLimit = findMetricsLimit;
    }

    public CompletableFuture<Boolean> repair() {
        try {
            return tryRepair();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Boolean> tryRepair() {
        var shard = shardResolver.resolveShardOrNull(params.getNumId());
        if (shard == null || !shard.isLoaded()) {
            return completedFuture(Boolean.FALSE);
        }

        return repairDeletedMetrics(shard)
            .thenApply(i -> Boolean.TRUE);
    }

    private CompletableFuture<?> repairDeletedMetrics(MetabaseShard shard) {
        return new RepairActor(shard).start();
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryCtx.retry(supplier);
    }

    @Override
    public void close() {
        retryCtx.close();
    }

    @ParametersAreNonnullByDefault
    private class RepairActor {

        private final MetabaseShard shard;
        private final AsyncActorRunner actorRunner;

        @Nullable
        private volatile CoremonMetricArray metrics;

        RepairActor(MetabaseShard shard) {
            this.shard = shard;
            this.actorRunner = new AsyncActorRunner(this::body, executor, 1);
        }

        CompletableFuture<?> start() {
            return nextMetrics(null)
                .thenCompose(i -> actorRunner.start())
                .whenComplete((i, t) -> {
                    var metrics = this.metrics;
                    if (metrics != null) {
                        metrics.close();
                    }
                });
        }

        private CompletableFuture<?> body() {
            try (var metrics = this.metrics) {
                assert metrics != null;
                this.metrics = null; // takes ownership

                if (metrics.isEmpty()) {
                    return completedFuture(AsyncActorBody.DONE_MARKER);
                }

                return processMetrics(metrics);
            }
        }

        private CompletableFuture<?> processMetrics(CoremonMetricArray metrics)
        {
            var lastKey = metrics.getLabels(metrics.size() - 1);
            var nextMetricsFuture = nextMetrics(lastKey);

            try {
                return tryProcessMetrics(metrics, nextMetricsFuture);
            } catch (Throwable t) {
                return allOf(failedFuture(t), nextMetricsFuture);
            }
        }

        private CompletableFuture<?> tryProcessMetrics(
            CoremonMetricArray metrics,
            CompletableFuture<?> nextMetricsFuture)
        {
            var keys = resolveActuallyNotDeleted(metrics);
            var deleteFuture = deleteFromDeletedMetrics(keys);

            return allOf(nextMetricsFuture, deleteFuture);
        }

        private Collection<Labels> resolveActuallyNotDeleted(CoremonMetricArray metrics) {
            return MetricsResolver.resolveExisting(metrics, shard.getStorage().getFileMetrics());
        }

        private CompletableFuture<Integer> deleteFromDeletedMetrics(Collection<Labels> keys) {
            return retry(() -> deletedMetricsDao.delete(params.getOperationId(), params.getNumId(), keys))
                .thenApply(i -> keys.size());
        }

        private CompletableFuture<?> nextMetrics(@Nullable Labels lastKey) {
            var nextMetrics = new CoremonMetricArray(findMetricsLimit);
            return findMetrics(nextMetrics, lastKey)
                .whenComplete((i, t) -> {
                    if (t == null) {
                        this.metrics = nextMetrics;
                    } else {
                        nextMetrics.close();
                    }
                });
        }

        private CompletableFuture<Void> findMetrics(
            CoremonMetricArray buffer,
            @Nullable Labels lastKey)
        {
            return retry(
                () -> deletedMetricsDao.find(
                    params.getOperationId(),
                    params.getNumId(),
                    findMetricsLimit,
                    lastKey,
                    buffer,
                    shard.getLabelAllocator()));
        }
    }

}
