package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiFunction;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import io.grpc.Status;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.coremon.api.task.DeleteMetricsRollbackProgress.RollbackDeletedMetricsProgress;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;

import static java.util.concurrent.CompletableFuture.allOf;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class RollbackDeletedMetrics implements AutoCloseable {

    private static final int DEFAULT_FIND_METRICS_LIMIT = 10_000;

    private final RetryContext retryCtx;

    private final DeletedMetricsDao deletedMetricsDao;
    private final MetabaseShardResolver<? extends MetabaseShard> shardResolver;
    private final Executor executor;

    private final DeleteMetricsParams params;

    private final AtomicReference<RollbackDeletedMetricsProgress> progress;
    private final boolean interrupted;

    private final int findMetricsLimit;

    private final AtomicInteger total;
    private final AtomicInteger stillDeleted;

    RollbackDeletedMetrics(
        RetryConfig retry,
        DeletedMetricsDao deletedMetricsDao,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params,
        RollbackDeletedMetricsProgress progress,
        boolean interrupted)
    {
        this(
            retry,
            deletedMetricsDao,
            shardResolver,
            executor,
            params,
            progress,
            interrupted,
            DEFAULT_FIND_METRICS_LIMIT);
    }

    @VisibleForTesting
    RollbackDeletedMetrics(
        RetryConfig retry,
        DeletedMetricsDao deletedMetricsDao,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params,
        RollbackDeletedMetricsProgress progress,
        boolean interrupted,
        int findMetricsLimit)
    {
        this.retryCtx = new RetryContext(retry);
        this.deletedMetricsDao = deletedMetricsDao;
        this.shardResolver = shardResolver;
        this.executor = executor;
        this.params = params;
        this.progress = new AtomicReference<>(progress);
        this.interrupted = interrupted;
        this.findMetricsLimit = findMetricsLimit;

        this.total = new AtomicInteger(progress.getTotalMetrics());
        this.stillDeleted = new AtomicInteger(progress.getStillDeletedMetrics());
    }

    public CompletableFuture<Void> start() {
        try {
            return tryStart();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public RollbackDeletedMetricsProgress progress() {
        return progress.get();
    }

    private CompletableFuture<Void> tryStart() {
        if (progress().getComplete()) {
            return completedFuture(null);
        }

        var shard = shardResolver.resolveShardOrNull(params.getNumId());
        if (shard == null || !shard.isLoaded()) {
            return completedFuture(null);
        }

        return updateMetricCount(shard.getId(), false)
            .thenCompose(i -> rollbackMetrics(shard))
            .thenCompose(i -> updateMetricCount(shard.getId(), true));
    }

    private CompletableFuture<Void> updateMetricCount(String shardId, boolean afterRollback) {
        return getMetricCount().thenAccept(count -> {
            total.set(Math.max(total.get(), Math.toIntExact(count)));
            stillDeleted.set(Math.toIntExact(count));

            if (!interrupted && afterRollback) {
                if (count != 0) {
                    updateProgress(false);
                    throw Status.ALREADY_EXISTS
                        .withDescription(
                            "metrics with same labels already exist in shard %s: count=%d".formatted(shardId, count))
                        .asRuntimeException();
                }
                updateProgress(true);
            } else {
                updateProgress(false);
            }
        });
    }

    private CompletableFuture<?> rollbackMetrics(MetabaseShard shard) {
        return new RollbackActor(shard).start();
    }

    private CompletableFuture<Long> getMetricCount() {
        return retry(() -> deletedMetricsDao.count(params.getOperationId(), params.getNumId()));
    }

    private void updateProgress(boolean complete) {
        RollbackDeletedMetricsProgress prev;
        RollbackDeletedMetricsProgress update;
        do {
            prev = progress.get();
            var total = this.total.get();
            var stillDeleted = Math.max(this.stillDeleted.get(), 0);
            var progress = total > 0
                ? (total - stillDeleted) / (double) total
                : 1;
            update = RollbackDeletedMetricsProgress.newBuilder()
                .setComplete(complete || prev.getComplete())
                .setTotalMetrics(total)
                .setStillDeletedMetrics(stillDeleted)
                .setProgress(progress)
                .build();
        } while (!progress.compareAndSet(prev, update));
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryCtx.retry(supplier);
    }

    @Override
    public void close() {
        retryCtx.close();
    }

    @ParametersAreNonnullByDefault
    private class RollbackActor {

        private final MetabaseShard shard;
        private final AsyncActorRunner actorRunner;

        @Nullable
        private volatile CoremonMetricArray metrics;

        RollbackActor(MetabaseShard shard) {
            this.shard = shard;
            this.actorRunner = new AsyncActorRunner(this::body, executor, 1);
        }

        CompletableFuture<?> start() {
            return nextMetrics(null)
                .thenCompose(i -> actorRunner.start())
                .whenComplete((i, t) -> {
                    var metrics = this.metrics;
                    if (metrics != null) {
                        metrics.close();
                    }
                });
        }

        private CompletableFuture<?> body() {
            try (var metrics = this.metrics) {
                assert metrics != null;
                this.metrics = null; // takes ownership

                if (metrics.isEmpty()) {
                    return completedFuture(AsyncActorBody.DONE_MARKER);
                }

                if (interrupted) {
                    return processMetrics(metrics, this::tryProcessMetricsNoWrite);
                }

                if (shard.reachFileMetricsQuota()) {
                    throw Status.RESOURCE_EXHAUSTED
                        .withDescription("more than " + shard.maxFileMetrics() + " metrics in shard " + shard.getId())
                        .asRuntimeException();
                }

                return processMetrics(metrics, this::tryProcessMetrics);
            }
        }

        private CompletableFuture<?> processMetrics(
            CoremonMetricArray metrics,
            BiFunction<CoremonMetricArray, CompletableFuture<?>, CompletableFuture<?>> processor)
        {
            var lastKey = metrics.getLabels(metrics.size() - 1);
            var nextMetricsFuture = nextMetrics(lastKey);

            try {
                return processor.apply(metrics, nextMetricsFuture);
            } catch (Throwable t) {
                return allOf(failedFuture(t), nextMetricsFuture);
            }
        }

        private CompletableFuture<?> tryProcessMetricsNoWrite(
            CoremonMetricArray metrics,
            CompletableFuture<?> nextMetricsFuture)
        {
            var keys = resolveProperlyWritten(metrics);
            var deleteFuture = deleteFromDeletedMetrics(keys)
                .whenComplete((count, t) -> {
                    if (t == null) {
                        stillDeleted.addAndGet(-count);
                        updateProgress(false);
                    }
                });

            return allOf(nextMetricsFuture, deleteFuture);
        }

        private CompletableFuture<?> tryProcessMetrics(
            CoremonMetricArray metrics,
            CompletableFuture<?> nextMetricsFuture)
        {
            var fileMetrics = shard.getStorage().getFileMetrics();

            CompletableFuture<Void> writeFuture;
            //TODO: reuse incoming CoremonMetricArray if no metrics actually filtered out
            try (var batch = new CoremonMetricArray(metrics.size())) {
                for (int i = 0; i < metrics.size(); i++) {
                    var labels = metrics.getLabels(i);

                    try (var existing = fileMetrics.getOrNull(labels)) {
                        if (existing == null) {
                            batch.add(
                                metrics.getShardId(i),
                                metrics.getLocalId(i),
                                labels,
                                metrics.getCreatedAtSeconds(i),
                                metrics.getType(i));
                        }
                    }
                }

                writeFuture = shard.getStorage().write(batch.retain());
            }

            metrics.retain();
            var writeAndDeleteFuture = writeFuture
                .thenCompose(keys -> deleteFromDeletedMetrics(resolveProperlyWritten(metrics)))
                .whenComplete((count, t) -> {
                    metrics.release();
                    if (t == null) {
                        stillDeleted.addAndGet(-count);
                        updateProgress(false);
                    }
                });
            return allOf(nextMetricsFuture, writeAndDeleteFuture);
        }

        private Collection<Labels> resolveProperlyWritten(CoremonMetricArray metrics) {
            return MetricsResolver.resolveExisting(metrics, shard.getStorage().getFileMetrics());
        }

        private CompletableFuture<Integer> deleteFromDeletedMetrics(Collection<Labels> keys) {
            return retry(() -> deletedMetricsDao.delete(params.getOperationId(), params.getNumId(), keys))
                .thenApply(i -> keys.size());
        }

        private CompletableFuture<?> nextMetrics(@Nullable Labels lastKey) {
            var nextMetrics = new CoremonMetricArray(findMetricsLimit);
            return findMetrics(nextMetrics, lastKey)
                .whenComplete((i, t) -> {
                    if (t == null) {
                        this.metrics = nextMetrics;
                    } else {
                        nextMetrics.close();
                    }
                });
        }

        private CompletableFuture<Void> findMetrics(
            CoremonMetricArray buffer,
            @Nullable Labels lastKey)
        {
            return retry(
                () -> deletedMetricsDao.find(
                    params.getOperationId(),
                    params.getNumId(),
                    findMetricsLimit,
                    lastKey,
                    buffer,
                    shard.getLabelAllocator()));
        }
    }

}
