package ru.yandex.solomon.coremon.tasks.removeShard;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import ru.yandex.coremon.api.task.RemoveShardProgress.RemoveMeta;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoveShardFromMetabase implements AutoCloseable {
    private final RetryContext retryContext;
    private final MetricsDao dao;

    private final AtomicReference<RemoveMeta> progress;
    private final AtomicInteger removedMetrics;
    private final AtomicInteger totalMetrics;

    public RemoveShardFromMetabase(RetryConfig retry, MetricsDao dao, RemoveMeta progress) {
        this.retryContext = new RetryContext(retry);
        this.dao = dao;
        this.progress = new AtomicReference<>(progress);
        this.removedMetrics = new AtomicInteger(progress.getRemovedMetrics());
        this.totalMetrics = new AtomicInteger(progress.getTotalMetrics());
    }

    public CompletableFuture<Void> start() {
        if (progress.get().getComplete()) {
            return CompletableFuture.completedFuture(null);
        }

        try {
            return updateMetricCount()
                    .thenCompose(ignore -> removeMetrics())
                    .thenCompose(ignore -> updateMetricCount())
                    .whenComplete((ignore, e) -> updateProgress(e == null));
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public RemoveMeta progress() {
        return progress.get();
    }

    private CompletableFuture<Void> updateMetricCount() {
        return retry(() -> dao.getMetricCount()
                .thenAccept(count -> {
                    totalMetrics.set(Math.max(totalMetrics.get(), Math.toIntExact(count)));
                    removedMetrics.set(totalMetrics.get() - Math.toIntExact(count));
                    updateProgress(false);
                }));
    }

    private CompletableFuture<?> removeMetrics() {
        CompletableFuture<?> doneFuture = new CompletableFuture<>();
        continueRemoveMetrics(doneFuture);
        return doneFuture;
    }

    private void continueRemoveMetrics(CompletableFuture<?> doneFuture) {
        retry(dao::deleteMetricsBatch)
                .whenComplete((count, e) -> {
                    if (e != null) {
                        doneFuture.completeExceptionally(e);
                    }

                    try {
                        removedMetrics.addAndGet(Math.toIntExact(count));
                        updateProgress(false);
                        if (count > 0) {
                            continueRemoveMetrics(doneFuture);
                        } else {
                            doneFuture.complete(null);
                        }
                    } catch (Throwable e2) {
                        doneFuture.completeExceptionally(e2);
                    }
                });
    }

    private void updateProgress(boolean complete) {
        RemoveMeta prev;
        RemoveMeta update;
        do {
            prev = progress.get();
            var removed = removedMetrics.get();
            var total = totalMetrics.get();
            update = RemoveMeta.newBuilder()
                    .setComplete(complete || prev.getComplete())
                    .setProgress(total > 0 ? (double) removed / (double) total : 1)
                    .setRemovedMetrics(removed)
                    .setTotalMetrics(total)
                    .build();
        } while (!progress.compareAndSet(prev, update));
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryContext.retry(supplier);
    }

    @Override
    public void close() {
        retryContext.close();
    }
}
