package ru.yandex.solomon.coremon.tasks.removeShard;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import com.google.protobuf.TextFormat;
import io.grpc.Status;

import ru.yandex.coremon.api.task.RemoveShardParams;
import ru.yandex.coremon.api.task.RemoveShardProgress.RemoveData;
import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TDeleteMetricByShardRequest;
import ru.yandex.stockpile.api.TDeleteMetricByShardRequest.TOwnerShard;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoveShardFromStockpile implements AutoCloseable {
    private final RetryContext retryContext;
    private final StockpileClient stockpile;
    private final RemoveShardParams params;

    private final AtomicReference<RemoveData> progress;
    private final AtomicInteger lastShardId;

    private final AsyncActorRunner actor;
    private final ConcurrentLinkedQueue<Integer> inflight = new ConcurrentLinkedQueue<>();

    public RemoveShardFromStockpile(RetryConfig retry, StockpileClient stockpile, Executor executor, RemoveShardParams params, RemoveData progress) {
        this.retryContext = new RetryContext(retry);
        this.stockpile = stockpile;
        this.params = params;
        this.progress = new AtomicReference<>(progress);
        this.lastShardId = new AtomicInteger(progress.getLastShardId());
        this.actor = new AsyncActorRunner(this::removeNext, executor, 100);
    }

    public CompletableFuture<Void> start() {
        if (progress.get().getComplete()) {
            return CompletableFuture.completedFuture(null);
        }

        return actor.start().whenComplete((ignore, e) -> updateProgress(e == null));
    }

    private CompletableFuture<?> removeNext() {
        if (lastShardId.get() >= stockpile.getTotalShardsCount()) {
            return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
        }

        return removeFromStockpile(lastShardId.incrementAndGet());
    }

    private void updateProgress(boolean complete) {
        RemoveData prev;
        RemoveData update;
        do {
            prev = progress.get();
            var lastShardId = lastCompleteShard();
            update = RemoveData.newBuilder()
                    .setComplete(complete || prev.getComplete())
                    .setProgress((double) lastShardId / (double) stockpile.getTotalShardsCount())
                    .setLastShardId(lastShardId)
                    .build();
        } while (!progress.compareAndSet(prev, update));
    }

    public RemoveData progress() {
        return progress.get();
    }

    private int lastCompleteShard() {
        int lastShardId = this.lastShardId.get();
        var inflight = this.inflight.peek();
        if (inflight != null) {
            return inflight - 1;
        }

        return lastShardId;
    }

    private CompletableFuture<Void> removeFromStockpile(int shardId) {
        inflight.add(shardId);
        var req = TDeleteMetricByShardRequest.newBuilder()
                .setShardId(shardId)
                .addOwnerShards(TOwnerShard.newBuilder()
                        .setProjectId(EProjectId.SOLOMON_VALUE)
                        .setShardId(params.getNumId())
                        .build())
                .addOwnerShards(TOwnerShard.newBuilder()
                        .setProjectId(EProjectId.GOLOVAN_VALUE)
                        .setShardId(params.getNumId())
                        .build())
                .build();

        return retry(() -> stockpile.deleteMetricByShard(req)
                .thenAccept(response -> {
                    if (response.getStatus() != EStockpileStatusCode.OK) {
                        String error = "Unable remove shard metrics for " + TextFormat.shortDebugString(params) +
                                " from stockpile shard " + shardId +
                                " caused by " + response.getStatus() +
                                ":" + response.getStatusMessage();
                        throw new StatusRuntimeExceptionNoStackTrace(Status.ABORTED.withDescription(error));
                    }
                }))
                .thenAccept(ignore -> {
                    inflight.remove(shardId);
                    updateProgress(false);
                });
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryContext.retry(supplier);
    }

    @Override
    public void close() {
        retryContext.close();
    }
}
