package ru.yandex.solomon.coremon.tasks.removeShard;

import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nullable;

import io.grpc.Status;
import io.grpc.protobuf.StatusProto;

import ru.yandex.coremon.api.task.RemoveShardParams;
import ru.yandex.coremon.api.task.RemoveShardProgress;
import ru.yandex.coremon.api.task.RemoveShardResult;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.tasks.AbstractTask;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
public final class RemoveShardTask extends AbstractTask<RemoveShardProgress> {

    private final RemoveShardFromStockpile removeFromStockpile;
    private final RemoveShardFromMetabase removeShardFromMetabase;

    RemoveShardTask(
        RetryConfig retryConfig,
        ExecutionContext context,
        RemoveShardParams params,
        RemoveShardProgress progress,
        Executor executor,
        ScheduledExecutorService timer,
        StockpileClient stockpile,
        MetricsDao metricsDao)
    {
        super(
            RemoveShardTaskHandler.TYPE,
            retryConfig,
            context,
            params,
            progress,
            executor,
            timer);

        this.removeFromStockpile = new RemoveShardFromStockpile(
            retryConfig(),
            stockpile,
            executor,
            params,
            progress.getRemoveData());
        this.removeShardFromMetabase = new RemoveShardFromMetabase(
            retryConfig(),
            metricsDao,
            progress.getRemoveMeta());
    }

    @Override
    protected CompletionStage<Void> onStart() {
        return removeFromStockpile.start()
            .thenCompose(i -> forceSave())
            .thenCompose(ignore -> removeShardFromMetabase.start());
    }

    @Override
    protected RemoveShardProgress onUpdateProgress(RemoveShardProgress prev, @Nullable Status status) {
        var update = prev.toBuilder()
            .setRemoveData(removeFromStockpile.progress())
            .setRemoveMeta(removeShardFromMetabase.progress());

        if (status != null) {
            if (status.isOk()) {
                update.clearAttempt();
                update.clearStatus();
            } else {
                update.setAttempt(prev.getAttempt() + 1);
                update.setStatus(StatusProto.fromStatusAndTrailers(status, null));
            }
        }

        return update.build();
    }

    @Override
    protected RemoveShardResult result(RemoveShardProgress progress) {
        return RemoveShardResult.newBuilder()
            .setRemovedMetrics(progress.getRemoveMeta().getRemovedMetrics())
            .build();
    }

    @Override
    protected int attempt(RemoveShardProgress progress) {
        return progress.getAttempt();
    }

    @Override
    protected void onClose() {
        removeFromStockpile.close();
        removeShardFromMetabase.close();
    }
}
