package ru.yandex.solomon.coremon.tasks.removeShard;

import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.Descriptors.Descriptor;
import io.grpc.Status;

import ru.yandex.coremon.api.task.RemoveShardParams;
import ru.yandex.coremon.api.task.RemoveShardResult;
import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.Permit;
import ru.yandex.solomon.scheduler.PermitLimiter;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.TaskHandler;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class RemoveShardTaskHandler implements TaskHandler {
    static final String TYPE = "remove_shard";
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(10)
            .withDelay(TimeUnit.SECONDS.toMillis(1))
            .withMaxDelay(TimeUnit.MINUTES.toMillis(1));

    private final PermitLimiter permitLimiter = new PermitLimiter(10);
    private final SolomonConfHolder confHolder;
    private final StockpileClient stockpile;
    private final MetricsDaoFactory daoFactory;
    private final Executor executor;
    private final ScheduledExecutorService timer;

    // for manager ui
    private final ConcurrentMap<String, RemoveShardTask> running = new ConcurrentHashMap<>();

    public RemoveShardTaskHandler(
            SolomonConfHolder confHolder,
            StockpileClient stockpile,
            MetricsDaoFactory daoFactory,
            Executor executor,
            ScheduledExecutorService timer)
    {
        this.confHolder = confHolder;
        this.stockpile = stockpile;
        this.daoFactory = daoFactory;
        this.executor = executor;
        this.timer = timer;
    }

    @Override
    public String type() {
        return TYPE;
    }

    @Override
    @Nullable
    public Permit acquire(String id, Any params) {
        if (confHolder.getConf() == null) {
            return null;
        }

        return permitLimiter.acquire();
    }

    @Override
    public void execute(ExecutionContext context) {
        var task = context.task();
        var params = RemoveShardTaskProto.params(task.params());
        if (isShardStillExist(params.getNumId())) {
            context.fail(new StatusRuntimeExceptionNoStackTrace(Status.ABORTED.withDescription("Unable remove active shard")));
            return;
        }
        var progress = RemoveShardTaskProto.progress(context.task().progress());

        var metricsDao = daoFactory.create(params.getNumId(), new InterningLabelAllocator());
        var process = new RemoveShardTask(RETRY_CONFIG, context, params, progress, executor, timer, stockpile, metricsDao);
        running.put(task.id(), process);
        try {
            process.start().whenComplete((ignore, e) -> {
                running.remove(task.id(), process);
                process.close();
            });
        } catch (Throwable e) {
            running.remove(task.id(), process);
            throw new RuntimeException(e);
        }
    }

    @Override
    public List<Descriptor> descriptors() {
        return List.of(RemoveShardParams.getDescriptor(), RemoveShardResult.getDescriptor());
    }

    private boolean isShardStillExist(int numId) {
        var conf = confHolder.getConfOrThrow().getShardByNumIdOrNull(numId);
        return conf != null;
    }

    public static Task removeShardTask(String projectId, String shardId, int numId) {
        var params = Any.pack(RemoveShardParams.newBuilder()
                .setNumId(numId)
                .setProjectId(projectId)
                .setShardId(shardId)
                .build());

        long executeAt = System.currentTimeMillis()
                + TimeUnit.DAYS.toMillis(7L)
                + DurationUtils.randomize(TimeUnit.DAYS.toMillis(1L));

        return Task.newBuilder()
                .setId(TYPE + "_" + Integer.toUnsignedLong(numId))
                .setType(TYPE)
                .setExecuteAt(executeAt)
                .setParams(params)
                .build();
    }

}
