package ru.yandex.solomon.coremon.www;

import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.coremon.CoremonShard;
import ru.yandex.solomon.coremon.CoremonState;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.html.HtmlProgressBar;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.WritableToHtml;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.util.time.DurationUtils;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
@Import({
    ManagerWriterContext.class,
    CoremonState.class
})
public class CoremonLocalShardsWww {
    @Autowired
    private HttpAuthenticator authenticator;
    @Autowired
    private InternalAuthorizer authorizer;
    @Autowired
    private ManagerWriterContext contect;
    @Autowired
    private CoremonState shards;

    @Bean
    public RootLink coremonLocalShardLinks() {
        return new RootLink("/local-shards", "Local shards");
    }

    @RequestMapping(value = "/local-shards", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> localShards(
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        @RequestParam(value = "limit", defaultValue = "100") int limit,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authSubject -> authorizer.authorize(authSubject))
            .thenApply(account -> localShardsImpl(sortBy, limit));
    }

    private String localShardsImpl(int sortBy, int limit) {
        List<Record> records = this.shards.getInitializedShards()
            .map(Record::of)
            .sorted((o1, o2) -> Integer.compareUnsigned(o1.numId, o2.numId))
            .collect(Collectors.toList());

        boolean hasNotLoaded = records.stream().anyMatch(record -> record.loadProgress != 100.0);
        List<Column<Record>> columns = makeColumns(hasNotLoaded);
        return new Table<>("Local shards(" + DataSize.shortString(records.size()) + ")", contect, columns, records, sortBy, limit).genString();
    }

    private List<Column<Record>> makeColumns(boolean hasNotLoaded) {
        return List.of(
            Column.of(
                "NumId",
                r -> {
                    String shardId = Integer.toUnsignedString(r.numId);
                    return new AHref(namedObjectLink(new NamedObjectId(CoremonShard.class, shardId)), shardId);
                },
                (o1, o2) -> Integer.compareUnsigned(o1.numId, o2.numId)),
            Column.of(
                "ShardId",
                r -> {
                    return new AHref(namedObjectLink(new NamedObjectId(CoremonShard.class, r.id)), r.id);
                },
                Comparator.comparing(r -> r.id)),
            Column.of("State", r -> r.state, Comparator.comparing(o2 -> o2.state)),
            Column.of(
                hasNotLoaded ? "Load" : "",
                r -> {
                    if (r.state != EShardState.READY) {
                        return new HtmlProgressBar(r.loadProgress);
                    }

                    return "";
                },
                Comparator.comparingDouble(o2 -> o2.loadProgress)),
            Column.of("Uptime",
                r -> DurationUtils.formatDurationMillisTruncated(r.uptimeMillis),
                Comparator.comparingLong(o2 -> o2.uptimeMillis)
            ),
            Column.of("CpuMs/sec",
                r -> Long.toString(r.cpu),
                Comparator.comparingDouble(o2 -> o2.cpu)
            ),
            Column.of("Metrics",
                r -> DataSize.shortString(r.metrics),
                Comparator.comparingLong(o2 -> o2.metrics)
            ),
            Column.of("ParseMetrics/sec",
                    r -> DataSize.shortString(r.parsedMetrics),
                    Comparator.comparingLong(o2 -> o2.parsedMetrics)
            ),
            Column.of("InboundBytes/sec",
                    r -> DataSize.shortString(r.inboundBytes)+"B",
                    Comparator.comparingLong(o2 -> o2.inboundBytes)
            ),
            Column.of("Errors",
                r -> DataSize.shortString(r.errors),
                Comparator.comparingLong(o2 -> o2.errors)
            ),
            Column.of("Last Error",
                r -> {
                  if (r.lastErrorInstant == 0) {
                      return "none";
                  } else {
                      return Instant.ofEpochMilli(r.lastErrorInstant);
                  }
                },
                Comparator.comparingLong(o2 -> o2.lastErrorInstant)
            ),
            Column.of("Kick", r -> {
                return (WritableToHtml) mw -> {
                    var hw = mw.getHtmlWriter();
                    hw.aHref("/balancer-old/kickShard?allowNewBalancer=true&shardId=" + Integer.toUnsignedLong(r.numId), () -> {
                        hw.tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-transfer"));
                    });
                };
            }, (o1, o2) -> Integer.compareUnsigned(o1.numId, o2.numId))
        );
    }

    private static class Record implements TableRecord {
        int numId;
        String id;
        EShardState state;
        long loadedRows;
        long estimateRows;
        double loadProgress;
        long uptimeMillis;
        long cpu;
        long metrics;
        long inboundBytes;
        long parsedMetrics;
        long errors;
        long lastErrorInstant;

        public static Record of(CoremonShard shard) {
            var r = new Record();
            r.numId = shard.getNumId();
            r.id = shard.getProcessingShard().getId();
            r.state = shard.getProcessingShard().getState();

            r.loadedRows = shard.getMetabaseShard().getStorage().getRowsLoaded();
            r.estimateRows = shard.getMetabaseShard().getStorage().getEstimatedRowsTotal();
            r.loadProgress = (r.loadedRows * 100.0) / (double) r.estimateRows;
            if (r.state == EShardState.READY) {
                r.loadProgress = 100;
            } else if (r.state == EShardState.INDEXING) {
                r.loadProgress = Math.min(r.loadProgress, 99.9);
            }

            var load = shard.getProcessingShard().getLoad();
            r.uptimeMillis = load.getUptimeMillis();
            r.cpu = TimeUnit.NANOSECONDS.toMillis(load.getCpuTimeNanos());
            r.metrics = load.getMetricsCount();
            if (r.metrics == 0) {
                r.metrics = r.loadedRows;
            }
            r.inboundBytes = load.getNetworkBytes();
            r.parsedMetrics = load.getParsedMetrics();

            r.errors = shard.getProcessingShard().getMetrics().errors.get();
            r.lastErrorInstant = shard.getProcessingShard().getLastExceptionInstant();
            return r;
        }
    }
}
