package ru.yandex.solomon.coremon;

import java.util.Collections;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.google.common.util.concurrent.MoreExecutors;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.coremon.TMetabaseMetricNameMigrationConfig;
import ru.yandex.solomon.core.conf.ClusterConfDetailed;
import ru.yandex.solomon.core.conf.ServiceConfDetailed;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.ClusterHostUrlConf;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.db.model.ValidationMode;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.stockpile.MetricProcessorFactory;
import ru.yandex.solomon.coremon.stockpile.write.StockpileBufferedWriter;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.metrics.client.StockpileClientStub;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class CoremonShardFactoryTest {

    @Rule
    public TestName testName = new TestName();

    private ExecutorService executorService;
    private CoremonShardFactory factory;

    @Before
    public void setUp() {
        this.executorService = Executors.newFixedThreadPool(2);
        StockpileClientStub stockpileClient = new StockpileClientStub(MoreExecutors.newDirectExecutorService());
        this.factory = new CoremonShardFactory(
            executorService,
            executorService,
            new InMemoryMetricsDaoFactory(),
            stockpileClient,
            new MetricProcessorFactory(new FeatureFlagHolderStub()),
            TMetabaseMetricNameMigrationConfig.getDefaultInstance(),
            new StockpileBufferedWriter(stockpileClient, executorService, MetricRegistry.root()));
    }

    @After
    public void tearDown() {
        executorService.shutdownNow();
    }

    @Test
    public void sameCfg() {
        ShardConfDetailed conf = newShardConf();

        CoremonShard one = factory.createShard(conf);
        assertNotNull(one);

        assertTrue(factory.updateShard(one, conf));
        assertTrue(factory.updateShard(one, newShardConf()));
    }

    @Test
    public void changeClusterNotReloadShard() {
        ShardConfDetailed init = newShardConf();

        CoremonShard one = factory.createShard(init);
        ShardConfDetailed changedCluster = patchConf(init, init.getCluster()
                .getRaw()
                .toBuilder()
                .setHostUrls(Collections.singletonList(ClusterHostUrlConf.of("solomon-test.yandex-team.ru", new String[0], true)))
                .build());

        assertTrue(factory.updateShard(one, changedCluster));
    }

    @Test
    public void changedMetricsUrlNotReloadShard() {
        ShardConfDetailed init = newShardConf();

        CoremonShard one = factory.createShard(init);
        ShardConfDetailed changed = patchConf(init, init.getService()
                .getRaw()
                .toBuilder()
                .setShardSettings(ShardSettings.of(ShardSettings.Type.PULL,
                        ShardSettings.PullSettings.newBuilder().setPath("/metrics/spack").setPort(8181).build(),
                        15,
                        37,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, new ServiceMetricConf.AggrRule[0], false),
                        15))
                .build());

        assertTrue(factory.updateShard(one,changed));
    }

    @Test
    public void addNewShardHostsNotReloadShard() {
        ShardConfDetailed init = newShardConf();

        CoremonShard one = factory.createShard(init);
        ShardConfDetailed updated = patchConf(init, init.getRaw());

        assertTrue(factory.updateShard(one, updated));
    }

    @Test
    public void changeAggregateReloadShard() {
        ShardConfDetailed init = newShardConf();

        CoremonShard one = factory.createShard(init);

        ShardConfDetailed v1 = patchConf(init, init.getService().getRaw()
                .toBuilder()
                .setMetricConf(ServiceMetricConf.of(new ServiceMetricConf.AggrRule[]{
                        AggrRule.of("host=*", "host=cluster", null)}, false))
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, new ServiceMetricConf.AggrRule[]{
                                AggrRule.of("host=*", "host=cluster", null)}, false),
                        11))
                .build());

        assertFalse(factory.updateShard(one, v1));
        CoremonShard two = factory.createShard(v1);
        assertNotSame(one, two);


        ShardConfDetailed v2 = patchConf(init, init.getService().getRaw()
                .toBuilder()
                .setMetricConf(ServiceMetricConf.of(new ServiceMetricConf.AggrRule[]{
                        AggrRule.of("host=*, dist=*", "host=cluster, dist=total", null)}, true))
                .build());

        assertFalse(factory.updateShard(two, v2));
        CoremonShard tree = factory.createShard(v2);
        assertNotSame(two, tree);
        assertNotSame(one, tree);
    }

    @Test
    public void changeValidationModeNotReloadShard() {
        ShardConfDetailed init = newShardConf();

        CoremonShard one = factory.createShard(init);

        ShardConfDetailed updated = patchConf(init, init.getRaw()
            .toBuilder()
            .setMaxMetricsPerUrl(1234)
            .setMaxFileMetrics(2345)
            .setMaxMemMetrics(3456)
            .setValidationMode(ValidationMode.STRICT_FAIL)
            .build());

        assertTrue(factory.updateShard(one, updated));

        CoremonShardOpts opts = one.getProcessingShard().getOpts();
        Assert.assertEquals(ValidationMode.STRICT_FAIL, opts.getValidationMode());
        Assert.assertEquals(new CoremonShardQuota(1234, 2345, 3456), opts.getQuota());
        Assert.assertEquals(1000, opts.getFetchMillis());
    }

    @Test
    public void changeIntervalNotReloadShard() {
        ShardConfDetailed init = newShardConf();

        CoremonShard one = factory.createShard(init);

        ShardConfDetailed updated = patchConf(init, init.getService()
            .getRaw()
            .toBuilder()
                .setShardSettings(ShardSettings.of(ShardSettings.Type.PULL,
                        ShardSettings.PullSettings.newBuilder().setPath("/metrics/json").setPort(8181).build(),
                        0,
                        37,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, new ServiceMetricConf.AggrRule[0], false),
                        12))
            .build());

        assertTrue(factory.updateShard(one, updated));
        Assert.assertEquals(12 * 1000, one.getProcessingShard().getOpts().getFetchMillis());
        Assert.assertEquals(12 * 1000, one.getProcessingShard().getOpts().getGridMillis());
    }

    private ShardConfDetailed newShardConf() {
        String projectId = "solomon";

        Project project = Project.newBuilder()
            .setId("solomon")
            .setName("solomon")
            .setOwner("user")
            .build();

        Cluster cluster = Cluster.newBuilder()
                .setId("c_" + testName.getMethodName())
                .setName("c_" + testName.getMethodName() + "_name")
                .setProjectId(projectId)
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.EMPTY,
                        0))
                .build();

        Service service = Service.newBuilder()
                .setId("s_" + testName.getMethodName())
                .setName("s_" + testName.getMethodName() + "_name")
                .setProjectId(projectId)
                .setShardSettings(ShardSettings.of(ShardSettings.Type.PULL,
                        ShardSettings.PullSettings.newBuilder().setPath("/metrics/json").setPort(8181).build(),
                        15,
                        37,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, new ServiceMetricConf.AggrRule[0], false),
                        1))
                .build();

        String shardId = "my_shard_" + testName.getMethodName();
        Shard shard = Shard.newBuilder()
                .setId(shardId)
                .setNumId(shardId.hashCode())
                .setClusterId(cluster.getId())
                .setClusterName(cluster.getName())
                .setServiceId(service.getId())
                .setServiceName(service.getName())
                .setProjectId(projectId)
                .setValidationMode(ValidationMode.STRICT_SKIP)
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.EMPTY,
                        0))
                .build();

        return newShardConf(shard, project, cluster, service);
    }

    private ShardConfDetailed newShardConf(Shard shard, Project project, Cluster cluster, Service service) {
        return new ShardConfDetailed(shard, project, new ClusterConfDetailed(cluster), new ServiceConfDetailed(service, null));
    }

    private ShardConfDetailed patchConf(ShardConfDetailed init, Shard shard) {
        return newShardConf(shard, init.getProject(), init.getCluster().getRaw(), init.getService().getRaw());
    }

    private ShardConfDetailed patchConf(ShardConfDetailed init, Cluster cluster) {
        return newShardConf(init.getRaw(), init.getProject(), cluster, init.getService().getRaw());
    }

    private ShardConfDetailed patchConf(ShardConfDetailed init, Service service) {
        return newShardConf(init.getRaw(), init.getProject(), init.getCluster().getRaw(), service);
    }
}
