package ru.yandex.solomon.coremon.aggregates;

import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import com.google.common.collect.Iterables;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.conf.aggr.AggrRuleConf;
import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpileResolveHelper;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpileWriteHelper;
import ru.yandex.solomon.coremon.stockpile.TestResolveHelper;
import ru.yandex.solomon.coremon.stockpile.TestStockpileWriteHelper;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricPoint;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPoints.point;

/**
 * @author Sergey Polovko
 */
public class AggrHelperTest {

    @Test
    public void processSingleRequestAggr() throws Exception {
        AggrHelper aggrHelper = new AggrHelper(
            new AggrRuleConf[] {
                AggrRuleConf.simpleReplaceRule("host", "cluster"),
                AggrRuleConf.simpleReplaceRule("host", "{{DC}}"),
            },
            EDecimPolicy.UNDEFINED
        );

        TestResolveHelper resolver = new TestResolveHelper(
            makeMetric(1, 1337, Labels.of("host", "my-host1", "sensor", "reads_count")),
            makeMetric(1, 1338, Labels.of("host", "my-host2", "sensor", "reads_count")),
            makeMetric(2, 1339, Labels.of("host", "cluster", "sensor", "reads_count"))
        );
        TestStockpileWriteHelper writeHelper = new TestStockpileWriteHelper();

        UnresolvedMetricPoint metricData = new UnresolvedMetricPoint(
            Labels.of("host", "my-host1", "sensor", "reads_count"),
            MetricType.DGAUGE,
            AggrPoint.shortPoint(System.currentTimeMillis(), 10, TimeUnit.SECONDS.toMillis(15)));

        CoremonMetric metric = resolver.tryResolveMetric(metricData.getLabels(), metricData.getType());
        Assert.assertNotNull(metric);

        AggrPoint point = new AggrPoint();
        metricData.get(0, point);

        aggrHelper.aggregatePoint(
            resolver, writeHelper,
            metricData.getLabels(),
            point,
            metricData.getType(),
            metricData.getWriteType(),
            metric,
            Labels.of("type", "production", "DC", "man"));

        // resolved aggr metrics
        {
            assertEquals(1, writeHelper.size());

            AggrGraphDataArrayList graphData = writeHelper.getPoints(2, 1339);
            assertEquals(1, graphData.getRecordCount());
            AggrPoint savedPoint = graphData.getAnyPoint(0);
            assertEquals(metricData.point().getValueNum(), savedPoint.getValueNum(), Double.MIN_VALUE);
            assertEquals(metricData.point().getValueDenom(), savedPoint.getValueDenom());
        }

        // not resolved aggr metrics
        {
            var aggrRequestsByHost = resolver.getUnresolvedAggrMetricsDataByHost();
            assertEquals(1, aggrRequestsByHost.size());

            var aggrRequests = aggrRequestsByHost.get("man");
            assertEquals(1, aggrRequests.size());

            var aggrRequest = (UnresolvedMetricPoint) Iterables.getFirst(aggrRequests, null);;
            assertEquals(Labels.of("host", "man", "sensor", "reads_count"), aggrRequest.getLabels());
            assertEquals(metricData.point().getValueNum(), aggrRequest.point().getValueNum(), Double.MIN_VALUE);
            assertEquals(metricData.point().getValueDenom(), aggrRequest.point().getValueDenom());
        }
    }

    @Test
    public void targetMetricsEmptyRules() throws Exception {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[0], EDecimPolicy.UNDEFINED);
        Labels[] target = targetMetrics(aggrHelper,
            Labels.of("host", "cluster", "sensor", "reads_count"),
            Labels.of("host", "man", "sensor", "reads_count"));
        assertEquals(0, target.length);
    }

    @Test
    public void targetMetricsNonMatchesRuleCond() throws Exception {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[] {
            AggrRuleConf.simpleReplaceRule("users", "total")
        }, EDecimPolicy.UNDEFINED);
        Labels[] target = targetMetrics(aggrHelper,
            Labels.of("host", "cluster", "sensor", "reads_count"),
            Labels.of("host", "man", "sensor", "reads_count"));
        assertEquals(0, target.length);
    }

    @Test
    public void targetMetricsSameTarget() throws Exception {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[] {
            AggrRuleConf.simpleReplaceRule("host", "my-awesome-host")
        }, EDecimPolicy.UNDEFINED);
        Labels[] target = targetMetrics(aggrHelper,
            Labels.of("host", "my-awesome-host", "sensor", "reads_count"),
            Labels.of("type", "production"));
        assertEquals(0, target.length);
    }

    @Test
    public void targetMetricsMatchesRuleCond() throws Exception {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[] {
            AggrRuleConf.simpleReplaceRule("host", "cluster")
        }, EDecimPolicy.UNDEFINED);
        Labels[] target = targetMetrics(aggrHelper,
            Labels.of("host", "my-host", "sensor", "reads_count"),
            Labels.of("DC", "man"));
        assertEquals(1, target.length);
        assertEquals(Labels.of("host", "cluster", "sensor", "reads_count"), target[0]);
    }

    @Test
    public void targetMetricsMatchesWithExprEval() throws Exception {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[] {
            AggrRuleConf.simpleReplaceRule("host", "{{DC}}")
        }, EDecimPolicy.UNDEFINED);
        Labels[] target = targetMetrics(aggrHelper,
            Labels.of("host", "my-host", "sensor", "reads_count"),
            Labels.of("DC", "man"));
        assertEquals(1, target.length);
        assertEquals(Labels.of("host", "man", "sensor", "reads_count"), target[0]);
    }

    @Test
    public void targetMetricsExprEvalFailed() throws Exception {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[] {
            AggrRuleConf.simpleReplaceRule("host", "{{DC}}")
        }, EDecimPolicy.UNDEFINED);
        Labels[] target = targetMetrics(aggrHelper,
            Labels.of("host", "my-host", "sensor", "reads_count"),
            Labels.of("type", "production"));
        assertEquals(0, target.length);
    }

    @Test
    public void allowToAddAbsentLabels() {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[] {
            makeAggrRule("host=-", "host=Aggr")
        }, EDecimPolicy.UNDEFINED);

        Labels oldLabels = Labels.of(
            "method", "FlushBlocks",
            "sensor", "yt.rpc.server.request_message_attachment_bytes.rate",
            "user", "jamel");

        var result = aggrHelper.targetMetricsSet(oldLabels, Labels.empty());
        assertEquals(Map.of(oldLabels.add("host", "Aggr"), MetricAggregation.SUM), result);
    }

    @Test
    public void doNotAllowToAddNewLabels() {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[] {
            makeAggrRule("host=*", "host=Aggr")
        }, EDecimPolicy.UNDEFINED);

        Labels oldLabels = Labels.of(
            "method", "FlushBlocks",
            "sensor", "yt.rpc.server.request_message_attachment_bytes.rate",
            "user", "jamel");

        var result = aggrHelper.targetMetricsSet(oldLabels, Labels.empty());
        assertEquals(Map.of(), result);
    }

    @Test
    public void dropMatchedLabel() {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[] {
            makeAggrRule("host=*", "host=-")
        }, EDecimPolicy.UNDEFINED);

        Labels[] target = targetMetrics(aggrHelper, Labels.of("host", "my-host", "sensor", "reads_count"), Labels.empty());

        assertEquals(1, target.length);
        assertEquals(Labels.of("sensor", "reads_count"), target[0]);
    }

    @Test
    public void dropNotMatchedLabel() {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[] {
            makeAggrRule("host=*", "sensor=-")
        }, EDecimPolicy.UNDEFINED);

        Labels[] target = targetMetrics(aggrHelper, Labels.of("host", "my-host", "sensor", "reads_count"), Labels.empty());

        assertEquals(1, target.length);
        assertEquals(Labels.of("host", "my-host"), target[0]);
    }

    @Test
    public void updatedOptLabels() {
        AggrHelper helper = new AggrHelper(new AggrRuleConf[] {
            AggrRuleConf.simpleReplaceRule("host", "{{DC}}")
        }, EDecimPolicy.UNDEFINED);


        CoremonMetric aggrMan = makeMetric(1, 1, Labels.of("host", "Man", "sensor", "reads_count"));
        CoremonMetric aggrSas = makeMetric(2, 2, Labels.of("host", "Sas", "sensor", "reads_count"));
        CoremonMetric metric = makeMetric(3, 3, Labels.of("host", "my-host-00", "sensor", "reads_count"));
        var resolver = new TestResolveHelper(aggrMan, aggrSas, metric);

        long tsMillis = System.currentTimeMillis();
        var aggrPoint = AggrPoint.shortPoint(tsMillis, 3.14);

        // first process (DC=Man)
        {
            TestStockpileWriteHelper writeHelper = new TestStockpileWriteHelper();
            Labels optLabels = Labels.of("DC", "Man");

            helper.aggregatePoint(
                resolver,
                writeHelper,
                metric.getLabels(),
                aggrPoint,
                metric.getType(),
                MetricTypeConverter.toNotNullProto(metric.getType()),
                metric,
                optLabels);

            Assert.assertNotNull(metric.getAggrMetrics());
            Assert.assertTrue(AggrMetrics.isInitialized(metric.getAggrMetrics()));
            assertEquals(1, AggrMetrics.size((int[]) metric.getAggrMetrics()));
            assertEquals(optLabels.hashCode(), AggrMetrics.optLabelsHash((int[]) metric.getAggrMetrics()));

            AggrGraphDataArrayList points = writeHelper.getPoints(aggrMan.getShardId(), aggrMan.getLocalId());
            Assert.assertNotNull(points);
            assertEquals(1, points.length());

            AggrPoint point = points.getAnyPoint(0);
            assertEquals(tsMillis, point.getTsMillis());
            assertEquals(3.14, point.getValueDivided(), Double.MIN_VALUE);
        }

        // first process (DC=Sas)
        {
            TestStockpileWriteHelper writeHelper = new TestStockpileWriteHelper();
            helper.aggregatePoint(
                resolver,
                writeHelper,
                metric.getLabels(),
                aggrPoint,
                metric.getType(),
                MetricTypeConverter.toNotNullProto(metric.getType()),
                metric,
                Labels.of("DC", "Sas"));

            Assert.assertNotNull(metric.getAggrMetrics());
            Assert.assertTrue(AggrMetrics.isInitialized(metric.getAggrMetrics()));

            AggrGraphDataArrayList points = writeHelper.getPoints(aggrSas.getShardId(), aggrSas.getLocalId());
            Assert.assertNotNull(points);
            assertEquals(1, points.length());

            AggrPoint point = points.getAnyPoint(0);
            assertEquals(tsMillis, point.getTsMillis());
            assertEquals(3.14, point.getValueDivided(), Double.MIN_VALUE);
        }
    }

    @Test
    public void targetMetricSum() {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[]{
                makeAggrRule("host=*", "host=cluster", MetricAggregation.SUM)
        }, EDecimPolicy.UNDEFINED);
        var targets = aggrHelper.targetMetricsSet(LabelsFormat.parse("host=test-000, metric=reads_count"), Labels.of());
        assertEquals(Map.of(LabelsFormat.parse("host=cluster, metric=reads_count"), MetricAggregation.SUM), targets);
    }

    @Test
    public void targetMetricLast() {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[]{
                makeAggrRule("host=*, metric=db.size", "host=-", MetricAggregation.LAST)
        }, EDecimPolicy.UNDEFINED);
        var targets = aggrHelper.targetMetricsSet(LabelsFormat.parse("host=test-000, metric=db.size"), Labels.of());
        assertEquals(Map.of(LabelsFormat.parse("metric=db.size"), MetricAggregation.LAST), targets);
    }

    @Test
    public void targetMetricLatestRuleWin() {
        AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[]{
                makeAggrRule("host=*", "host=-", MetricAggregation.SUM),
                makeAggrRule("host=*, metric=db.size", "host=-", MetricAggregation.LAST),
        }, EDecimPolicy.UNDEFINED);
        {
            var labels = LabelsFormat.parse("host=test-000, metric=db.size");
            var targets = aggrHelper.targetMetricsSet(labels, Labels.of());
            assertEquals(Map.of(LabelsFormat.parse("metric=db.size"), MetricAggregation.LAST), targets);
        }
        {
            var labels = LabelsFormat.parse("host=test-000, metric=db.requests.count");
            var targets = aggrHelper.targetMetricsSet(labels, Labels.of());
            assertEquals(Map.of(LabelsFormat.parse("metric=db.requests.count"), MetricAggregation.SUM), targets);
        }
    }

    @Test
    public void differentAggregationFunction() {
        var helper = new AggrHelper(new AggrRuleConf[]{
                makeAggrRule("host=*", "host=-", MetricAggregation.SUM),
                makeAggrRule("host=*, metric=db.size", "host=-", MetricAggregation.LAST),
        }, EDecimPolicy.UNDEFINED);

        CoremonMetric aggrDbSize = makeMetric("metric=db.size");
        CoremonMetric aggrDbRps = makeMetric("metric=db.requests.count");
        CoremonMetric metricDbSize = makeMetric("host=test-000, metric=db.size");
        CoremonMetric metricDbRps = makeMetric("host=test-000, metric=db.requests.count");
        var resolver = new TestResolveHelper(aggrDbSize, aggrDbRps, metricDbRps, metricDbSize);
        var writer = new TestStockpileWriteHelper();

        {
            processPoint(helper, resolver, writer, metricDbRps, point("2021-06-01T13:00:00Z", 1d, true, 1));
            processPoint(helper, resolver, writer, metricDbRps, point("2021-06-01T13:01:00Z", 2d, true, 1));
            processPoint(helper, resolver, writer, metricDbRps, point("2021-06-01T13:02:00Z", 3d, true, 1));
            processPoint(helper, resolver, writer, metricDbRps, point("2021-06-01T13:03:00Z", 4d, true, 1));

            var expected = AggrGraphDataArrayList.of(
                    point("2021-06-01T13:00:00Z", 1d, true, 1),
                    point("2021-06-01T13:01:00Z", 2d, true, 1),
                    point("2021-06-01T13:02:00Z", 3d, true, 1),
                    point("2021-06-01T13:03:00Z", 4d, true, 1)
            );
            var points = writer.getPoints(aggrDbRps.getShardId(), aggrDbRps.getLocalId());
            assertEquals("aggregation sum", expected, points);
        }

        {
            processPoint(helper, resolver, writer, metricDbSize, point("2021-06-01T13:00:00Z", 1d, true, 1));
            processPoint(helper, resolver, writer, metricDbSize, point("2021-06-01T13:01:00Z", 2d, true, 1));
            processPoint(helper, resolver, writer, metricDbSize, point("2021-06-01T13:02:00Z", 3d, true, 1));
            processPoint(helper, resolver, writer, metricDbSize, point("2021-06-01T13:03:00Z", 4d, true, 1));

            var expected = AggrGraphDataArrayList.of(
                    point("2021-06-01T13:00:00Z", 1d, false, 1),
                    point("2021-06-01T13:01:00Z", 2d, false, 1),
                    point("2021-06-01T13:02:00Z", 3d, false, 1),
                    point("2021-06-01T13:03:00Z", 4d, false, 1));
            var points = writer.getPoints(aggrDbSize.getShardId(), aggrDbSize.getLocalId());
            assertEquals("aggregation last", expected, points);
        }
    }

    private CoremonMetric makeMetric(String labelsStr) {
        var shardId = StockpileShardId.random();
        var localId = StockpileLocalId.random();
        var labels = LabelsFormat.parse(labelsStr);
        return makeMetric(shardId, localId, labels);
    }

    private static AggrRuleConf makeAggrRule(String condition, String target) {
        var rule = AggrRule.of(condition, target, null);
        var shard = new ShardKey("fake-project", "fake-cluster", "fake-service");
        return new AggrRuleConf(rule, shard);
    }

    private static AggrRuleConf makeAggrRule(String condition, String target, @Nullable MetricAggregation aggregation) {
        var rule = AggrRule.of(condition, target, aggregation);
        var shard = new ShardKey("fake-project", "fake-cluster", "fake-service");
        return new AggrRuleConf(rule, shard);
    }

    private static CoremonMetric makeMetric(int shardId, long localId, Labels labels) {
        return new FileCoremonMetric(shardId, localId, labels, MetricType.DGAUGE);
    }

    private Labels[] targetMetrics(AggrHelper helper, Labels metricLabels, Labels optLabels) {
        return helper.targetMetricsSet(metricLabels, optLabels).keySet().toArray(Labels[]::new);
    }

    private boolean processPoint(
            AggrHelper helper,
            CoremonShardStockpileResolveHelper resolver,
            CoremonShardStockpileWriteHelper writer,
            CoremonMetric metric,
            AggrPoint point)
    {
        return helper.aggregatePoint(resolver,
                writer,
                metric.getLabels(),
                point,
                metric.getType(),
                MetricTypeConverter.toNotNullProto(metric.getType()),
                metric,
                Labels.of());
    }
}
