package ru.yandex.solomon.coremon.aggregates;

import java.time.Instant;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPoints.point;


/**
 * @author Vladimir Gordiychuk
 */
public class AggrWeightAvgCollectorTest {

    @Test
    public void weightedAvgFromSingleTimeSeries() {
        final long now = Instant.parse("2000-01-02T03:04:05Z").toEpochMilli();
        final int windowSizeMillis = 15_000;

        { // 1 point: [ *, 10, * ] -> [ 10 ]
            TimeSeries timeSeries = TimeSeries.newDouble(now, 10);
            AggrGraphDataArrayList avgSeries = new AggrGraphDataArrayList();
            writeWeightedAvg(windowSizeMillis, avgSeries, timeSeries);

            avgSeries.foldDenomIntoOne();
            assertEquals(AggrGraphDataArrayList.of(
                    point("2000-01-02T03:04:00Z", 10.0, true, 1)
            ), avgSeries);
        }
        { // 2 points: [ *, 10, 20 ] -> [ 10 ]
            TimeSeries timeSeries = TimeSeries.empty()
                    .addDouble(now, 10.0) // window 1
                    .addDouble(now + 5_000, 20.0); // window 1

            AggrGraphDataArrayList avgSeries = new AggrGraphDataArrayList();
            writeWeightedAvg(windowSizeMillis, avgSeries, timeSeries);

            avgSeries.foldDenomIntoOne();
            assertEquals(AggrGraphDataArrayList.of(
                    point("2000-01-02T03:04:00Z", 10.0, true, 2)
            ), avgSeries);
        }
        { // 3 points: [ *, 10, 20 ], [ 30, *, * ] -> [ 10, 10 ]
            TimeSeries timeSeries = TimeSeries.empty()
                    .addDouble(now         , 10.0) // window 1
                    .addDouble(now +  5_000, 20.0) // window 1
                    .addDouble(now + 10_000, 30.0); // window 2

            AggrGraphDataArrayList avgSeries = new AggrGraphDataArrayList();
            writeWeightedAvg(windowSizeMillis, avgSeries, timeSeries);

            avgSeries.foldDenomIntoOne();
            assertEquals(AggrGraphDataArrayList.of(
                    point("2000-01-02T03:04:00Z", 10.0, true, 2),
                    point("2000-01-02T03:04:15Z", 10.0, true, 1)
            ), avgSeries);
        }
        { // 5 points: [ *, 10, 20 ], [ 30, 40, 50 ] -> [ 10, 40 ]
            TimeSeries timeSeries = TimeSeries.newDouble(5)
                    .addDouble(now, 10.0)
                    .addDouble(now +  5_000, 20.0)
                    .addDouble(now + 10_000, 30.0)
                    .addDouble(now + 15_000, 40.0)
                    .addDouble(now + 20_000, 50.0);

            AggrGraphDataArrayList avgSeries = new AggrGraphDataArrayList();
            writeWeightedAvg(windowSizeMillis, avgSeries, timeSeries);

            avgSeries.foldDenomIntoOne();
            assertEquals(AggrGraphDataArrayList.of(
                    point("2000-01-02T03:04:00Z", 10.0, true, 2),
                    point("2000-01-02T03:04:15Z", 40.0, true, 3)
            ), avgSeries);
        }
    }

    @Test
    public void weightedAvgFromMultipleTimeSeries() throws Exception {
        final long now = Instant.parse("2000-01-02T03:04:05Z").toEpochMilli();
        final int windowSizeMillis = 15_000;

        AggrGraphDataArrayList avgSeries = new AggrGraphDataArrayList();

        { // one window [ *, 10, 20 ]
            TimeSeries timeSeries = TimeSeries.newDouble(2)
                    .addDouble(now, 10)
                    .addDouble(now + 5_000, 20);
            writeWeightedAvg(windowSizeMillis, avgSeries, timeSeries);
        }
        { // two windows (full and incomplete) [ 30, 40, 50 ], [ 60, *, * ]
            TimeSeries timeSeries = TimeSeries.newDouble(4)
                    .addDouble(now + 10_000, 30.0)
                    .addDouble(now + 15_000, 40.0)
                    .addDouble(now + 20_000, 50.0)
                    .addDouble(now + 25_000, 60.0);
            writeWeightedAvg(windowSizeMillis, avgSeries, timeSeries);
        }
        { // one window (to complete previous one) [ *, 70, 80 ]
            TimeSeries timeSeries = TimeSeries.empty()
                    .addDouble(now + 30_000, 70.0)
                    .addDouble(now + 35_000, 80.0);
            writeWeightedAvg(windowSizeMillis, avgSeries, timeSeries);
        }

        avgSeries.foldDenomIntoOne();
        assertEquals(AggrGraphDataArrayList.of(
                point("2000-01-02T03:04:00Z", 10.0, true, 2),
                point("2000-01-02T03:04:15Z", 40.0, true, 3),
                point("2000-01-02T03:04:30Z", 70.0, true, 3)
        ), avgSeries);
    }

    @Test
    public void empty() {
        AggrWeightAvgCollector collector = new AggrWeightAvgCollector(15_000);
        assertFalse(collector.compute(new AggrPoint()));
    }

    @Test
    public void rememberFirstPoint() {
        AggrWeightAvgCollector collector = new AggrWeightAvgCollector(15_000);
        assertFalse(collector.append(point("2000-01-02T03:04:05Z", 42)));
    }

    @Test
    public void one() {
        final long now = Instant.parse("2000-01-02T03:04:05Z").toEpochMilli();
        AggrWeightAvgCollector collector = new AggrWeightAvgCollector(15_000);
        collector.append(point(now, 42.0));
        AggrPoint result = new AggrPoint();
        collector.compute(result);
        Assert.assertEquals(point("2000-01-02T03:04:00Z", 42.0, true, 1), result);
    }

    @Test
    public void fillWeightedSumAsOnlyWindowBorderChanged() {
        final long now = Instant.parse("2000-01-02T03:04:00Z").toEpochMilli();
        AggrWeightAvgCollector collector = new AggrWeightAvgCollector(15_000);

        AggrPoint point = new AggrPoint();
        {
            point.setTsMillis(now);
            point.setValue(20);
            assertFalse(collector.append(point));
        }
        {
            point.setTsMillis(now + 10_000);
            point.setValue(30);
            assertFalse(collector.append(point));
        }
        {
            point.setTsMillis(now + 15_000);
            point.setValue(40);
            assertTrue(collector.append(point));

            AggrPoint expectedPoint = new AggrPoint();
            expectedPoint.setTsMillis(now);
            expectedPoint.setValue(350, 15_000);
            expectedPoint.setMerge(true);
            expectedPoint.setCount(2);
            Assert.assertEquals(expectedPoint, point);
        }
        {
            AggrPoint expectedPoint = new AggrPoint();
            expectedPoint.setTsMillis(now + 15_000);
            expectedPoint.setValue(200, 15_000);
            expectedPoint.setMerge(true);
            expectedPoint.setCount(1);

            assertTrue(collector.compute(point));
            Assert.assertEquals(expectedPoint, point);
        }
    }

    private static void assertEquals(AggrGraphDataArrayList expected, AggrGraphDataArrayList actual) {
        actual.sortAndMerge();
        actual.foldDenomIntoOne();
        Assert.assertEquals(expected, actual);
    }

    private void writeWeightedAvg(int windowSizeMillis, AggrGraphDataArrayList avgSeries, TimeSeries timeSeries) {
        AggrWeightAvgCollector collector = new AggrWeightAvgCollector(windowSizeMillis);
        AggrPoint temp = new AggrPoint();
        for (int index = 0; index < timeSeries.size(); index++) {
            temp.setTsMillis(timeSeries.tsMillisAt(index));
            temp.setValue(timeSeries.doubleAt(index), ValueColumn.DEFAULT_DENOM);
            if (collector.append(temp)) {
                avgSeries.addRecord(temp);
            }
        }

        if (collector.compute(temp)) {
            avgSeries.addRecord(temp);
        }
    }

}
