package ru.yandex.solomon.coremon.aggregates;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Month;
import java.time.ZoneOffset;

import org.junit.Test;

import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPoints.dpoint;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricAggregatorTest {

    @Test
    public void onePointPerWindow() {
        var source = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                dpoint(ts("00:00:15"), 2),
                dpoint(ts("00:00:30"), 3),
                dpoint(ts("00:00:45"), 4),
                dpoint(ts("00:01:00"), 5),
                dpoint(ts("00:01:15"), 6));

        var expected = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                dpoint(ts("00:00:15"), 2),
                dpoint(ts("00:00:30"), 3),
                dpoint(ts("00:00:45"), 4),
                dpoint(ts("00:01:00"), 5),
                dpoint(ts("00:01:15"), 6));

        {
            var result = aggregate(source, 15_000, MetricAggregation.SUM);
            assertEquals(expected, result);
        }
        {
            var result = aggregate(source, 15_000, MetricAggregation.LAST);
            assertEquals(expected, result);
        }
    }

    @Test
    public void onePointPerWindowOneIntervalGap() {
        var source = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                dpoint(ts("00:00:15"), 2),
                //dpoint(ts("00:00:30"), 3), gap
                dpoint(ts("00:00:45"), 4),
                //dpoint(ts("00:01:00"), 5), gap
                dpoint(ts("00:01:15"), 6));

        var expected = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                dpoint(ts("00:00:15"), 2),
                //dpoint(ts("00:00:30"), 3), gap
                dpoint(ts("00:00:45"), 4),
                //dpoint(ts("00:01:00"), 5), gap
                dpoint(ts("00:01:15"), 6));

        {
            var result = aggregate(source, 15_000, MetricAggregation.SUM);
            assertEquals(expected, result);
        }
        {
            var result = aggregate(source, 15_000, MetricAggregation.LAST);
            assertEquals(expected, result);
        }
    }

    @Test
    public void onePointPerWindowTwoIntervalGaps() {
        var source = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                //dpoint(ts("00:00:15"), 2), gap
                //dpoint(ts("00:00:30"), 3), gap
                dpoint(ts("00:00:45"), 4),
                dpoint(ts("00:01:00"), 5),
                dpoint(ts("00:01:15"), 6));

        var expected = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                //dpoint(ts("00:00:15"), 2), gap
                //dpoint(ts("00:00:30"), 3), gap
                dpoint(ts("00:00:45"), 4),
                dpoint(ts("00:01:00"), 5),
                dpoint(ts("00:01:15"), 6));

        {
            var result = aggregate(source, 15_000, MetricAggregation.SUM);
            assertEquals(expected, result);
        }
        {
            var result = aggregate(source, 15_000, MetricAggregation.LAST);
            assertEquals(expected, result);
        }
    }

    @Test
    public void sumManyPointPerWindow() {
        var source = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1), //   \
                dpoint(ts("00:00:15"), 1), //   | fully filled
                dpoint(ts("00:00:30"), 1), //   | window
                dpoint(ts("00:00:45"), 2), //   /
                dpoint(ts("00:01:00"), 2), //   \  partially (half) filled
                dpoint(ts("00:01:15"), 2), //   |  window
                dpoint(ts("00:01:30"), 2));//   /

        var expected = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1.25),
                dpoint(ts("00:01:00"), 1.5));

        var result = aggregate(source, 60_000, MetricAggregation.SUM);
        assertEquals(expected, result);
    }

    @Test
    public void lastManyPointPerWindow() {
        var source = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1), //   \
                dpoint(ts("00:00:15"), 1), //   | fully filled
                dpoint(ts("00:00:30"), 2), //   | window
                dpoint(ts("00:00:45"), 3), //   /
                dpoint(ts("00:01:00"), 2), //   \  partially (half) filled
                dpoint(ts("00:01:15"), 2), //   |  window
                dpoint(ts("00:01:30"), 2));//   /

        var expected = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 3),
                dpoint(ts("00:01:00"), 2));

        var result = aggregate(source, 60_000, MetricAggregation.LAST);
        assertEquals(expected, result);
    }

    private AggrGraphDataArrayList aggregate(AggrGraphDataArrayList timeSeries, long gridMillis, MetricAggregation aggregation) {
        int from = timeSeries.length();
        MetricAggregator.aggregate(timeSeries, gridMillis, aggregation);
        var result = AggrGraphDataArrayList.of(timeSeries.slice(from, timeSeries.length()));
        timeSeries.truncate(from);
        result.foldDenomIntoOne();
        return result;
    }

    private static long ts(String time) {
        return LocalDateTime.of(LocalDate.of(2000, Month.JANUARY, 1), LocalTime.parse(time))
                .toInstant(ZoneOffset.UTC)
                .toEpochMilli();
    }
}
