package ru.yandex.solomon.coremon.aggregates;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Month;
import java.time.ZoneOffset;

import org.junit.Test;

import ru.yandex.solomon.coremon.aggregates.WeightedAvgSum.PointConsumer;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.DataPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.util.time.InstantUtils;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPoints.dpoint;


/**
 * @author Sergey Polovko
 */
public class WeightedAvgSumTest {

    @Test
    public void sumOverTwoMetrics() {
        long stepMillis = 60_000;

        AggrGraphDataArrayList list = new AggrGraphDataArrayList();
        PointConsumer consumer = (tsMillis, num, denom) -> {
            AggrPoint point = new AggrPoint();
            point.reset();
            point.setValue(num, denom);
            point.setMerge(true);
            point.setCount(1);

            tsMillis -= (tsMillis % stepMillis);
            point.setTsMillis(tsMillis);
            point.setStepMillis(stepMillis);

            list.addRecord(point);
        };

        var ts1 = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1), //   \
                dpoint(ts("00:00:15"), 1), //   | fully filled
                dpoint(ts("00:00:30"), 1), //   | window
                dpoint(ts("00:00:45"), 1), //   /
                dpoint(ts("00:01:00"), 1), //   \  partially (half) filled
                dpoint(ts("00:01:15"), 1));//   /  window


        var ts2 = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:05"), 2), //   \
                dpoint(ts("00:00:20"), 2), //   | fully filled
                dpoint(ts("00:00:35"), 2), //   | window
                dpoint(ts("00:00:50"), 2), //   /
                dpoint(ts("00:01:05"), 2), //   \  partially (half) filled
                dpoint(ts("00:01:20"), 2));//   /  window

        WeightedAvgSum.consume(ts1, stepMillis, consumer);
        WeightedAvgSum.consume(ts2, stepMillis, consumer);

        list.sortAndMerge();
        GraphData graphData = list.toGraphDataShort();

        assertEquals(
            GraphData.of(
                DataPoint.point(ts("00:00:00"), 1.0 + 2.0),
                DataPoint.point(ts("00:01:00"), 1.0 / 2 + 2.0 / 2)),
            graphData);
    }

    @Test
    public void onePointPerWindow() {
        var source = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                dpoint(ts("00:00:15"), 2),
                dpoint(ts("00:00:30"), 3),
                dpoint(ts("00:00:45"), 4),
                dpoint(ts("00:01:00"), 5),
                dpoint(ts("00:01:15"), 6));

        var expected = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                dpoint(ts("00:00:15"), 2),
                dpoint(ts("00:00:30"), 3),
                dpoint(ts("00:00:45"), 4),
                dpoint(ts("00:01:00"), 5),
                dpoint(ts("00:01:15"), 6));

        var result = calculate(source, 15_000);
        result.foldDenomIntoOne();

        assertEquals(expected, result);
    }

    @Test
    public void onePointPerWindowOneIntervalGap() {
        var source = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                dpoint(ts("00:00:15"), 2),
                //dpoint(ts("00:00:30"), 3), gap
                dpoint(ts("00:00:45"), 4),
                //dpoint(ts("00:01:00"), 5), gap
                dpoint(ts("00:01:15"), 6));

        var expected = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                dpoint(ts("00:00:15"), 2),
                //dpoint(ts("00:00:30"), 3), gap
                dpoint(ts("00:00:45"), 4),
                //dpoint(ts("00:01:00"), 5), gap
                dpoint(ts("00:01:15"), 6));

        var result = calculate(source, 15_000);
        result.foldDenomIntoOne();

        assertEquals(expected, result);
    }

    @Test
    public void onePointPerWindowTwoIntervalGaps() {
        var source = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                //dpoint(ts("00:00:15"), 2), gap
                //dpoint(ts("00:00:30"), 3), gap
                dpoint(ts("00:00:45"), 4),
                dpoint(ts("00:01:00"), 5),
                dpoint(ts("00:01:15"), 6));

        var expected = AggrGraphDataArrayList.of(
                dpoint(ts("00:00:00"), 1),
                //dpoint(ts("00:00:15"), 2), gap
                //dpoint(ts("00:00:30"), 3), gap
                dpoint(ts("00:00:45"), 4),
                dpoint(ts("00:01:00"), 5),
                dpoint(ts("00:01:15"), 6));

        var result = calculate(source, 15_000);
        result.foldDenomIntoOne();

        assertEquals(expected, result);
    }

    private AggrGraphDataArrayList calculate(AggrGraphDataArrayList timeSeries, long gridMillis) {
        AggrGraphDataArrayList list = new AggrGraphDataArrayList();
        PointConsumer consumer = (tsMillis, num, denom) -> {
            AggrPoint point = new AggrPoint();
            point.setValue(num, denom);
            point.setTsMillis(InstantUtils.truncate(tsMillis, gridMillis));
            list.addRecord(point);
        };

        WeightedAvgSum.consume(timeSeries, gridMillis, consumer);
        return list;
    }

    private static long ts(String time) {
        return LocalDateTime.of(LocalDate.of(2000, Month.JANUARY, 1), LocalTime.parse(time))
            .toInstant(ZoneOffset.UTC)
            .toEpochMilli();
    }
}
