package ru.yandex.solomon.coremon.balancer;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;

import com.google.common.primitives.Ints;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.coremon.balancer.db.BalancerShard;
import ru.yandex.solomon.coremon.balancer.db.memory.InMemoryBalancerShardsDao;
import ru.yandex.solomon.locks.ReadOnlyDistributedLockStub;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.util.host.HostUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toUnmodifiableSet;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.coremon.balancer.CoremonShardsHolder.RECONCILIATION_THRESHOLD;

/**
 * @author Stanislav Kashirin
 */
public class CoremonShardsHolderTest {

    private static final long OLD_TS = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(1);

    private InMemoryBalancerShardsDao dao;
    private ReadOnlyDistributedLockStub lock;
    private CoremonShardsHolder holder;

    @Before
    public void setUp() {
        dao = new InMemoryBalancerShardsDao();
        lock = new ReadOnlyDistributedLockStub(new ManualClock());
        lock.setOwner(HostUtils.getFqdn());

        holder = new CoremonShardsHolder(dao, lock);
    }

    @Test
    public void emptyShards() {
        var reloadFuture = holder.reload();
        assertEquals(unsignedOf(), holder.getShards());

        holder.onConfigurationLoad(confWithCtx());
        reloadFuture.join();
        assertEquals(unsignedOf(), holder.getShards());
    }

    @Test
    public void reloadShards() {
        dao.upsert(oldShard(42)).join();
        var reloadFuture = holder.reload();

        holder.onConfigurationLoad(confWithCtx(42));
        reloadFuture.join();
        assertEquals(unsignedOf(42), holder.getShards());

        dao.upsert(oldShard(-322)).join();
        holder.reload().join();
        assertEquals(unsignedOf(42), holder.getShards());

        holder.onConfigurationLoad(confWithCtx(42, -322));
        assertEquals(unsignedOf(42, -322), holder.getShards());
    }

    @Test
    public void getShardsWhenConfEmpty() {
        holder.onConfigurationLoad(confWithCtx());

        dao.upsert(oldShard(42)).join();
        dao.upsert(oldShard(-322)).join();

        holder.reload().join();
        assertEquals(unsignedOf(42, -322), holder.getShards());
    }

    @Test
    public void someShardsMissingInConf() {
        dao.upsert(oldShard(42)).join();
        dao.upsert(oldShard(-322)).join();
        dao.upsert(recentShard(777)).join();
        dao.upsert(oldShard(888)).join();
        dao.upsert(recentShard(999)).join();
        var reloadFuture = holder.reload();

        holder.onConfigurationLoad(confWithCtx(888, 999));
        reloadFuture.join();
        assertEquals(unsignedOf(777, 888, 999), holder.getShards());

        add(666).join();
        assertEquals(unsignedOf(666, 777, 888, 999), holder.getShards());

        holder.onConfigurationLoad(confWithCtx(999));
        assertEquals(unsignedOf(666, 777, 999), holder.getShards());
    }

    @Test
    public void readMultipleTimes() {
        dao.upsert(oldShard(42)).join();
        dao.upsert(oldShard(1337)).join();
        dao.upsert(oldShard(-322)).join();

        var reloadFuture = holder.reload();
        holder.onConfigurationLoad(confWithCtx(42, 1337, -322));
        reloadFuture.join();

        assertEquals(unsignedOf(42, 1337, -322), holder.getShards());
        assertEquals(unsignedOf(42, 1337, -322), holder.getShards());
        assertEquals(unsignedOf(42, 1337, -322), holder.getShards());
    }

    @Test
    public void addShardsFromScratch() {
        holder.onConfigurationLoad(confWithCtx());
        holder.reload().join();
        add(777).join();
        assertEquals(unsignedOf(777), holder.getShards());

        holder.onConfigurationLoad(confWithCtx(777));
        add(42).join();
        assertEquals(unsignedOf(777, 42), holder.getShards());

        holder.onConfigurationLoad(confWithCtx(777, 42, 888));
        assertEquals(unsignedOf(777, 42), holder.getShards());

        add(888).join();
        assertEquals(unsignedOf(777, 42, 888), holder.getShards());
    }

    @Test
    public void addShardsAfterReload() {
        dao.upsert(oldShard(42)).join();
        dao.upsert(oldShard(1337)).join();
        holder.onConfigurationLoad(confWithCtx(42, 1337));
        holder.reload().join();

        add(-322).join();
        assertEquals(unsignedOf(42, 1337, -322), holder.getShards());

        holder.onConfigurationLoad(confWithCtx(42));
        assertEquals(unsignedOf(42, -322), holder.getShards());

        add(888).join();
        assertEquals(unsignedOf(42, -322, 888), holder.getShards());
    }

    @Test
    public void addShardsBeforeReload() {
        dao.upsert(oldShard(42)).join();
        dao.upsert(oldShard(1337)).join();
        dao.upsert(oldShard(13)).join();

        add(777).join();
        var reloadFuture = holder.reload();
        holder.onConfigurationLoad(confWithCtx(42, 1337));
        reloadFuture.join();
        assertEquals(unsignedOf(42, 1337, 777), holder.getShards());

        holder.onConfigurationLoad(confWithCtx(42, 1337, 777));
        assertEquals(unsignedOf(42, 1337, 777), holder.getShards());

        add(888).join();
        assertEquals(unsignedOf(42, 1337, 777, 888), holder.getShards());
    }

    @Test
    public void addShardsAfterInit() {
        dao.upsert(recentShard(4)).join();
        dao.upsert(oldShard(777)).join();
        dao.upsert(oldShard(-1)).join();

        var conf = confWithCtx(-1, 1, 2, 3);
        holder.onConfigurationLoad(conf);
        holder.init(conf).join();

        holder.reload().join();
        assertEquals(unsignedOf(-1, 1, 2, 3, 4), holder.getShards());

        holder.onConfigurationLoad(confWithCtx(-1, 1, 2, 3, 4));
        assertEquals(unsignedOf(-1, 1, 2, 3, 4), holder.getShards());

        add(5).join();
        assertEquals(unsignedOf(-1, 1, 2, 3, 4, 5), holder.getShards());
    }

    @Test
    public void deleteShard() {
        dao.upsert(oldShard(888)).join();
        dao.upsert(oldShard(777)).join();
        holder.onConfigurationLoad(confWithCtx(888, 777));
        holder.reload().join();

        delete(888).join();
        assertEquals(unsignedOf(777), holder.getShards());
        assertEquals(unsignedOf(777), dao.allIds());

        delete(777).join();
        assertEquals(unsignedOf(), holder.getShards());
        assertEquals(unsignedOf(), dao.allIds());
    }

    @Test
    public void cannotDeleteRecent() {
        dao.upsert(oldShard(888)).join();
        holder.onConfigurationLoad(confWithCtx(888));
        holder.reload().join();

        add(777).join();
        delete(888).join();
        assertEquals(unsignedOf(777), holder.getShards());
        assertEquals(unsignedOf(777), dao.allIds());

        delete(777).join();
        assertEquals(unsignedOf(777), holder.getShards());
        assertEquals(unsignedOf(777), dao.allIds());
    }

    @Test
    public void deleteNotExist() {
        holder.onConfigurationLoad(confWithCtx());
        holder.reload().join();

        add(777).join();

        delete(888).join();
        assertEquals(unsignedOf(777), holder.getShards());
        assertEquals(unsignedOf(777), dao.allIds());
    }

    @Test
    public void deleteAdd() {
        dao.upsert(recentShard(888)).join();
        dao.upsert(oldShard(777)).join();
        dao.upsert(oldShard(42)).join();
        holder.onConfigurationLoad(confWithCtx(777));
        holder.reload().join();

        delete(42); // don't wait for completion
        add(42).join();
        assertEquals(unsignedOf(42, 777, 888), holder.getShards());
        assertEquals(unsignedOf(42, 777, 888), dao.allIds());
    }

    @Test
    public void deleteMultipleTimes() {
        dao.upsert(oldShard(42)).join();
        holder.onConfigurationLoad(confWithCtx(42));
        holder.reload().join();

        IntStream.range(1, 100)
            .parallel()
            .mapToObj(i -> delete(42))
            .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid))
            .join();
        assertEquals(unsignedOf(), holder.getShards());
        assertEquals(unsignedOf(), dao.allIds());
    }

    @Test
    public void unableDeleteWhenNotLeader() {
        dao.upsert(oldShard(42)).join();
        holder.onConfigurationLoad(confWithCtx(42));
        holder.reload().join();

        lock.setOwner(null);

        var exception = delete(42)
            .thenApply(unused -> null)
            .exceptionally(e -> e)
            .join();

        assertNotNull(exception);
        assertEquals(unsignedOf(42), dao.allIds());
    }

    @Test
    public void removeFromShardsWhenDeleteInFlight() {
        dao.upsert(oldShard(-1)).join();
        holder.onConfigurationLoad(confWithCtx(-1));
        holder.reload().join();

        assertEquals(unsignedOf(-1), holder.getShards());
        delete(-1); // don't wait for completion
        assertEquals(unsignedOf(), Set.copyOf(holder.getShards()));
    }

    @Test
    public void reconciliation() throws InterruptedException {
        holder.onConfigurationLoad(confWithCtx());
        dao.upsert(recentShard(888)).join();
        dao.upsert(oldShard(777)).join();
        dao.upsert(oldShard(666)).join();
        dao.upsert(oldShard(555)).join();
        dao.upsert(oldShard(444)).join();
        holder.reload().join();

        holder.add(999).join();

        final int expectedAmendments = 3; // 2x delete, 1x bulkUpsert
        var latch = new CountDownLatch(expectedAmendments);
        dao.afterSupplier = () -> {
            latch.countDown();
            return completedFuture(null);
        };

        var conf = confWithCtx(555, 444, 333, 222, 111);
        for (int i = 0; i < RECONCILIATION_THRESHOLD + 1; i++) {
            holder.onConfigurationLoad(conf);
            holder.getShards();
        }

        assertTrue(latch.await(5, TimeUnit.SECONDS));
        assertEquals(unsignedOf(111, 222, 333, 444, 555, 888, 999), holder.getShards());
        assertEquals(unsignedOf(111, 222, 333, 444, 555, 888, 999), dao.allIds());
    }

    private CompletableFuture<Void> add(int numId) {
        return holder.add(Integer.toUnsignedString(numId));
    }

    private CompletableFuture<Void> delete(int numId) {
        return holder.delete(Integer.toUnsignedString(numId));
    }

    private static BalancerShard oldShard(int numId) {
        return new BalancerShard(Integer.toUnsignedString(numId), OLD_TS);
    }

    private static BalancerShard recentShard(int numId) {
        return new BalancerShard(Integer.toUnsignedString(numId), System.currentTimeMillis());
    }

    private static Set<String> unsignedOf(int... numIds) {
        return Arrays.stream(numIds)
            .mapToObj(Integer::toUnsignedString)
            .collect(toUnmodifiableSet());
    }

    private static SolomonConfWithContext confWithCtx(int... numIds) {
        Collections.shuffle(Ints.asList(numIds));

        var projects = IntStream.of(numIds)
            .mapToObj(
                numId -> Project.newBuilder()
                    .setId("p_" + numId)
                    .setName("p_name_" + numId)
                    .setOwner("any")
                    .build())
            .collect(toList());

        var clusters = IntStream.of(numIds)
            .mapToObj(
                numId -> Cluster.newBuilder()
                    .setProjectId("p_" + numId)
                    .setId("cluster_id")
                    .setName("cluster_id")
                    .build())
            .collect(toList());

        var services = IntStream.of(numIds)
            .mapToObj(
                numId -> Service.newBuilder()
                    .setProjectId("p_" + numId)
                    .setId("service_id")
                    .setName("service_id")
                    .build())
            .collect(toList());

        var shards = IntStream.of(numIds)
            .mapToObj(
                numId -> Shard.newBuilder()
                    .setProjectId("p_" + numId)
                    .setId("shard_" + numId)
                    .setNumId(numId)
                    .setClusterId("cluster_id")
                    .setClusterName("cluster_id")
                    .setServiceId("service_id")
                    .setServiceName("service_id")
                    .build())
            .collect(toList());

        var conf = new SolomonRawConf(List.of(), projects, clusters, services, shards);
        return SolomonConfWithContext.create(conf);
    }

}
