package ru.yandex.solomon.coremon.balancer;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.IntSupplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.ImmutableSet;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntSet;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.coremon.balancer.Routines.AssignLostShards;
import ru.yandex.solomon.coremon.balancer.Routines.AssignNewShard;
import ru.yandex.solomon.coremon.balancer.Routines.MoveShards;
import ru.yandex.solomon.coremon.balancer.Routines.Rebalance;
import ru.yandex.solomon.coremon.balancer.cluster.CoremonHost;
import ru.yandex.solomon.coremon.balancer.db.MemShardAssignmentsDao;
import ru.yandex.solomon.coremon.balancer.db.ShardAssignments;
import ru.yandex.solomon.coremon.balancer.db.ShardAssignmentsDao;
import ru.yandex.solomon.coremon.balancer.db.ShardBalancerOptions;
import ru.yandex.solomon.coremon.balancer.state.ShardIds;
import ru.yandex.solomon.coremon.balancer.state.ShardLoad;
import ru.yandex.solomon.coremon.balancer.state.ShardsLoadMap;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Sergey Polovko
 */
public class RoutinesTest {

    private SolomonConfWithContext conf;
    private ShardAssignmentsDao assignmentsDao = new MemShardAssignmentsDao();

    @Before
    public void setUp() {
        this.conf = solomonConf(
            shard("a", 1, ShardState.ACTIVE),
            shard("b", 2, ShardState.ACTIVE),
            shard("c", 3, ShardState.ACTIVE),
            shard("d", 4, ShardState.ACTIVE),
            shard("e", 5, ShardState.INACTIVE),
            shard("f", 6, ShardState.INACTIVE));
    }

    @Test
    public void moveShards() {
        var host0 = new StubCoremonHost("host-man-00", ShardIds.ofWholeShards(1, 2));
        var host1 = new StubCoremonHost("host-man-01", ShardIds.ofWholeShards(3, 4));
        var host2 = new StubCoremonHost("host-man-02", ShardIds.EMPTY);
        var host3 = new StubCoremonHost("host-man-03", ShardIds.EMPTY);

        var r = new MoveShards(List.of(host2, host3), assignmentsDao);

        // (1) first host goes offline
        r.run(host0);
        assertEquals(ShardIds.ofWholeShards(1), host2.getState(false).getAssignments());
        assertEquals(ShardIds.ofWholeShards(2), host3.getState(false).getAssignments());
        assertEquals(ShardIds.EMPTY, host0.getState(false).getAssignments());

        assertEquals(ShardAssignments.copyOf(Map.of(
            1, "host-man-02",
            2, "host-man-03"
        )), join(assignmentsDao.load()));

        // (2) second host goes offline
        r.run(host1);
        assertEquals(ShardIds.ofWholeShards(1, 4), host2.getState(false).getAssignments());
        assertEquals(ShardIds.ofWholeShards(2, 3), host3.getState(false).getAssignments());
        assertEquals(ShardIds.EMPTY, host1.getState(false).getAssignments());

        assertEquals(ShardAssignments.copyOf(Map.of(
            1, "host-man-02",
            2, "host-man-03",
            3, "host-man-03",
            4, "host-man-02"
        )), join(assignmentsDao.load()));
    }

    @Test
    public void assignLostShards() {
        var host0 = new StubCoremonHost("host-man-00", ShardIds.ofWholeShards(1));
        var host1 = new StubCoremonHost("host-man-01", ShardIds.ofWholeShards(3));
        var host2 = new StubCoremonHost("host-man-02", ShardIds.EMPTY);
        var host3 = new StubCoremonHost("host-man-03", ShardIds.ofWholeShards(2));
        var host4 = new StubCoremonHost("host-man-04", ShardIds.ofWholeShards(4));
        var host5 = new StubCoremonHost("host-man-05", ShardIds.EMPTY);
        var host6 = new StubCoremonHost("host-man-06", ShardIds.EMPTY);

        var onlineHosts = List.<CoremonHost>of(host0, host1, host2, host3, host4, host5, host6);

        var currentAssignments = ShardAssignments.copyOf(Map.of(
            1, "host-man-00",
            2, "host-man-03",
            3, "host-man-01",
            4, "host-man-04"
        ));

        var r = new AssignLostShards(conf, onlineHosts, assignmentsDao, ShardBalancerOptions.DEFAULT);
        r.run(currentAssignments);

        ShardAssignments newAssignments = join(assignmentsDao.load());
        assertEquals(Set.of(5, 6), newAssignments.getIds());

        Set<String> bottomHosts = Set.of("host-man-02", "host-man-05", "host-man-06");

        assertTrue(bottomHosts.contains(newAssignments.get(5)));
        assertTrue(bottomHosts.contains(newAssignments.get(6)));
    }

    @Test
    public void rebalance() {
        var host0 = new StubCoremonHost("host-man-00", ShardIds.ofWholeShards(1, 2));
        var host1 = new StubCoremonHost("host-man-01", ShardIds.ofWholeShards(3, 4));
        var host2 = new StubCoremonHost("host-man-02", ShardIds.EMPTY);
        var host3 = new StubCoremonHost("host-man-03", ShardIds.EMPTY);
        var onlineHosts = List.<CoremonHost>of(host0, host1, host2, host3);

        var assignments = ShardAssignments.combine(List.of(
            ShardAssignments.ofShardIds(host0.getFqdn(), host0.getState(false).getAssignments().getShards()),
            ShardAssignments.ofShardIds(host1.getFqdn(), host1.getState(false).getAssignments().getShards())));
        join(assignmentsDao.save(assignments));
        assertEquals(ShardAssignments.copyOf(Map.of(
            1, "host-man-00",
            2, "host-man-00",
            3, "host-man-01",
            4, "host-man-01"
        )), assignments);

        var r = new Rebalance(onlineHosts, assignmentsDao, new ShardBalancerMetrics(new MetricRegistry()));

        r.run(assignments, ShardBalancerOptions.DEFAULT);
        assignments = join(assignmentsDao.load());
        assertEquals(ShardAssignments.copyOf(Map.of(
            1, "host-man-00",
            2, "host-man-00",
            3, "host-man-01",
            4, "host-man-02"
        )), assignments);

        r.run(assignments, ShardBalancerOptions.DEFAULT);
        assignments = join(assignmentsDao.load());
        assertEquals(ShardAssignments.copyOf(Map.of(
            1, "host-man-00",
            2, "host-man-03",
            3, "host-man-01",
            4, "host-man-02"
        )), assignments);
    }

    @Test
    public void assignNewShard() {
        var host0 = new StubCoremonHost("host-man-00", ShardIds.ofWholeShards(1, 2));
        var host1 = new StubCoremonHost("host-man-01", ShardIds.ofWholeShards(3, 4));
        var host2 = new StubCoremonHost("host-man-02", ShardIds.EMPTY);
        var host3 = new StubCoremonHost("host-man-03", ShardIds.ofWholeShards(5));
        var host4 = new StubCoremonHost("host-man-04", ShardIds.EMPTY);

        var onlineHosts = List.<CoremonHost>of(host0, host1, host2, host3, host4);

        var r = new AssignNewShard(assignmentsDao, onlineHosts, ShardBalancerOptions.DEFAULT);
        String host = join(r.run(42));

        Set<String> bottomHosts = Set.of("host-man-02", "host-man-03", "host-man-04");
        assertTrue(bottomHosts.contains(host));

        var assignedHost = onlineHosts.stream()
                .filter(coremonHost -> host.equals(coremonHost.getFqdn()))
                .findFirst()
                .get();

        ShardIds assignments = assignedHost.getState(true).getAssignments();
        assertTrue(assignments.containsShard(42));
    }

    private static Shard shard(String id, int numId, ShardState state) {
        return Shard.newBuilder()
            .setId(id)
            .setNumId(numId)
            .setProjectId("project-id")
            .setServiceId("service-id-" + id)
            .setServiceName("service-name-" + id)
            .setClusterId("cluster-id-" + id)
            .setClusterName("cluster-name-" + id)
            .setState(state)
            .build();
    }

    private static SolomonConfWithContext solomonConf(Shard... shards) {
        List<Project> projects = Arrays.stream(shards)
            .map(s -> Project.newBuilder()
                .setId(s.getProjectId())
                .setName(s.getProjectId())
                .setOwner("jamel")
                .build())
            .distinct()
            .collect(Collectors.toList());

        List<Service> services = Arrays.stream(shards)
            .map(s -> Service.newBuilder()
                .setId(s.getServiceId())
                .setName(s.getServiceName())
                .setProjectId(s.getProjectId())
                .build())
            .distinct()
            .collect(Collectors.toList());

        List<Cluster> clusters = Arrays.stream(shards)
            .map(s -> Cluster.newBuilder()
                .setId(s.getClusterId())
                .setName(s.getClusterName())
                .setProjectId(s.getProjectId())
                .build())
            .distinct()
            .collect(Collectors.toList());

        SolomonRawConf rawConf = new SolomonRawConf(List.of(), projects, clusters, services, Arrays.asList(shards));
        return SolomonConfWithContext.create(rawConf);
    }

    private static ImmutableSet<String> clusterHosts(String format) {
        return ImmutableSet.copyOf(IntStream.range(0, 10)
            .mapToObj(i -> String.format(format, i))
            .toArray(String[]::new));
    }

    /**
     * STUB COREMON HOST
     */
    private static final class StubCoremonHost implements CoremonHost {
        private final String fqdn;
        private ShardIds shardIds;
        private ShardsLoadMap shardsLoadMap;

        private StubCoremonHost(String fqdn, ShardIds shardIds) {
            this.fqdn = fqdn;
            this.shardIds = shardIds;
            this.shardsLoadMap = makeShardsLoadMap(shardIds);
        }

        private static ShardsLoadMap makeShardsLoadMap(ShardIds shardIds) {
            var map = new Int2ObjectOpenHashMap<ShardLoad>();
            for (var it = shardIds.getShards().iterator(); it.hasNext(); ) {
                var s = new ShardLoad(it.nextInt(), EShardState.READY, TimeUnit.MINUTES.toMillis(10), 20, 30, 40, 0);
                map.put(s.getId(), s);
            }
            return ShardsLoadMap.ownOf(map);
        }

        @Override
        public String getFqdn() {
            return fqdn;
        }

        @Override
        public long getSeenAliveTimeMillis() {
            return System.currentTimeMillis();
        }

        @Override
        public void startPinging(long leaderSeqNo, IntSupplier totalShardCount) {
        }

        @Override
        public void stopPinging() {
        }

        @Override
        public State getState(boolean refreshShardsStatus) {
            return new State() {
                @Override
                public ShardIds getAssignments() {
                    synchronized (StubCoremonHost.this) {
                        return shardIds;
                    }
                }

                @Override
                public ShardsLoadMap getShards() {
                    synchronized (StubCoremonHost.this) {
                        return shardsLoadMap;
                    }
                }

                @Override
                public boolean isSynced() {
                    synchronized (StubCoremonHost.this) {
                        return shardIds.getHash() == shardsLoadMap.getIdsHash() &&
                            shardIds.getShards().equals(shardsLoadMap.getIds());
                    }
                }

                @Override
                public long getUptimeMillis() {
                    return 10;
                }

                @Override
                public long getCpuTimeNanos() {
                    return 20;
                }

                @Override
                public long getMemoryBytes() {
                    return 30;
                }

                @Override
                public long getNetworkBytes() {
                    return 40;
                }
            };
        }

        @Override
        public CompletableFuture<Void> setAssignments(IntSet shardIds) {
            synchronized (this) {
                this.shardIds = ShardIds.ofWholeShards(shardIds);
                this.shardsLoadMap = makeShardsLoadMap(this.shardIds);
            }
            return completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> changeAssignments(IntSet shardIdsAdd, IntSet shardIdsRemove) {
            synchronized (this) {
                this.shardIds = shardIds.addRemoveShards(shardIdsAdd, shardIdsRemove);
                this.shardsLoadMap = makeShardsLoadMap(this.shardIds);
            }
            return completedFuture(null);
        }

        @Override
        public void close() {
        }
    }
}
