package ru.yandex.solomon.coremon.balancer.db;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Stream;

import org.junit.Test;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;
import static org.junit.Assert.assertEquals;

/**
 * @author Stanislav Kashirin
 */
public abstract class BalancerShardsDaoTest {
    protected abstract BalancerShardsDao getDao();

    @Test
    public void findEmpty() {
        var result = getDao().findAll().join();
        assertEquals(List.of(), result);
    }

    @Test
    public void upsertFind() {
        var dao = getDao();
        var shard = new BalancerShard("alice", System.currentTimeMillis());
        dao.upsert(shard).join();

        var result = dao.findAll().join();

        assertEquals(List.of(shard), result);
    }

    @Test
    public void repeatUpsert() {
        var dao = getDao();
        var shard = new BalancerShard("alice", System.currentTimeMillis());
        for (int i = 0; i < 3; i++) {
            dao.upsert(shard).join();
        }

        var result = dao.findAll().join();

        assertEquals(List.of(shard), result);
    }

    @Test
    public void upsertDifferentShard() {
        var dao = getDao();
        var alice = new BalancerShard("alice", System.currentTimeMillis());
        var bob = new BalancerShard("bob", System.currentTimeMillis() + 10);

        dao.upsert(alice).join();
        assertEquals(List.of(alice), dao.findAll().join());

        dao.upsert(bob).join();
        var result = new ArrayList<>(dao.findAll().join());
        result.sort(Comparator.comparing(BalancerShard::id));
        assertEquals(List.of(alice, bob), result);
    }

    @Test
    public void upsertDifferentTime() {
        var dao = getDao();
        var alice1 = new BalancerShard("alice", 1);
        var alice2 = new BalancerShard("alice", 2);

        dao.upsert(alice1).join();
        assertEquals(List.of(alice1), dao.findAll().join());

        dao.upsert(alice2).join();
        assertEquals(List.of(alice2), dao.findAll().join());
    }

    @Test
    public void delete() {
        var dao = getDao();
        var alice = new BalancerShard("alice", System.currentTimeMillis());
        var bob = new BalancerShard("bob", System.currentTimeMillis());
        dao.upsert(alice).join();
        dao.upsert(bob).join();

        dao.delete(alice.id()).join();
        assertEquals(List.of(bob), dao.findAll().join());

        dao.delete(bob.id()).join();
        assertEquals(List.of(), dao.findAll().join());
    }

    @Test
    public void bulkUpsertOfEmpty() {
        var dao = getDao();
        var alice = new BalancerShard("alice", System.currentTimeMillis());

        dao.bulkUpsert(List.of()).join();
        assertEquals(List.of(), dao.findAll().join());

        dao.upsert(alice).join();
        dao.bulkUpsert(List.of());
        assertEquals(List.of(alice), dao.findAll().join());
    }

    @Test
    public void bulkUpsertToEmpty() {
        var dao = getDao();
        var alice = new BalancerShard("alice", System.currentTimeMillis());
        var bob  = new BalancerShard("bob", System.currentTimeMillis());
        var charlie  = new BalancerShard("charlie", System.currentTimeMillis());

        dao.bulkUpsert(List.of(alice, bob, charlie)).join();

        var result = new ArrayList<>(dao.findAll().join());
        result.sort(Comparator.comparing(BalancerShard::id));
        assertEquals(List.of(alice, bob, charlie), result);
    }

    @Test
    public void bulkUpsertToNotEmpty() {
        var dao = getDao();
        var alice0 = new BalancerShard("alice", 42);

        var alice = new BalancerShard("alice", System.currentTimeMillis());
        var bob  = new BalancerShard("bob", System.currentTimeMillis());
        var charlie  = new BalancerShard("charlie", System.currentTimeMillis());

        dao.upsert(alice0).join();

        dao.bulkUpsert(List.of(alice, bob, charlie)).join();

        var result = new ArrayList<>(dao.findAll().join());
        result.sort(Comparator.comparing(BalancerShard::id));
        assertEquals(List.of(alice, bob, charlie), result);
    }

    @Test
    public void bulkUpsertMedium() {
        var dao = getDao();

        var shards = Stream.generate(
                () -> new BalancerShard(
                    randomAlphanumeric(16),
                    System.currentTimeMillis() + ThreadLocalRandom.current().nextInt(10_000)))
            .limit(3_333)
            .collect(toList());

        dao.bulkUpsert(shards).join();

        var result = new ArrayList<>(dao.findAll().join());
        shards.sort(Comparator.comparing(BalancerShard::id));
        result.sort(Comparator.comparing(BalancerShard::id));
        assertEquals(shards, result);
    }

    @Test
    public void bulkUpsertBig() {
        var dao = getDao();

        var shards = Stream.generate(
                () -> new BalancerShard(
                    randomAlphanumeric(16),
                    System.currentTimeMillis() + ThreadLocalRandom.current().nextInt(10_000)))
            .limit(22_222)
            .collect(toList());

        dao.bulkUpsert(shards).join();

        var result = new ArrayList<>(dao.findAll().join());
        shards.sort(Comparator.comparing(BalancerShard::id));
        result.sort(Comparator.comparing(BalancerShard::id));
        assertEquals(shards, result);
    }

}
