package ru.yandex.solomon.coremon.balancer.db;

import java.util.Map;
import java.util.Random;

import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * @author Sergey Polovko
 */
public abstract class ShardAssignmentsDaoTest {

    abstract ShardAssignmentsDao getDao();

    @Test
    public void emptyLoad() {
        ShardAssignments assignments = getDao().load().join();
        assertTrue(assignments.isEmpty());
    }

    @Test
    public void saveAndLoad() {
        var assignments = ShardAssignments.copyOf(Map.of(1, "host1", 2, "host2", 3, "host3"));
        getDao().save(assignments).join();

        var fromDb = getDao().load().join();
        assertEquals(assignments, fromDb);
    }

    @Test
    public void saveManyAssignments() {
        int count = 10 * YdbShardAssignmentsDao.BATCH_SIZE + 137;
        var rnd = new Random(17);

        var map = new Int2ObjectOpenHashMap<String>(count);
        for (int i = 0; i < count; i++) {
            map.put(rnd.nextInt(), RandomStringUtils.random(10));
        }

        var assignments = ShardAssignments.ownOf(map);
        getDao().save(assignments).join();
        ShardAssignments fromDb = getDao().load().join();

        assertEquals(assignments, fromDb);
    }

    @Test
    public void delete() {
        var assignments = ShardAssignments.copyOf(Map.of(1, "host1", 2, "host2", 3, "host3"));
        getDao().save(assignments).join();

        {
            getDao().delete(intSet(1)).join();
            var fromDb = getDao().load().join();
            assertEquals(ShardAssignments.copyOf(Map.of(2, "host2", 3, "host3")), fromDb);
        }

        {
            getDao().delete(intSet(2, 3)).join();
            var fromDb = getDao().load().join();
            assertEquals(ShardAssignments.EMPTY, fromDb);
        }
    }

    @Test
    public void deleteManyAssignments() {
        int count = 10 * YdbShardAssignmentsDao.BATCH_SIZE + 137;
        var rnd = new Random(17);

        int lastShardId = 0;
        var map = new Int2ObjectOpenHashMap<String>(count);
        for (int i = 0; i < count; i++) {
            lastShardId = rnd.nextInt();
            map.put(lastShardId, RandomStringUtils.random(10));
        }

        var assignments = ShardAssignments.ownOf(map);
        getDao().save(assignments).join();

        // delete all assignments except last one
        String lastHost = map.remove(lastShardId);
        getDao().delete(map.keySet()).join();

        ShardAssignments fromDb = getDao().load().join();
        assertEquals(ShardAssignments.copyOf(Map.of(lastShardId, lastHost)), fromDb);
    }

    @Test
    public void changeKnownAssignment() {
        var assignments = ShardAssignments.copyOf(Map.of(1, "host1", 2, "host2", 3, "host3"));
        getDao().save(assignments).join();

        getDao().update(1, "hostA").join();

        {
            ShardAssignments fromDb = getDao().load().join();
            assertEquals(3, fromDb.size());
            assertEquals("hostA", fromDb.get(1));
            assertEquals("host2", fromDb.get(2));
            assertEquals("host3", fromDb.get(3));
        }

        getDao().update(2, "hostB").join();

        {
            ShardAssignments fromDb = getDao().load().join();
            assertEquals(3, fromDb.size());
            assertEquals("hostA", fromDb.get(1));
            assertEquals("hostB", fromDb.get(2));
            assertEquals("host3", fromDb.get(3));
        }

        getDao().update(3, "hostC").join();

        {
            ShardAssignments fromDb = getDao().load().join();
            assertEquals(3, fromDb.size());
            assertEquals("hostA", fromDb.get(1));
            assertEquals("hostB", fromDb.get(2));
            assertEquals("hostC", fromDb.get(3));
        }
    }

    @Test
    public void changeUnknownAssignment() {
        var assignments = ShardAssignments.copyOf(Map.of(1, "host1", 2, "host2", 3, "host3"));
        getDao().save(assignments).join();

        getDao().update(10, "host10").join();

        ShardAssignments fromDb = getDao().load().join();
        assertEquals(assignments, fromDb);
    }

    private static IntSet intSet(int... values) {
        return new IntOpenHashSet(values);
    }
}
