package ru.yandex.solomon.coremon.balancer.db.memory;

import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.coremon.balancer.db.BalancerShard;
import ru.yandex.solomon.coremon.balancer.db.BalancerShardsDao;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.delayedExecutor;
import static java.util.stream.Collectors.toUnmodifiableSet;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public class InMemoryBalancerShardsDao implements BalancerShardsDao {
    private final ConcurrentHashMap<String, BalancerShard> shards = new ConcurrentHashMap<>();

    public volatile Supplier<CompletableFuture<?>> beforeSupplier;
    public volatile Supplier<CompletableFuture<?>> afterSupplier;

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return async(() -> null);
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return async(() -> {
            shards.clear();
            return null;
        });
    }

    @Override
    public CompletableFuture<Void> upsert(BalancerShard shard) {
        return async(() -> {
            shards.put(shard.id(), shard);
            return null;
        });
    }

    @Override
    public CompletableFuture<Void> delete(String shardId) {
        return async(() -> {
            shards.remove(shardId);
            return null;
        }, delayedExecutor(10, TimeUnit.MILLISECONDS));
    }

    @Override
    public CompletableFuture<List<BalancerShard>> findAll() {
        return async(() -> List.copyOf(shards.values()));
    }

    @Override
    public CompletableFuture<Void> bulkUpsert(List<BalancerShard> shards) {
        return async(() -> {
            shards.forEach(shard -> this.shards.put(shard.id(), shard));
            return null;
        });
    }

    public Set<String> allIds() {
        return shards.values().stream()
            .map(BalancerShard::id)
            .collect(toUnmodifiableSet());
    }

    private <T> CompletableFuture<T> async(Supplier<T> fn) {
        return async(fn, ForkJoinPool.commonPool());
    }

    private <T> CompletableFuture<T> async(Supplier<T> fn, Executor executor) {
        return before().thenComposeAsync(i -> {
            var result = fn.get();
            return after().thenApply(a -> result);
        }, executor);
    }

    private CompletableFuture<?> before() {
        var sup = beforeSupplier;
        if (sup == null) {
            return completedFuture(null);
        }

        return sup.get();
    }

    private CompletableFuture<?> after() {
        var sup = afterSupplier;
        if (sup == null) {
            return completedFuture(null);
        }

        return sup.get();
    }

}
