package ru.yandex.solomon.coremon.balancer.state;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.junit.Test;

import ru.yandex.monitoring.coremon.EShardState;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class LoadCalcTest {
    @Test
    public void stopMoveWhenLeastLoadedHostOverloaded() {
        ShardsLoadMap alice = loadMap(randomLoad(), randomLoad(), randomLoad());
        ShardsLoadMap bob = loadMap();
        ShardsLoadMap eva = loadMap();

        var cal = calc(new ShardsLoadMap[]{bob, alice, eva});
        var reassign = cal.getShardsToMove(10, 0);
        assertEquals(1, reassign.fromIdx());
        assertNotEquals(1, reassign.toIdx());
        assertEquals(1, reassign.toMove().size());
    }

    @Test
    public void avoidMoveOneOverloadedShard() {
        ShardsLoadMap alice = loadMap(emptyLoad(), emptyLoad(), cpu(emptyLoad(), 10000));
        ShardsLoadMap bob = loadMap(cpu(emptyLoad(), 1000), cpu(emptyLoad(), 500));
        ShardsLoadMap eva = loadMap(emptyLoad(), emptyLoad());

        var cal = calc(new ShardsLoadMap[]{alice, bob, eva});
        var reassign = cal.getShardsToMove(10, 0);
        assertEquals("from", 1, reassign.fromIdx());
        assertEquals("to", 2, reassign.toIdx());
        assertEquals(1, reassign.toMove().size());
    }

    @Test
    public void avoidMoveAllShardsFromMostLoadedHost() {
        ShardsLoadMap alice = loadMap(
                cpu(emptyLoad(), 100),
                cpu(emptyLoad(), 110)
        );
        ShardsLoadMap bob = loadMap(cpu(emptyLoad(), 200));
        ShardsLoadMap eva = loadMap(emptyLoad(), emptyLoad());

        var cal = calc(new ShardsLoadMap[]{alice, bob, eva});
        var reassign = cal.getShardsToMove(10, 0);
        assertEquals("from", 0, reassign.fromIdx());
        assertEquals("to", 2, reassign.toIdx());
        assertEquals(1, reassign.toMove().size());
    }

    @Test
    public void stopReassignAsOnlyTargetDiffReached() {
        ShardsLoadMap alice = loadMap(
                cpu(emptyLoad(), 900),
                cpu(emptyLoad(), 100),
                cpu(emptyLoad(), 100),
                cpu(emptyLoad(), 100),
                cpu(emptyLoad(), 100),
                cpu(emptyLoad(), 100),
                cpu(emptyLoad(), 100),
                cpu(emptyLoad(), 100),
                cpu(emptyLoad(), 100)
        );
        ShardsLoadMap bob = loadMap(cpu(emptyLoad(), 900));
        ShardsLoadMap eva = loadMap(cpu(emptyLoad(), 200));

        var cal = calc(new ShardsLoadMap[]{alice, bob, eva});
        var reassign = cal.getShardsToMove(10, 0.4);
        assertEquals("from", 0, reassign.fromIdx());
        assertEquals("to", 2, reassign.toIdx());
        assertEquals(2, reassign.toMove().size());

    }

    private ShardLoad randomLoad() {
        var random = ThreadLocalRandom.current();
        return new ShardLoad(
                random.nextInt(),
                EShardState.READY,
                TimeUnit.HOURS.toMillis(1) + random.nextLong(1, 100_000),
                random.nextLong(0, 1_000),
                random.nextLong(0, 1_000),
                random.nextLong(0, 1_000),
                random.nextLong(0, 1_000));
    }


    private ShardsLoadMap loadMap(ShardLoad... loads) {
        var shards = new Int2ObjectOpenHashMap<ShardLoad>();
        for (var load : loads) {
            shards.put(load.getId(), load);
        }
        return ShardsLoadMap.ownOf(shards);
    }

    private ShardLoad cpu(ShardLoad load, long count) {
        return new ShardLoad(load.getId(), load.getState(), load.getUptimeMillis(), count, load.getMetricsCount(), load.getNetworkBytes(), load.getParsedMetrics());
    }

    private ShardLoad emptyLoad() {
        var random = ThreadLocalRandom.current();
        return new ShardLoad(
                random.nextInt(),
                EShardState.READY,
                TimeUnit.HOURS.toMillis(1) + random.nextLong(1, 100_000),
                0,
                0,
                0,
                0);
    }

    private LoadCalc calc(ShardsLoadMap[] hostShardsLoad) {
        return new LoadCalc(hostShardsLoad, 1, 1, 1);
    }
}
