package ru.yandex.solomon.coremon.balancer.state;

import java.util.Collections;
import java.util.Set;

import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import org.junit.Test;

import ru.yandex.solomon.core.db.model.ShardPartitionId;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertTrue;

/**
 * @author Sergey Polovko
 */
public class ShardIdsTest {

    @Test
    public void empty() {
        ShardIds ids = ShardIds.EMPTY;
        assertTrue(ids.isEmpty());
        assertEquals(0L, ids.getHash());
        assertEquals(intSet(), ids.getShards());

        ShardIds newIds = ids.addRemoveShards(intSet(1, 2, 3), intSet(1, 2, 3));
        assertNotSame(newIds, ids);
        assertEquals(intSet(1, 2, 3), newIds.getShards());
    }

    @Test
    public void nonEmpty() {
        ShardIds ids = ShardIds.ofWholeShards(1, 2, 3);
        assertFalse(ids.isEmpty());
        assertNotEquals(0L, ids.getHash());
        assertEquals(intSet(1, 2, 3), ids.getShards());

        assertTrue(ids.containsShard(1));
        assertTrue(ids.containsShard(2));
        assertTrue(ids.containsShard(3));
        assertFalse(ids.containsShard(0));
    }

    @Test
    public void addRemove() {
        ShardIds ids = ShardIds.ofWholeShards(1, 2, 3);
        ShardIds newIds = ids.addRemoveShards(intSet(2, 3, 4, 5), intSet(1, 2, 3, 4));
        assertNotSame(newIds, ids);
        assertEquals(intSet(1, 2, 3), ids.getShards());
        assertEquals(intSet(2, 3, 4, 5), newIds.getShards());
    }

    @Test
    public void containsPartition() {
        var ids = ShardIds.ofPartitions(ShardPartitionId.of(10, 1, 1),
                ShardPartitionId.of(10, 5, 2),
                ShardPartitionId.of(21, 5, 2));

        assertTrue(ids.containsPartition(ShardPartitionId.of(10, 1, 1)));
        assertTrue(ids.containsPartition(ShardPartitionId.of(21, 5, 2)));
        assertFalse(ids.containsPartition(ShardPartitionId.of(21, 1, 1)));
    }

    @Test
    public void addRemovePartitions_onlyRemove() {
        var ids = ShardIds.ofPartitions(ShardPartitionId.of(10, 1, 1),
                ShardPartitionId.of(10, 5, 1),
                ShardPartitionId.of(21, 5, 2));

        var ids2 = ids.addRemovePartitions(
                Collections.emptySet(),
                Set.of(
                    ShardPartitionId.of(21, 5, 2),
                    ShardPartitionId.of(10, 1, 10)
                )
        );
        assertTrue(ids.containsPartition(ShardPartitionId.of(21, 5, 2))
                && !ids2.containsShard(21));
        assertFalse(ids2.containsPartition(ShardPartitionId.of(10, 1, 10)));
    }

    @Test
    public void addRemovePartitions_onlyAdd() {
        var ids = ShardIds.ofPartitions(
                ShardPartitionId.of(10, 1, 5),
                ShardPartitionId.of(10, 5, 7),
                ShardPartitionId.of(21, 5, 1));

        var ids2 = ids.addRemovePartitions(
                Set.of(
                    ShardPartitionId.of(55, 18, 1),
                    ShardPartitionId.of(55, 20, 1),
                    ShardPartitionId.of(10, 3, 1)
                ),
                Collections.emptySet()
        );
        assertFalse(ids2.containsPartition(ShardPartitionId.of(55,  100, 8)));
        assertFalse(ids.containsPartition(ShardPartitionId.of(10,  3, 9)));
        assertTrue(ids2.containsPartition(ShardPartitionId.of(10,  3, 1)));
    }

    @Test
    public void addRemovePartitions() {
        var ids = ShardIds.ofPartitions(
                ShardPartitionId.of(10, 1, 1),
                ShardPartitionId.of(10, 2, 1),
                ShardPartitionId.of(10, 3, 1));

        var ids2 = ids.addRemovePartitions(
                Set.of(ShardPartitionId.of(55, 4, 1), ShardPartitionId.of(55, 8, 1)),
                Set.of(
                    ShardPartitionId.of(10, 1, 1),
                    ShardPartitionId.of(10, 2, 1),
                    ShardPartitionId.of(10, 3, 1)
                )
        );

        assertTrue(ids.containsPartition(ShardPartitionId.of(10, 1, 1))
                && !ids2.containsPartition(ShardPartitionId.of(10, 1, 1)));
        assertFalse(ids.containsShard(55));
        assertTrue(ids2.containsPartition(ShardPartitionId.of(55,  4, 1)));
        assertTrue(ids2.containsPartition(ShardPartitionId.of(55,  8, 1)));
    }

    private static IntSet intSet(int... ids) {
        return new IntOpenHashSet(ids);
    }
}
