package ru.yandex.solomon.coremon.balancer.state;

import java.util.Map;
import java.util.Random;

import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import org.junit.Test;

import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.monitoring.coremon.TShardsLoad;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;

/**
 * @author Sergey Polovko
 */
public class ShardsLoadMapTest {

    private final ShardLoad shard1 = new ShardLoad(1, EShardState.NEW, 10, 20, 30, 40, 0);
    private final ShardLoad shard2 = new ShardLoad(2, EShardState.LOADING, 11, 21, 31, 41, 0);
    private final ShardLoad shard3 = new ShardLoad(3, EShardState.INDEXING, 12, 22, 32, 42, 0);
    private final ShardLoad shard4 = new ShardLoad(4, EShardState.READY, 13, 23, 33, 43, 0);
    private final ShardsLoadMap shards = ShardsLoadMap.copyOf(Map.of(1, shard1, 2, shard2, 3, shard3, 4, shard4));

    @Test
    public void empty() {
        ShardsLoadMap shards = ShardsLoadMap.EMPTY;
        assertEquals(0L, shards.getIdsHash());
        assertTrue(shards.getIds().isEmpty());
        assertNull(shards.get(1));
    }

    @Test
    public void nonEmpty() {
        assertNotEquals(0L, shards.getIdsHash());
        assertEquals(intSet(1, 2, 3, 4), shards.getIds());
        assertNull(shards.get(0));
        assertEquals(shard1, shards.get(1));
        assertEquals(shard2, shards.get(2));
        assertEquals(shard3, shards.get(3));
        assertEquals(shard4, shards.get(4));
        assertNull(shards.get(5));
    }

    @Test
    public void toAndFromPb() {
        // empty
        {
            TShardsLoad pb = ShardsLoadMap.EMPTY.toPb();
            assertEquals(0, pb.getShardIdsCount());
            assertEquals(0, pb.getStatesCount());
            assertEquals(0, pb.getUptimeMillisCount());
            assertEquals(0, pb.getCpuTimeNanosCount());
            assertEquals(0, pb.getMetricsCountCount());
            assertEquals(0, pb.getNetworkBytesCount());

            ShardsLoadMap newShards = ShardsLoadMap.fromPb(pb);
            assertSame(ShardsLoadMap.EMPTY, newShards);
        }

        // non empty
        {
            TShardsLoad pb = shards.toPb();
            assertEquals(4, pb.getShardIdsCount());
            assertEquals(4, pb.getStatesCount());
            assertEquals(4, pb.getUptimeMillisCount());
            assertEquals(4, pb.getCpuTimeNanosCount());
            assertEquals(4, pb.getMetricsCountCount());
            assertEquals(4, pb.getNetworkBytesCount());

            ShardsLoadMap newShards = ShardsLoadMap.fromPb(pb);
            assertEquals(shards, newShards);
        }
    }

    @Test
    public void retainAll() {
        {
            var newShards = shards.retainAll(intSet());
            assertSame(ShardsLoadMap.EMPTY, newShards);
            assertEquals(intSet(1, 2, 3, 4), shards.getIds());
        }
        {
            var newShards = shards.retainAll(intSet(1));
            assertEquals(intSet(1), newShards.getIds());
            assertEquals(intSet(1, 2, 3, 4), shards.getIds());
        }
        {
            var newShards = shards.retainAll(intSet(2));
            assertEquals(intSet(2), newShards.getIds());
            assertEquals(intSet(1, 2, 3, 4), shards.getIds());
        }
        {
            var newShards = shards.retainAll(intSet(2, 3));
            assertEquals(intSet(2, 3), newShards.getIds());
            assertEquals(intSet(1, 2, 3, 4), shards.getIds());
        }
        {
            var newShards = shards.retainAll(intSet(0, 1, 3, 5));
            assertEquals(intSet(1, 3), newShards.getIds());
            assertEquals(intSet(1, 2, 3, 4), shards.getIds());
        }
    }

    @Test
    public void pbSize() {
        Random rnd = new Random(17);
        EShardState[] states = EShardState.values();

        for (int count = 1; count <= 1_000_000; count *= 10) {
            var map = new Int2ObjectOpenHashMap<ShardLoad>(count);
            for (int shardId = 0; shardId < count; shardId++) {
                map.put(shardId, new ShardLoad(
                    shardId,
                    states[rnd.nextInt(states.length - 1)], // skip unrecognized enum value
                    rnd.nextLong(),
                    rnd.nextLong(),
                    rnd.nextLong(),
                    rnd.nextLong(),
                    rnd.nextLong()));
            }

            int bytesSize = ShardsLoadMap.ownOf(map).toPb().getSerializedSize();
            System.out.printf("count: %d, bytesSize: %d\n", count, bytesSize);
            assertTrue(bytesSize <= 37 * count + 6 * 5); // 6 fields (1 marker byte + up to 4 bytes for size)
        }
    }

    private static IntSet intSet(int... ids) {
        return new IntOpenHashSet(ids);
    }
}
