package ru.yandex.solomon.coremon.shards;

import org.junit.Test;

import ru.yandex.solomon.config.protobuf.coremon.TCoremonCreateShardConfig;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.core.db.model.ServiceProviderShardSettings;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.labels.shard.ShardKey;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.coremon.shards.ConfigMaker.makeShortId;

/**
 * @author Sergey Polovko
 */
public class ConfigMakerTest {

    @Test
    public void ids() {
        var conf = TCoremonCreateShardConfig.getDefaultInstance();
        var confMaker = new ConfigMaker(new ShardKey("solomon", "production", "stockpile"), "jamel", conf);
        assertEquals("solomon_production", confMaker.getClusterId());
        assertEquals("solomon_stockpile", confMaker.getServiceId());
        assertEquals("solomon_production_stockpile", confMaker.getShardId());
    }

    @Test
    public void makeProject() {
        var conf = TCoremonCreateShardConfig.getDefaultInstance();
        var confMaker = new ConfigMaker(new ShardKey("solomon", "production", "stockpile"), "jamel", conf);

        Project project = confMaker.makeProject("abc");
        assertEquals("solomon", project.getId());
        assertEquals("solomon", project.getName());
        assertEquals("abc", project.getAbcService());
        assertTrue(project.isOnlyAuthRead());
        assertTrue(project.isOnlyAuthPush());
        assertEquals("jamel", project.getOwner());
        assertEquals("jamel", project.getCreatedBy());
        assertEquals("jamel", project.getUpdatedBy());
    }

    @Test
    public void makeCluster() {
        var conf = TCoremonCreateShardConfig.getDefaultInstance();
        var confMaker = new ConfigMaker(new ShardKey("solomon", "production", "stockpile"), "jamel", conf);

        Cluster cluster = confMaker.makeCluster("folder");
        assertEquals("solomon", cluster.getProjectId());
        assertEquals("solomon_production", cluster.getId());
        assertEquals("production", cluster.getName());
        assertEquals("jamel", cluster.getCreatedBy());
        assertEquals("jamel", cluster.getUpdatedBy());
        assertEquals("folder", cluster.getFolderId());
    }

    @Test
    public void makeService() {
        var conf = TCoremonCreateShardConfig.getDefaultInstance();
        var confMaker = new ConfigMaker(new ShardKey("solomon", "production", "stockpile"), "jamel", conf);

        Service service = confMaker.makeService("", ServiceProviderShardSettings.EMPTY);
        assertEquals("solomon", service.getProjectId());
        assertEquals("solomon_stockpile", service.getId());
        assertEquals("stockpile", service.getName());
        assertEquals("jamel", service.getCreatedBy());
        assertEquals("jamel", service.getUpdatedBy());
    }

    @Test
    public void makeServiceProviderService() {
        var conf = TCoremonCreateShardConfig.getDefaultInstance();
        var confMaker = new ConfigMaker(new ShardKey("solomon", "production", "stockpile"), "jamel", conf);

        Service service = confMaker.makeService("testSp", new ServiceProviderShardSettings(null, 30, 15, 0));
        assertEquals("solomon", service.getProjectId());
        assertEquals("solomon_stockpile", service.getId());
        assertEquals("stockpile", service.getName());
        assertEquals("testSp", service.getServiceProvider());
        assertEquals(15, service.getShardSettings().getGrid());
        assertEquals(30, service.getShardSettings().getMetricsTtl());
        assertEquals("jamel", service.getCreatedBy());
        assertEquals("jamel", service.getUpdatedBy());
    }

    @Test
    public void makeService_emptySettings() {
        var conf = TCoremonCreateShardConfig.getDefaultInstance();
        var confMaker = new ConfigMaker(new ShardKey("solomon", "production", "stockpile"), "jamel", conf);

        Service service = confMaker.makeService("", ServiceProviderShardSettings.EMPTY);
        assertEquals(ServiceMetricConf.empty(), service.getMetricConf());
        assertEquals(0, service.getShardSettings().getGrid());
        assertEquals(0, service.getShardSettings().getMetricsTtl());
    }

    @Test
    public void makeService_nonEmptySettings() {
        var conf = TCoremonCreateShardConfig.getDefaultInstance();
        var confMaker = new ConfigMaker(new ShardKey("solomon", "production", "stockpile"), "jamel", conf);

        var metricConf = ServiceMetricConf.of(new ServiceMetricConf.AggrRule[] {
                AggrRule.of("host=*", "host=cluster", null)
        }, true);

        Service service = confMaker.makeService("", new ServiceProviderShardSettings(metricConf, 30, 15, 0));
        assertEquals(metricConf, service.getMetricConf());
        assertEquals(15, service.getShardSettings().getGrid());
        assertEquals(30, service.getShardSettings().getMetricsTtl());
    }

    @Test
    public void makeShard() {
        var conf = TCoremonCreateShardConfig.newBuilder()
                .setMetricNameLabel("name")
                .setTtlDays(77)
                .setMaxFileMetrics(1000)
                .setMaxMemOnlyMetrics(2000)
                .build();
        var confMaker = new ConfigMaker(new ShardKey("solomon", "production", "stockpile"), "jamel", conf);

        Cluster cluster = confMaker.makeCluster("");
        Service service = confMaker.makeService("", ServiceProviderShardSettings.EMPTY);

        Shard shard = confMaker.makeShard(cluster, service, 1337);
        assertEquals("solomon", shard.getProjectId());
        assertEquals("solomon_production_stockpile", shard.getId());
        assertEquals(1337, shard.getNumId());
        assertEquals(cluster.getId(), shard.getClusterId());
        assertEquals(cluster.getName(), shard.getClusterName());
        assertEquals(service.getId(), shard.getServiceId());
        assertEquals(service.getName(), shard.getServiceName());
        assertEquals(1000, shard.getMaxFileMetrics());
        assertEquals(2000, shard.getMaxMemMetrics());
        assertEquals(77, shard.getShardSettings().getMetricsTtl());
    }

    @Test
    public void shortIdTest() {
        assertIdSize(makeShortId("yc-mk8s-masters_cloud_preprod_l7-nodes-a5d9q6igrj5vfu93511t", 55), 55);
        assertIdSize(makeShortId("yc-mk8s-masters", 55), 15);
        assertEquals("yc-mk8s-masters_cloud_preprod_l7-nodes-a5d9q6ig_d43d83d",
                makeShortId("yc-mk8s-masters_cloud_preprod_l7-nodes-a5d9q6igrj5vfu93511t", 55));
    }

    private static void assertIdSize(String id, int expected) {
        assertEquals(id, expected, id.length());
    }
}
