package ru.yandex.solomon.coremon.shards;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.concurrent.CountDownLatches;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolverStub;
import ru.yandex.solomon.config.protobuf.coremon.TCoremonCreateShardConfig;
import ru.yandex.solomon.core.conf.ShardNumIdGenerator;
import ru.yandex.solomon.core.conf.ShardNumIdGeneratorInMemory;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.db.SchemaAwareDao;
import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbClustersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServicesDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.ServiceProviderShardSettings;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.kikimr.LocalKikimr;
import ru.yandex.solomon.kikimr.YdbHelper;
import ru.yandex.solomon.labels.shard.ShardKey;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.core.db.dao.ConfigDaoContext.createObjectMapper;

/**
 * @author Sergey Polovko
 */
@YaExternal
public class ShardCreatorTest {

    private static final String FOLDER_ID = "folder_" +  randomAlphanumeric(10);
    private static final String CLOUD_ID = "cloud_" + randomAlphanumeric(10);

    @ClassRule
    public static final LocalKikimr localKikimr = new LocalKikimr();

    @Rule
    public TestName name = new TestName();

    private ExecutorService executor;
    private ShardNumIdGenerator numIdGenerator;
    private YdbHelper ydb;
    private ShardsDao shardsDao;
    private ClustersDao clustersDao;
    private ServicesDao servicesDao;
    private ProjectsDao projectsDao;
    private FolderResolverStub folderResolver;

    @Before
    public void setUp() {
        executor = Executors.newSingleThreadExecutor();
        numIdGenerator = new ShardNumIdGeneratorInMemory();
        ydb = new YdbHelper(localKikimr, ShardCreatorTest.class.getSimpleName() + '_' + name.getMethodName());

        shardsDao = new YdbShardsDao(ydb.getTableClient(), ydb.resolvePath("Shards"), createObjectMapper(), ForkJoinPool.commonPool());
        clustersDao = new YdbClustersDao(ydb.getTableClient(), ydb.resolvePath("Clusters"), createObjectMapper(), ForkJoinPool.commonPool());
        servicesDao = new YdbServicesDao(ydb.getTableClient(), ydb.resolvePath("Services"), createObjectMapper(), ForkJoinPool.commonPool());
        projectsDao = new YdbProjectsDao(ydb.getTableClient(), ydb.resolvePath("Projects"), createObjectMapper(), ForkJoinPool.commonPool());

        var futures = Stream.of(shardsDao, clustersDao, servicesDao, projectsDao)
                .map(SchemaAwareDao::createSchemaForTests)
                .toArray(CompletableFuture[]::new);
        CompletableFuture.allOf(futures).join();

        var project = Project.newBuilder()
                .setId("solomon")
                .setName("solomon")
                .setOwner("jamel")
                .build();
        projectsDao.insert(project).join();
        var cloud = Project.newBuilder()
                .setId(CLOUD_ID)
                .setName(CLOUD_ID)
                .setOwner("hpple")
                .build();
        projectsDao.insert(cloud).join();

        folderResolver = new FolderResolverStub();
        folderResolver.add(FOLDER_ID, CLOUD_ID);
    }

    @After
    public void tearDown() throws Exception {
        join(shardsDao.dropSchemaForTests());
        ydb.close();
        executor.shutdown();
        assertTrue(executor.awaitTermination(5, TimeUnit.SECONDS));
    }

    @Test
    public void createClusterAndService() {
        ShardCreator creator = makeCreator(TCoremonCreateShardConfig.getDefaultInstance());
        Shard shard = creator.createShard(new ShardKey("solomon", "production", "stockpile"), "jamel").join();

        var cluster = clustersDao.findOne("solomon", "", shard.getClusterId()).join();
        assertEquals("", cluster.orElseThrow().getFolderId());

        assertEquals("solomon_production_stockpile", shard.getId());
        assertNotEquals(0, shard.getNumId());
        assertEquals("solomon", shard.getProjectId());
        assertEquals("solomon_production", shard.getClusterId());
        assertEquals("production", shard.getClusterName());
        assertEquals("solomon_stockpile", shard.getServiceId());
        assertEquals("stockpile", shard.getServiceName());
        assertEquals(Shard.DEFAULT_FILE_METRIC_QUOTA, shard.getMaxFileMetrics());
        assertEquals(Shard.DEFAULT_MEM_METRIC_QUOTA, shard.getMaxMemMetrics());
        assertEquals(Shard.DEFAULT_PER_URL_QUOTA, shard.getMaxMetricsPerUrl());
        assertEquals(Shard.DEFAULT_RESPONSE_SIZE_QUOTA, shard.getMaxResponseSizeBytes());
        assertEquals(0, shard.getShardSettings().getMetricsTtl());
        assertEquals("", shard.getMetricNameLabel());
        assertEquals("jamel", shard.getCreatedBy());
        assertEquals("jamel", shard.getUpdatedBy());
    }

    @Test
    public void createClusterWithFolder() {
        var creator = makeCreator(TCoremonCreateShardConfig.getDefaultInstance());

        var shard = creator.createShard(new ShardKey(CLOUD_ID, FOLDER_ID, "stockpile"), "hpple").join();

        var cluster = clustersDao.findOne(CLOUD_ID, "", shard.getClusterId()).join();
        assertEquals(FOLDER_ID, cluster.orElseThrow().getFolderId());
    }

    @Test
    public void useClusterAndService() {
        Cluster cluster = Cluster.newBuilder()
                .setProjectId("solomon")
                .setId("solomon_production")
                .setName("production")
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.EMPTY,
                        11))
                .build();
        assertTrue(clustersDao.insert(cluster).join());

        Service service = Service.newBuilder()
                .setProjectId("solomon")
                .setId("solomon_stockpile")
                .setName("stockpile")
                .setShardSettings(ShardSettings.of(ShardSettings.Type.PUSH,
                        null,
                        15,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, ServiceMetricConf.empty().getAggrRules(), false),
                        15))
                .build();
        assertTrue(servicesDao.insert(service).join());

        ShardCreator creator = makeCreator(TCoremonCreateShardConfig.getDefaultInstance());
        Shard shard = creator.createShard(new ShardKey("solomon", "production", "stockpile"), "jamel").join();

        assertEquals("solomon_production_stockpile", shard.getId());
        assertNotEquals(0, shard.getNumId());
        assertEquals("solomon", shard.getProjectId());
        assertEquals("solomon_production", shard.getClusterId());
        assertEquals("production", shard.getClusterName());
        assertEquals("solomon_stockpile", shard.getServiceId());
        assertEquals("stockpile", shard.getServiceName());
        assertEquals(Shard.DEFAULT_FILE_METRIC_QUOTA, shard.getMaxFileMetrics());
        assertEquals(Shard.DEFAULT_MEM_METRIC_QUOTA, shard.getMaxMemMetrics());
        assertEquals(Shard.DEFAULT_PER_URL_QUOTA, shard.getMaxMetricsPerUrl());
        assertEquals(Shard.DEFAULT_RESPONSE_SIZE_QUOTA, shard.getMaxResponseSizeBytes());
        assertEquals(0, shard.getShardSettings().getMetricsTtl());
        assertEquals("", shard.getMetricNameLabel());
        assertEquals("jamel", shard.getCreatedBy());
        assertEquals("jamel", shard.getUpdatedBy());

        List<Cluster> clusters = clustersDao.findAll().join();
        assertEquals(List.of(cluster), clusters);

        List<Service> services = servicesDao.findAll().join();
        assertEquals(List.of(service), services);
    }

    @Test
    public void settingsFromServiceProvider() {
        ServiceMetricConf metricConf = ServiceMetricConf.of(new AggrRule[] {
                AggrRule.of("host=*", "host=cluster", null)
        }, true);

        ServiceProvider serviceProvider = ServiceProvider.newBuilder()
                .setId("compute")
                .setAbcService("yandex-cloud")
                .setShardSettings(new ServiceProviderShardSettings(metricConf, 30, 15, 0))
                .build();

        ShardCreator creator = makeCreator(TCoremonCreateShardConfig.getDefaultInstance());
        creator.onConfigurationLoad(SolomonConfWithContext.create(new SolomonRawConf(
                List.of(serviceProvider), List.of(), List.of(), List.of(), List.of()
        )));

        Shard shard = creator.createShard(new ShardKey("solomon", "production", "compute"), "robot-solomon").join();
        assertEquals("solomon_production_compute", shard.getId());
        assertNotEquals(0, shard.getNumId());
        assertEquals("solomon", shard.getProjectId());
        assertEquals("solomon_production", shard.getClusterId());
        assertEquals("production", shard.getClusterName());
        assertEquals("solomon_compute", shard.getServiceId());
        assertEquals("compute", shard.getServiceName());
        assertEquals(Shard.DEFAULT_FILE_METRIC_QUOTA, shard.getMaxFileMetrics());
        assertEquals(Shard.DEFAULT_MEM_METRIC_QUOTA, shard.getMaxMemMetrics());
        assertEquals(Shard.DEFAULT_PER_URL_QUOTA, shard.getMaxMetricsPerUrl());
        assertEquals(Shard.DEFAULT_RESPONSE_SIZE_QUOTA, shard.getMaxResponseSizeBytes());
        assertEquals(0, shard.getShardSettings().getMetricsTtl());
        assertEquals("", shard.getMetricNameLabel());
        assertEquals("robot-solomon", shard.getCreatedBy());
        assertEquals("robot-solomon", shard.getUpdatedBy());

        List<Service> services = servicesDao.findAll().join();
        assertEquals(1, services.size());

        Service service = services.get(0);
        assertEquals("solomon_compute", service.getId());
        assertEquals("compute", service.getName());
        assertEquals(30, service.getShardSettings().getMetricsTtl());
        assertEquals(15, service.getShardSettings().getGrid());
        assertEquals(metricConf, service.getMetricConf());
    }

    @Test
    public void settingsFromConfig() {
        ShardCreator creator = makeCreator(TCoremonCreateShardConfig.newBuilder()
                .setTtlDays(30)
                .setMaxFileMetrics(10_000)
                .setMaxMemOnlyMetrics(100_000)
                .setMetricNameLabel("name")
                .build());

        Shard shard = creator.createShard(new ShardKey("solomon", "production", "compute"), "robot-solomon").join();
        assertEquals("solomon_production_compute", shard.getId());
        assertNotEquals(0, shard.getNumId());
        assertEquals("solomon", shard.getProjectId());
        assertEquals("solomon_production", shard.getClusterId());
        assertEquals("production", shard.getClusterName());
        assertEquals("solomon_compute", shard.getServiceId());
        assertEquals("compute", shard.getServiceName());
        assertEquals(10_000, shard.getMaxFileMetrics());
        assertEquals(100_000, shard.getMaxMemMetrics());
        assertEquals(Shard.DEFAULT_PER_URL_QUOTA, shard.getMaxMetricsPerUrl());
        assertEquals(Shard.DEFAULT_RESPONSE_SIZE_QUOTA, shard.getMaxResponseSizeBytes());
        assertEquals(30, shard.getShardSettings().getMetricsTtl());
        assertEquals("name", shard.getMetricNameLabel());
        assertEquals("robot-solomon", shard.getCreatedBy());
        assertEquals("robot-solomon", shard.getUpdatedBy());

        List<Service> services = servicesDao.findAll().join();
        assertEquals(1, services.size());

        Service service = services.get(0);
        assertEquals("solomon_compute", service.getId());
        assertEquals("compute", service.getName());
        assertEquals(0, service.getShardSettings().getMetricsTtl());
        assertEquals(Service.GRID_UNKNOWN, service.getShardSettings().getGrid());
    }

    @Test
    public void concurrentCreation() {
        final int threads = 5;
        ExecutorService executor = Executors.newFixedThreadPool(threads);
        CountDownLatch latch = new CountDownLatch(1);

        ShardKey shardKey = new ShardKey("solomon", "production", "stockpile");
        ShardCreator creator = makeCreator(TCoremonCreateShardConfig.getDefaultInstance());

        List<CompletableFuture<Shard>> futures = new ArrayList<>(threads);
        for (int i = 0; i < threads; i++) {
            var future = CompletableFuture.supplyAsync(() -> {
                        CountDownLatches.await(latch);
                        return creator.createShard(shardKey, "jamel");
                    }, executor)
                    .thenCompose(f -> f);
            futures.add(future);
        }

        var shardsFuture = CompletableFutures.allOf(futures);
        latch.countDown();

        ListF<Shard> shards = shardsFuture.join();
        assertEquals(threads, shards.size());

        Shard firstShard = shards.get(0);
        for (int i = 1; i < shards.size(); i++) {
            assertEquals(firstShard, shards.get(i));
        }
    }

    @Test
    public void notValidShardKey() {
        ShardCreator creator = makeCreator(TCoremonCreateShardConfig.getDefaultInstance());
        ShardKey key = new ShardKey("не валидный проект", "не валидный кластер", "не валидный сервис");
        var error = creator.createShard(key, "gordiychuk")
                .thenApply(shard -> (Throwable) null)
                .exceptionally(e -> e)
                .join();
        assertNotNull(error);
        error.printStackTrace();
    }

    private ShardCreator makeCreator(TCoremonCreateShardConfig shardConf) {
        var creator = new ShardCreator(
                projectsDao,
                shardsDao,
                clustersDao,
                servicesDao,
                numIdGenerator,
                shardConf,
                executor,
                MetricRegistry.root(),
                Optional.of(folderResolver));
        creator.onConfigurationLoad(SolomonConfWithContext.create(SolomonRawConf.EMPTY));
        return creator;
    }
}
