package ru.yandex.solomon.coremon.stockpile;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.TreeMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.MetricEncoder;
import ru.yandex.monlib.metrics.encode.json.MetricJsonEncoder;
import ru.yandex.monlib.metrics.encode.spack.MetricSpackEncoder;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.core.conf.ShardKeyAndId;
import ru.yandex.solomon.core.conf.aggr.AggrRuleConf;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.db.model.ValidationMode;
import ru.yandex.solomon.coremon.CoremonShardOpts;
import ru.yandex.solomon.coremon.CoremonShardQuota;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.MetricsCollection;
import ru.yandex.solomon.coremon.meta.db.MetabaseShardStorage;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.stockpile.test.TreeParserForTest;
import ru.yandex.solomon.coremon.stockpile.write.StockpileBufferedWriter;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.util.collection.Collectors2;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Stepan Koltsov
 */
public class CoremonShardStockpileTest {

    private static final int STEP_SECONDS = 15;
    private static final int STEP_MILLIS = STEP_SECONDS * 1000;
    private static final int SHARD_METRICS_COUNT = 4_000;
    private static final double EPSILON = 0.001;

    private TreeParserForTest treeParser;
    private StockpileClientStub stockpileClient;
    private CoremonShardStockpileWriteHelper writeHelper;
    private final ExecutorService executorService = Executors.newScheduledThreadPool(4);

    private ShardKeyAndId shardKeyAndId;
    private final int numId = 42;
    private CoremonShardStockpile coremonShardStockpile;
    private MetabaseShardImpl metabaseShard;
    private MetricsDaoFactory metricsDaoFactory;

    @Before
    public void setUp() {
        treeParser = new TreeParserForTest("");
        stockpileClient = new StockpileClientStub(executorService);

        metricsDaoFactory = new InMemoryMetricsDaoFactory();
        ShardKey shardKey = new ShardKey("xx", "yy", "zz");
        shardKeyAndId = new ShardKeyAndId(shardKey, CoremonShardStockpileTest.class.getSimpleName(), numId);

        AggrRuleConf[] aggrRules = {
            AggrRuleConf.simpleReplaceRule("host", "DC"),
            AggrRuleConf.simpleReplaceRule("host", "cluster")
        };

        CoremonMericConfDetailed metricConfDetailed =
            new CoremonMericConfDetailed(aggrRules, false);

        CoremonShardOpts opts = CoremonShardOpts.newBuilder()
                .withId(shardKeyAndId)
                .withFetchMillis(TimeUnit.SECONDS.toMillis(STEP_SECONDS))
                .withDecimPolicy(DecimPolicy.DEFAULT)
                .withQuota(CoremonShardQuota.DEFAULT)
                .withValidationMode(ValidationMode.STRICT_FAIL)
                .withItemState(ShardState.RW)
                .withMetricNameLabel("")
                .build();

        MetricProcessorFactory processorFactory = new MetricProcessorFactory(new FeatureFlagHolderStub());
        StockpileBufferedWriter stockpileWriter = new StockpileBufferedWriter(stockpileClient, executorService, MetricRegistry
            .root());
        metabaseShard = new MetabaseShardImpl(
                opts.getId().getNumId(),
                opts.getId().getShardId(),
                opts.getFolderId(),
                opts.getNumPartitions(),
                opts.getId().getShardKey(),
                metricsDaoFactory,
                Labels.allocator,
                executorService,
                stockpileClient
        );
        coremonShardStockpile = new CoremonShardStockpile(
                metricConfDetailed,
                treeParser,
                Labels.allocator,
                executorService,
                executorService,
                stockpileClient,
                processorFactory,
                opts,
                stockpileWriter,
                metabaseShard);
        coremonShardStockpile.serverFormat = StockpileFormat.CURRENT;
        writeHelper = stockpileWriter.newHelper(coremonShardStockpile.getNumId(), coremonShardStockpile.getMetrics());
        // need synchronization because old reference to MetabaseShardStorage can leak to CoremonShardStockpileResolveHelper
        metabaseShard.awaitReady();
    }

    @After
    public void tearDown() {
        if (coremonShardStockpile != null) {
            coremonShardStockpile.stop();
        }
        if (metabaseShard != null) {
            metabaseShard.stop();
        }
        executorService.shutdownNow();
    }

    @Nonnull
    private TreeParserForTest.MetricPoint newRequest(Labels labels, long nowMillis, int valueNum) {
        return newRequest(labels, nowMillis, valueNum, MetricType.DGAUGE);
    }

    @Nonnull
    private TreeParserForTest.MetricPoint newRequest(
        Labels labels,
        long nowMillis,
        int valueNum,
        MetricType metricType)
    {
        return newRequest(labels, nowMillis, valueNum, metricType, false);
    }

    @Nonnull
    private TreeParserForTest.MetricPoint newRequest(
        Labels labels,
        long nowMillis,
        int valueNum,
        MetricType metricType,
        boolean memOnly)
    {
        return new TreeParserForTest.MetricPoint(
            labels,
            valueNum,
            memOnly,
            metricType,
            nowMillis);
    }

    private void processTwoPagesSync(long nowMillis, TreeParserForTest.MetricPoint[] request) {
        ByteBuf requestData = treeParser.addMetricBatch(request);

        CoremonShardStockpileResolveHelper resolveHelper = createResolveHelper();
        coremonShardStockpile.processTwoPages(
            Labels.empty(),
            Labels.empty(),
            MetricFormat.METRIC_FORMAT_UNSPECIFIED,
            requestData, nowMillis,
            Unpooled.EMPTY_BUFFER, nowMillis - STEP_MILLIS,
            false,
            false,
            resolveHelper,
            writeHelper
        );

        writeHelper.write().join();
        coremonShardStockpile.postProcessUnresolvedMetrics(resolveHelper).join();
    }

    @Test
    public void dontAllowWriteToReadOnlyShard() {
        CoremonShardOpts opts = coremonShardStockpile.getOpts();
        CoremonShardOpts.Builder builder =
            new CoremonShardOpts.Builder(opts);
        {
            TwoResponsesRequest twoResponsesRequest = generateTwoResponsesRequest();
            coremonShardStockpile.setOpts(builder.withItemState(ShardState.READ_ONLY).build());
            var result = coremonShardStockpile.pushPage(twoResponsesRequest).join();
            Assert.assertEquals(UrlStatusType.SHARD_IS_NOT_WRITABLE, result.getStatusType());
        }
        {
            TwoResponsesRequest twoResponsesRequest = generateTwoResponsesRequest();
            coremonShardStockpile.setOpts(builder.withItemState(ShardState.INACTIVE).build());
            var result = coremonShardStockpile.pushPage(twoResponsesRequest).join();
            Assert.assertEquals(UrlStatusType.SHARD_IS_NOT_WRITABLE, result.getStatusType());
        }
        {
            TwoResponsesRequest twoResponsesRequest = generateTwoResponsesRequest();
            coremonShardStockpile.setOpts(builder.withItemState(ShardState.ACTIVE).build());
            var result = coremonShardStockpile.pushPage(twoResponsesRequest).join();
            Assert.assertEquals(UrlStatusType.OK, result.getStatusType());
        }
        {
            TwoResponsesRequest twoResponsesRequest = generateTwoResponsesRequest();
            coremonShardStockpile.setOpts(builder.withItemState(ShardState.WRITE_ONLY).build());
            var result = coremonShardStockpile.pushPage(twoResponsesRequest).join();
            Assert.assertEquals(UrlStatusType.OK, result.getStatusType());
        }
    }

    private TwoResponsesRequest generateTwoResponsesRequest() {
        String date = "2015-12-13T12:13:14Z";
        CoremonMetric metric = new FileCoremonMetric(
            14,
            230420384L,
            Labels.of("host", "aabb", "sensor", "xxyy"),
            InstantUtils.parseToSeconds(date),
            MetricType.DGAUGE);

        long expectedTs = TimeUnit.SECONDS.toMillis(metric.getCreatedAtSeconds());
        int expectedValue = 33;
        TreeParserForTest.MetricPoint request = newRequest(metric.getLabels(), expectedTs, expectedValue);

        processTwoPagesSync(expectedTs, new TreeParserForTest.MetricPoint[]{request});

        ByteBuf requestData = treeParser.addMetricBatch(request);
        return new TwoResponsesRequest(
            Labels.empty(),
            Labels.empty(),
            MetricFormat.METRIC_FORMAT_UNSPECIFIED,
            requestData, expectedTs,
            Unpooled.EMPTY_BUFFER, expectedTs - STEP_MILLIS,
            false,
            false
        );
    }

    @Test
    public void metricLoadedFromMetaBase() {
        CoremonMetric metric = new FileCoremonMetric(
            14,
            230420384L,
            Labels.of("host", "aabb", "sensor", "xxyy"),
            InstantUtils.parseToSeconds("2015-12-13T12:13:14Z"),
            MetricType.DGAUGE);

        long expectedTsMillis = TimeUnit.SECONDS.toMillis(metric.getCreatedAtSeconds());
        int expectedValue = 33;
        TreeParserForTest.MetricPoint request = newRequest(metric.getLabels(), expectedTsMillis, expectedValue);

        processTwoPagesSync(expectedTsMillis, new TreeParserForTest.MetricPoint[]{request});

        // raw metric, dc aggr, cluster aggr
        List<CoremonMetric> metrics = findMetrics(shardKeyAndId.getShardId());
        Assert.assertEquals(3, metrics.size());

        for (CoremonMetric metricRecordResolved : metrics) {
            GraphData data = readGraphData(metricRecordResolved);
            Assert.assertEquals((double) expectedValue, data.getValues().at(0), EPSILON);
        }
    }

    private AggrGraphDataArrayList readAggrList(CoremonMetric s) {
        MetricArchiveImmutable archive = stockpileClient.getTimeSeries(s.getShardId(), s.getLocalId());
        if (archive == null) {
            return AggrGraphDataArrayList.empty();
        }
        return archive.toAggrGraphDataArrayList();
    }

    private GraphData readGraphData(CoremonMetric s) {
        return readAggrList(s).toGraphDataShort();
    }

    @Test
    public void metricsCreatedDataStored() {
        long nowMillis = Instant.parse("2015-12-13T12:13:14Z").toEpochMilli();
        TreeParserForTest.MetricPoint request1 = newRequest(Labels.of("host", "qwqw", "sensor", "writes"), nowMillis, 22);
        TreeParserForTest.MetricPoint request2 = newRequest(Labels.of("host", "qwqw", "sensor", "reads"), nowMillis, 33);
        TreeParserForTest.MetricPoint request3 = newRequest(Labels.of("host", "aabb", "sensor", "reads"), nowMillis, 44);

        processTwoPagesSync(nowMillis, new TreeParserForTest.MetricPoint[]{request1, request2});
        processTwoPagesSync(nowMillis, new TreeParserForTest.MetricPoint[]{request3});

        Map<Labels, CoremonMetric> metrics = findMetrics(shardKeyAndId.getShardId())
            .stream().collect(Collectors2.toMapMappingToKey(CoremonMetric::getLabels));
        Assert.assertEquals(7, metrics.size()); // 3 raw, reads/writes in dc, reads/writes in cluster

        CoremonMetric metric1 = Objects.requireNonNull(metrics.get(request1.getKey()));
        CoremonMetric metric2 = Objects.requireNonNull(metrics.get(request2.getKey()));
        CoremonMetric metric3 = Objects.requireNonNull(metrics.get(request3.getKey()));

        // must be created in the same group, because pushed in single batch
        Assert.assertEquals(metric1.getShardId(), metric2.getShardId());
        // must be created in different groups, because in different batches
        Assert.assertNotEquals(metric1.getShardId(), metric3.getShardId());

        GraphData data = readGraphData(metric1);
        TreeMap<Long, Double> expected = new TreeMap<>();
        expected.put(nowMillis, request1.getValue());

        Assert.assertEquals(expected, data.toTreeMap());
    }

    @Test
    public void numDenom() {
        long ts = Instant.parse("2017-01-28T12:13:15Z").toEpochMilli();

        TreeParserForTest.MetricPoint request1 =
            newRequest(Labels.of("host", "hhh1", "sensor", "writes"), ts, 30);

        TreeParserForTest.MetricPoint request2 =
            newRequest(Labels.of("host", "hhh2", "sensor", "writes"), ts, 45);

        processTwoPagesSync(ts, new TreeParserForTest.MetricPoint[]{request1, request2});

        CoremonMetric metric1 = findSingleMetric(shardKeyAndId.getShardId(), Labels.of("host", "hhh1", "sensor", "writes"));
        CoremonMetric metric2 = findSingleMetric(shardKeyAndId.getShardId(), Labels.of("host", "hhh2", "sensor", "writes"));
        CoremonMetric metricC = findSingleMetric(shardKeyAndId.getShardId(), Labels.of("host", "cluster", "sensor", "writes"));

        GraphData data1 = readGraphData(metric1);
        GraphData data2 = readGraphData(metric2);
        GraphData dataC = readGraphData(metricC);

        Assert.assertEquals(30, data1.single().getValue(), EPSILON);
        Assert.assertEquals(45, data2.single().getValue(), EPSILON);
        Assert.assertEquals(75, dataC.single().getValue(), EPSILON);
    }

    @Test
    public void aggrTwoMetricsOneTsTwoTargetsDifferentRequests() {
        long ts = Instant.parse("2015-12-13T12:13:00Z").toEpochMilli();
        TreeParserForTest.MetricPoint request1
            = newRequest(Labels.of("host", "aabb", "sensor", "writes"), ts, 17);
        TreeParserForTest.MetricPoint request2
            = newRequest(Labels.of("host", "ccdd", "sensor", "writes"), ts, 19);

        processTwoPagesSync(ts, new TreeParserForTest.MetricPoint[]{request1});

        {
            List<CoremonMetric> metrics = findMetrics(shardKeyAndId.getShardId());
            Assert.assertEquals(3, metrics.size()); /* raw1, dc, cluster */
        }

        // push in different batches so += merged on storage
        processTwoPagesSync(ts, new TreeParserForTest.MetricPoint[]{request2});

        {
            List<CoremonMetric> metrics = findMetrics(shardKeyAndId.getShardId());
            Assert.assertEquals(4, metrics.size()); /* raw1, raw2, dc, cluster */

            CoremonMetric sc = metrics.stream()
                .filter(s -> s.getLabels().equals(Labels.of("host", "cluster", "sensor", "writes")))
                .collect(CollectorsF.single());

            AggrGraphDataArrayList expected = new AggrGraphDataArrayList();
            expected.addRecordFullForTest(ts, 36, true, 2, 15000);

            AggrPoint point = readAggrList(sc).getAnyPoint(0);
            Assert.assertEquals(17 + 19, point.valueNum, EPSILON);
            Assert.assertEquals(2, point.getCount());
            Assert.assertEquals(STEP_MILLIS, point.getStepMillis());
            Assert.assertTrue(point.isMerge());
        }
    }

    @Test
    public void aggrTwoMetricsOneTsTwoTargetsOneRequest() {
        long ts = Instant.parse("2015-12-13T12:13:00Z").toEpochMilli();
        TreeParserForTest.MetricPoint request1 = newRequest(Labels.of("host", "aabb", "sensor", "writes"), ts, 17);
        TreeParserForTest.MetricPoint request2 = newRequest(Labels.of("host", "ccdd", "sensor", "writes"), ts, 19);

        processTwoPagesSync(ts, new TreeParserForTest.MetricPoint[]{request1, request2});

        {
            List<CoremonMetric> metrics = findMetrics(shardKeyAndId.getShardId());
            Assert.assertEquals(4, metrics.size()); /* raw1, raw2, dc, cluster */

            CoremonMetric sc = metrics.stream()
                .filter(s -> s.getLabels().equals(Labels.of("host", "cluster", "sensor", "writes")))
                .collect(CollectorsF.single());

            AggrGraphDataArrayList expected = new AggrGraphDataArrayList();
            expected.addRecordFullForTest(ts, 36, true, 2, 15000);

            AggrGraphDataArrayList actual = readAggrList(sc);
            Assert.assertEquals(expected, actual);
        }
    }

    @Test
    public void aggrOneMetricTwoTsOneTarget() {
        long ts = Instant.parse("2015-12-13T12:13:00Z").toEpochMilli();
        TreeParserForTest.MetricPoint request1 = newRequest(Labels.of("host", "aabb", "sensor", "writes"), ts, 17);
        TreeParserForTest.MetricPoint
            request2 = newRequest(Labels.of("host", "aabb", "sensor", "writes"), ts + 15000, 19);

        processTwoPagesSync(ts, new TreeParserForTest.MetricPoint[]{request1, request2});

        {
            List<CoremonMetric> metrics = findMetrics(shardKeyAndId.getShardId());
            Assert.assertEquals(3, metrics.size()); /* raw, dc, cluster */

            CoremonMetric sc = metrics.stream()
                .filter(s -> s.getLabels().equals(Labels.of("host", "cluster", "sensor", "writes")))
                .collect(CollectorsF.single());

            AggrGraphDataArrayList expected = new AggrGraphDataArrayList();
            expected.addRecordFullForTest(ts, 17, true, 1, 15000);
            expected.addRecordFullForTest(ts + 15000, 19, true, 1, 15000);

            AggrGraphDataArrayList actual = readAggrList(sc);
            Assert.assertEquals(expected, actual);
        }
    }

    @Test
    public void rawDataMemOnly() {
        long ts = Instant.parse("2015-12-13T12:13:00Z").toEpochMilli();

        int value = 14000;
        TreeParserForTest.MetricPoint request1 = newRequest(
            Labels.of("host", "qwqw", "sensor", "writes"),
            ts,
            value,
            MetricType.DGAUGE,
            true
        );

        processTwoPagesSync(ts, new TreeParserForTest.MetricPoint[]{ request1 });

        List<CoremonMetric> metrics = findMetrics(shardKeyAndId.getShardId());
        Assert.assertEquals(2, metrics.size()); /* dc, cluster */

        for (String aggr : Arrays.asList("cluster", "DC")) {

            CoremonMetric metric = metrics.stream()
                .filter(x -> {
                    Label host = x.getLabels().findByKey("host");
                    return host != null && host.getValue().equals(aggr);
                })
                .collect(CollectorsF.single());

            Label host = metric.getLabels().findByKey("host");
            Assert.assertNotNull(host);
            Assert.assertEquals(aggr, host.getValue());

            GraphData graphData = readGraphData(metric);
            Assert.assertEquals(value, graphData.getValues().at(0), EPSILON);
        }
    }

    @Test
    public void aggrMetricsSpread() {
        long ts = Instant.parse("2015-12-13T12:13:00Z").toEpochMilli();
        ArrayList<TreeParserForTest.MetricPoint> requests = new ArrayList<>();
        int numMetricsRaw = SHARD_METRICS_COUNT * 2;
        for (int i = 0; i < numMetricsRaw; i++) {
            requests.add(newRequest(Labels.of("host", "aabb", "sensor", "sensor_" + i), ts, 17));
        }
        processTwoPagesSync(ts, requests.toArray(new TreeParserForTest.MetricPoint[requests.size()]));

        List<CoremonMetric> metrics = findMetrics(shardKeyAndId.getShardId());
        Assert.assertEquals(3 * numMetricsRaw, metrics.size());  // n raw, n for dc, n for cluster

        long countDifferentShards = metrics.stream()
            .map(CoremonMetric::getShardId)
            .distinct()
            .count();
        Assert.assertEquals(3, countDifferentShards);

        for (String aggrHost : List.of("DC", "cluster")) {
            Map<Integer, Long> collect = metrics.stream()
                .filter(metric -> {
                    Label host = metric.getLabels().findByKey("host");
                    return host != null && host.getValue().equals(aggrHost);
                })
                .map(CoremonMetric::getShardId)
                .collect(Collectors.groupingBy(x -> x, Collectors.counting()));

            // verification stage
            Assert.assertEquals(1, collect.size()); // 8k metrics in one shard

            for (long metricNumInShard : collect.values()) {
                Assert.assertEquals(SHARD_METRICS_COUNT * 2, metricNumInShard);
            }
        }
    }

    @Test
    public void updateMetricMode() {
        Labels labels = Labels.of("host", "qwqw", "sensor", "writes");

        CoremonMetric metric = new FileCoremonMetric(
            14,
            230420384L,
            labels,
            InstantUtils.parseToSeconds("2015-12-13T12:13:00Z"),
            MetricType.RATE);

        MetricsDao metricsDao = metricsDaoFactory.create(numId);
        try (CoremonMetricArray metrics = new CoremonMetricArray(metric)) {
            CompletableFutures.join(metricsDao.replaceMetrics(metrics));
        }
        Assert.assertEquals(1, findMetrics(shardKeyAndId.getShardId()).size());

        TreeParserForTest.MetricPoint requestAsIs = newRequest(labels, 0, 14000, MetricType.DGAUGE);
        processTwoPagesSync(System.currentTimeMillis(), new TreeParserForTest.MetricPoint[]{requestAsIs});

        CoremonMetric updatedMetric = findSingleMetric(shardKeyAndId.getShardId(), labels);
        Assert.assertEquals(MetricType.RATE, updatedMetric.getType());
    }

    @Test
    public void asyncDbUpdateFromAnotherPartition() {
        long ts0 = Instant.parse("2015-12-13T12:13:14Z").toEpochMilli();

        int expectedValue = 33;
        TreeParserForTest.MetricPoint
            request1 = newRequest(Labels.of("host", "aabb", "sensor", "xxyy"), ts0, expectedValue);
        TreeParserForTest.MetricPoint
            request2 = newRequest(Labels.of("host", "ccdd", "sensor", "qqww"), ts0, expectedValue);

        CoremonShardStockpileResolveHelper resolveHelper1 = createResolveHelper();
        CoremonShardStockpileResolveHelper resolveHelper2 = createResolveHelper();

        // Partition #1
        coremonShardStockpile.processTwoPages(
            Labels.empty(),
            Labels.empty(),
            MetricFormat.METRIC_FORMAT_UNSPECIFIED,
            treeParser.addMetricBatch(request1), ts0,
            Unpooled.EMPTY_BUFFER, ts0 - STEP_MILLIS,
            false,
            false,
            resolveHelper1,
            writeHelper
        );

        // Partition #2
        coremonShardStockpile.processTwoPages(
            Labels.empty(),
            Labels.empty(),
            MetricFormat.METRIC_FORMAT_UNSPECIFIED,
            treeParser.addMetricBatch(request2), ts0,
            Unpooled.EMPTY_BUFFER, ts0 - STEP_MILLIS,
            false,
            false,
            resolveHelper2,
            writeHelper
        );

        coremonShardStockpile.postProcessUnresolvedMetrics(resolveHelper1).join();
        coremonShardStockpile.postProcessUnresolvedMetrics(resolveHelper2).join();

        // (raw, dc, cluster) * 2
        Assert.assertEquals(6, findMetrics(shardKeyAndId.getShardId()).size());

        for (String metricStr : Arrays.asList("aabb", "ccdd")) {
            CoremonMetric metric = findMetrics(shardKeyAndId.getShardId()).stream()
                .filter(x -> {
                    Label host = x.getLabels().findByKey("host");
                    return host != null && host.getValue().equals(metricStr);
                })
                .collect(CollectorsF.single());

            GraphData data = readGraphData(metric);
            TreeMap<Long, Double> expected = new TreeMap<>();
            expected.put(ts0, (double) expectedValue);

            Assert.assertEquals(expected, data.toTreeMap());
        }
    }

    private CoremonShardStockpileResolveHelper createResolveHelper() {
        MetabaseShardStorage storage = metabaseShard.getStorage();
        MetricsCollection<CoremonMetric> fileMetrics = storage.getFileMetrics();
        return new CoremonShardStockpileResolveHelper(fileMetrics, CoremonShardQuota.DEFAULT);
    }

    @Test
    public void asyncDbUpdateFromAnotherProcess() {
        CoremonMetric metric = new FileCoremonMetric(
            14,
            230420384L,
            Labels.of("host", "aabb", "sensor", "xxyy"),
            InstantUtils.parseToSeconds("2015-12-13T12:13:14Z"),
            MetricType.DGAUGE);

        long expectedTsMillis = TimeUnit.SECONDS.toMillis(metric.getCreatedAtSeconds());
        int expectedValue = 33;
        TreeParserForTest.MetricPoint request = newRequest(
            metric.getLabels(), expectedTsMillis, expectedValue, metric.getType(), true);

        processTwoPagesSync(expectedTsMillis, new TreeParserForTest.MetricPoint[]{request});

        List<CoremonMetric> metrics = findMetrics(shardKeyAndId.getShardId());
        // dc & cluster aggregates
        Assert.assertEquals(2, metrics.size());

        GraphData data = readGraphData(metric);

        Assert.assertEquals(0, data.getValues().length());
        for (CoremonMetric aggrMetric : metrics) {
            GraphData aggrData = readGraphData(aggrMetric);
            Assert.assertEquals(expectedValue, aggrData.getValues().at(0), EPSILON);
        }
    }

    @Test
    public void spackParseRateSameSize() {
        long ts = Instant.parse("2015-12-13T12:13:00Z").toEpochMilli();
        MetricFormat format = MetricFormat.SPACK;

        MetricRegistry registry = new MetricRegistry();
        registry.rate("alice").set(100);
        registry.rate("bob").set(42);
        var one = encode(format, registry);

        registry.rate("alice").set(5000);
        registry.rate("bob").set(200);
        var two = encode(format, registry);

        registry.rate("alice").set(5000);
        registry.rate("bob").set(500);
        var tree = encode(format, registry);

        processTwoPagesSync(format, one, two, ts);
        processTwoPagesSync(format, two, tree, ts + STEP_MILLIS);

        Assert.assertEquals(2, findMetrics(shardKeyAndId.getShardId()).size());

        {
            var metric = findSingleMetric(shardKeyAndId.getShardId(), Labels.of("sensor", "alice"));
            Assert.assertNotNull(metric);

            var data = readAggrList(metric);
            var expected = AggrGraphDataArrayList.of(
                AggrPoint.builder()
                    .time(ts)
                    .doubleValue(4900, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build(),

                AggrPoint.builder()
                    .time(ts + STEP_MILLIS)
                    .doubleValue(0, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build());
            Assert.assertEquals(expected, data);
        }

        {
            var metric = findSingleMetric(shardKeyAndId.getShardId(), Labels.of("sensor", "bob"));

            var data = readAggrList(metric);
            var expected = AggrGraphDataArrayList.of(
                AggrPoint.builder()
                    .time(ts)
                    .doubleValue(200 - 42, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build(),

                AggrPoint.builder()
                    .time(ts + STEP_MILLIS)
                    .doubleValue(500 - 200, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build());
            Assert.assertEquals(expected, data);
        }
    }

    @Test
    public void spackParserDifferentSize() {
        long ts = Instant.parse("2015-12-13T12:13:00Z").toEpochMilli();
        MetricFormat format = MetricFormat.SPACK;

        MetricRegistry registry = new MetricRegistry();
        registry.rate("bob").set(42);
        var one = encode(format, registry);

        registry.rate("alice").set(5000);
        registry.rate("bob").set(200);
        var two = encode(format, registry);

        registry.rate("alice").set(9000);
        registry.rate("bob").set(500);
        registry.rate("eva").set(100);
        var tree = encode(format, registry);

        registry.rate("alice").set(10000);
        registry.rate("bob").set(600);
        registry.rate("eva").set(900);
        registry.rate("carol").set(2);
        var four = encode(format, registry);

        processTwoPagesSync(format, one, two, ts);
        processTwoPagesSync(format, two, tree, ts + STEP_MILLIS);
        processTwoPagesSync(format, tree, four, ts + STEP_MILLIS * 2);

        Assert.assertEquals(3, findMetrics(shardKeyAndId.getShardId()).size());

        {
            var metric = findSingleMetric(shardKeyAndId.getShardId(), Labels.of("sensor", "alice"));
            Assert.assertNotNull(metric);

            var data = readAggrList(metric);
            var expected = AggrGraphDataArrayList.of(
                AggrPoint.builder()
                    .time(ts + STEP_MILLIS)
                    .doubleValue(9000 - 5000, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build(),

                AggrPoint.builder()
                    .time(ts + STEP_MILLIS * 2)
                    .doubleValue(10000 - 9000, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build());
            Assert.assertEquals(expected, data);
        }

        {
            var metric = findSingleMetric(shardKeyAndId.getShardId(), Labels.of("sensor", "bob"));
            Assert.assertNotNull(metric);

            var data = readAggrList(metric);
            var expected = AggrGraphDataArrayList.of(
                AggrPoint.builder()
                    .time(ts)
                    .doubleValue(200 - 42, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build(),

                AggrPoint.builder()
                    .time(ts + STEP_MILLIS)
                    .doubleValue(500 - 200, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build(),

                AggrPoint.builder()
                    .time(ts + STEP_MILLIS * 2)
                    .doubleValue(600 - 500, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build()
                );
            Assert.assertEquals(expected, data);
        }

        {
            var metric = findSingleMetric(shardKeyAndId.getShardId(), Labels.of("sensor", "eva"));
            Assert.assertNotNull(metric);

            var data = readAggrList(metric);
            var expected = AggrGraphDataArrayList.of(
                AggrPoint.builder()
                    .time(ts + STEP_MILLIS * 2)
                    .doubleValue(900 - 100, STEP_MILLIS)
                    .stepMillis(STEP_MILLIS)
                    .build()
            );
            Assert.assertEquals(expected, data);
        }
    }

    private void processTwoPagesSync(MetricFormat format, ByteBuf prev, ByteBuf now, long nowMillis) {
        CoremonShardStockpileResolveHelper resolveHelper = createResolveHelper();
        coremonShardStockpile.processTwoPages(
            Labels.empty(),
            Labels.empty(),
            format,
            now, nowMillis,
            prev, nowMillis - STEP_MILLIS,
            false,
            false,
            resolveHelper,
            writeHelper
        );

        writeHelper.write().join();
        coremonShardStockpile.postProcessUnresolvedMetrics(resolveHelper).join();
    }

    private ByteBuf encode(MetricFormat format, MetricRegistry registry) {
        ByteArrayOutputStream out = new ByteArrayOutputStream(8 << 10); // 8 KiB
        try (MetricEncoder encoder = createEncoder(format, out)) {
            registry.supply(0, encoder);
        } catch (Exception e) {
            throw new IllegalStateException("cannot encode metrics", e);
        }

        return Unpooled.wrappedBuffer(out.toByteArray());
    }

    private MetricEncoder createEncoder(MetricFormat format, OutputStream out) {
        switch (format) {
            case SPACK:
                return new MetricSpackEncoder(TimePrecision.MILLIS, CompressionAlg.LZ4, out);
            case JSON:
                return new MetricJsonEncoder(out);
            default:
                throw new UnsupportedOperationException("unsupported format: " + format);
        }
    }

    private List<CoremonMetric> findMetrics(String shardId) {
        List<CoremonMetric> result = new ArrayList<>();
        MetricsDao metricsDao = metricsDaoFactory.create(numId);
        metricsDao.findMetrics(chunk -> {
            for (int i = 0; i < chunk.size(); i++) {
                result.add(new FileCoremonMetric(chunk.get(i)));
            }
        }, OptionalLong.empty()).join();
        return result;
    }

    private CoremonMetric findSingleMetric(String shardId, Labels labels) {
        return findMetrics(shardId).stream()
            .filter(s -> s.getLabels().equals(labels))
            .collect(CollectorsF.single());
    }
}
