package ru.yandex.solomon.coremon.stockpile;

import java.time.Instant;
import java.util.List;

import javax.annotation.Nullable;

import com.google.common.collect.Iterators;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.core.conf.aggr.AggrRuleConf;
import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.core.db.model.ValidationMode;
import ru.yandex.solomon.core.urlStatus.UrlStatusTypeException;
import ru.yandex.solomon.coremon.CoremonShardQuota;
import ru.yandex.solomon.coremon.aggregates.AggrHelper;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.TestMetricsCollection;
import ru.yandex.solomon.coremon.meta.mem.MemOnlyMetricsCollectionImpl;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricData;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricPoint;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricTimeSeries;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.parser.TreeParser;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPoints;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static java.util.Objects.requireNonNull;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricProcessorImplTest {
    private static final int STEP_MILLIS = 15_000;
    private static final long RESPONSE_TS_MILLIS = Instant.parse("2017-12-02T13:14:15.167Z").toEpochMilli();

    private static final AggrHelper aggrHelper = new AggrHelper(new AggrRuleConf[]{
            makeAggrRule("host=*", "host=cluster"),
            makeAggrRule("host=*, core=*", "host=cluster, core=total"),
            makeAggrRule("host=*, disk=*", "host=cluster, disk=total"),
            makeAggrRule("host=*, name=*last", "host=cluster", MetricAggregation.LAST),
    }, EDecimPolicy.UNDEFINED);

    private TestMetricsCollection<CoremonMetric> fileMetrics;
    private MemOnlyMetricsCollectionImpl memOnlyMetrics;
    private TestStockpileWriteHelper writer;
    private MetricProcessorImpl processor;
    private MetricsPrevValuesImpl prevValues;
    private CoremonShardStockpileResolveHelper resolveHelper;


    private static AggrRuleConf makeAggrRule(String cond, String target) {
        return makeAggrRule(cond, target, null);
    }

    private static AggrRuleConf makeAggrRule(String cond, String target, @Nullable MetricAggregation aggregation) {
        var rule = AggrRule.of(cond, target, aggregation);
        var shard = new ShardKey("fake-project", "fake-cluster", "fake-service");
        return new AggrRuleConf(rule, shard);
    }

    @Before
    public void setUp() {
        fileMetrics = new TestMetricsCollection<>();
        memOnlyMetrics = new MemOnlyMetricsCollectionImpl();
        writer = new TestStockpileWriteHelper();
        prevValues = new MetricsPrevValuesImpl();
        resolveHelper = new CoremonShardStockpileResolveHelper(fileMetrics, CoremonShardQuota.DEFAULT);
        processor = createProcessor(false, ValidationMode.STRICT_FAIL, STEP_MILLIS, false);
    }

    @Test
    public void onePointCounter() {
        CoremonMetric metric = makeMetric(
                Labels.of("sensor", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name()),
                ru.yandex.solomon.model.protobuf.MetricType.COUNTER);
        fileMetrics.put(metric);

        long now = System.currentTimeMillis();
        long value = 42;
        processor.onMetricBegin(metric.getType(), metric.getLabels(), false);
        processor.onPoint(now, value);

        AggrPoint expectedPoint = new AggrPoint();
        expectedPoint.setTsMillis(now);
        expectedPoint.setValue(value, ValueColumn.DEFAULT_DENOM);
        expectedPoint.setStepMillis(STEP_MILLIS);

        MetricArchiveMutable archive = writer.getArchive(metric.getShardId(), metric.getLocalId());
        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE, archive.getType());
        assertEquals(1, archive.getRecordCount());
        assertEquals(expectedPoint, getFirstPoint(archive));
    }

    @Test
    public void manyPointsCounter() {
        CoremonMetric metric = makeMetric(
                Labels.of("sensor", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name()),
                ru.yandex.solomon.model.protobuf.MetricType.COUNTER);
        fileMetrics.put(metric);

        long ts0 = System.currentTimeMillis();
        processor.onMetricBegin(metric.getType(), metric.getLabels(), false);
        processor.onTimeSeries(TimeSeries.empty()
                .addLong(ts0, 1)
                .addLong(ts0 + 15_000, 2)
                .addLong(ts0 + 30_000, 3));

        AggrGraphDataArrayList expectedTimeSeries = AggrGraphDataArrayList.of(
                AggrPoint.shortPoint(ts0, 1.0, 15_000),
                AggrPoint.shortPoint(ts0 + 15_000, 2.0, 15_000),
                AggrPoint.shortPoint(ts0 + 30_000, 3.0, 15_000)
        );

        MetricArchiveMutable archive = writer.getArchive(metric.getShardId(), metric.getLocalId());
        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE, archive.getType());
        assertEquals(3, archive.getRecordCount());
        assertEquals(expectedTimeSeries, archive.toAggrGraphDataArrayList());
    }

    @Test
    public void onePointUnresolvedMetricsDelayWrite() {
        Labels labels = Labels.of("sensor", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name());

        long now = System.currentTimeMillis();
        long value = 42;
        processor.onMetricBegin(MetricType.COUNTER, labels, false);
        processor.onPoint(now, value);

        AggrPoint expectedPoint = new AggrPoint();
        expectedPoint.setTsMillis(now);
        expectedPoint.setValue(value, ValueColumn.DEFAULT_DENOM);
        expectedPoint.setStepMillis(STEP_MILLIS);

        assertEquals(0, writer.size());
        List<UnresolvedMetricData> unresolvedList = resolveHelper.getUnresolvedMetricData();
        assertEquals(1, unresolvedList.size());

        UnresolvedMetricData unresolved = unresolvedList.get(0);
        assertEquals(labels, unresolved.getLabels());
        assertEquals(MetricType.COUNTER, unresolved.getType());
        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE, unresolved.getWriteType());

        AggrPoint delayedPoint = new AggrPoint();
        unresolved.get(0, delayedPoint);
        assertEquals(expectedPoint, delayedPoint);
    }

    @Test
    public void manyPointsUnresolvedMetricsDelayWrite() {
        Labels labels = Labels.of("sensor", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name());

        long ts0 = System.currentTimeMillis();
        processor.onMetricBegin(MetricType.COUNTER, labels, false);
        processor.onTimeSeries(TimeSeries.empty()
                .addLong(ts0, 1)
                .addLong(ts0 + 15_000, 2)
                .addLong(ts0 + 30_000, 3));

        assertEquals(0, writer.size());
        List<UnresolvedMetricData> unresolvedList = resolveHelper.getUnresolvedMetricData();
        assertEquals(1, unresolvedList.size());

        UnresolvedMetricData unresolved = unresolvedList.get(0);
        assertEquals(labels, unresolved.getLabels());
        assertEquals(MetricType.COUNTER, unresolved.getType());
        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE, unresolved.getWriteType());

        AggrGraphDataArrayList expectedTimeSeries = AggrGraphDataArrayList.of(
                AggrPoint.shortPoint(ts0, 1.0, 15_000),
                AggrPoint.shortPoint(ts0 + 15_000, 2.0, 15_000),
                AggrPoint.shortPoint(ts0 + 30_000, 3.0, 15_000)
        );

        AggrPoint delayedPoint = new AggrPoint();
        AggrGraphDataArrayList delayedPoints = new AggrGraphDataArrayList();
        for (int index = 0; index < 3; index++) {
            unresolved.get(index, delayedPoint);
            delayedPoints.addRecord(delayedPoint);
        }
        assertEquals(expectedTimeSeries, delayedPoints);
    }

    @Test
    public void onePointAggregate() {
        CoremonMetric aggregate = makeMetric(
                Labels.of("sensor", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name(), "host", "cluster"),
                ru.yandex.solomon.model.protobuf.MetricType.COUNTER);
        fileMetrics.put(aggregate);

        CoremonMetric metric = makeMetric(
                Labels.of("sensor", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name(), "host", "test"),
                ru.yandex.solomon.model.protobuf.MetricType.COUNTER);
        fileMetrics.put(metric);

        long now = System.currentTimeMillis();
        long value = 42;
        processor.onMetricBegin(metric.getType(), metric.getLabels(), true);
        processor.onPoint(now, value);

        AggrPoint expectedPoint = new AggrPoint();
        // aggregate always time alighted
        expectedPoint.setTsMillis(now - ((now % STEP_MILLIS)));
        expectedPoint.setValue(42, ValueColumn.DEFAULT_DENOM);
        expectedPoint.setStepMillis(STEP_MILLIS);
        expectedPoint.setMerge(true);
        expectedPoint.setCount(1);

        assertEquals(1, writer.size());
        MetricArchiveMutable archive = writer.getArchive(aggregate.getShardId(), aggregate.getLocalId());
        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE, archive.getType());
        assertEquals(1, archive.getRecordCount());
        assertEquals(expectedPoint, getFirstPoint(archive));
    }

    @Test
    public void onePointUnresolvedAggregateDelayWrite() {
        Labels labels = Labels.of("sensor", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name(), "host", "test");

        long now = System.currentTimeMillis();
        long value = 42;
        processor.onMetricBegin(MetricType.COUNTER, labels, true);
        processor.onPoint(now, value);

        AggrPoint expectedPoint = new AggrPoint();
        // aggregate always time alighted
        expectedPoint.setTsMillis(now - ((now % STEP_MILLIS)));
        expectedPoint.setValue(42, ValueColumn.DEFAULT_DENOM);
        expectedPoint.setMerge(true);
        expectedPoint.setStepMillis(STEP_MILLIS);
        expectedPoint.setCount(1);

        assertEquals(0, writer.size());

        var unresolvedList = List.copyOf(resolveHelper
                .getUnresolvedAggrMetricsDataByHost()
                .values());
        assertEquals(1, unresolvedList.size());

        UnresolvedMetricPoint unresolved = (UnresolvedMetricPoint) unresolvedList.get(0);

        assertEquals(labels.removeByKey("host").add("host", "cluster"), unresolved.getLabels());
        assertEquals(MetricType.COUNTER, unresolved.getType());
        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE, unresolved.getWriteType());
        assertEquals(expectedPoint, unresolved.point());
    }

    @Test
    public void manyPointUnresolvedAggregateDelayWrite() {
        Labels labels = Labels.of("sensor", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name(), "host", "test");

        long ts0 = System.currentTimeMillis();
        processor.onMetricBegin(MetricType.COUNTER, labels, true);
        processor.onTimeSeries(TimeSeries.empty()
                .addLong(ts0, 1)
                .addLong(ts0 + 15_000, 2)
                .addLong(ts0 + 30_000, 3));

        var unresolvedList = List.copyOf(resolveHelper
                .getUnresolvedAggrMetricsDataByHost()
                .values());
        assertEquals(1, unresolvedList.size());

        var unresolved = (UnresolvedMetricTimeSeries) unresolvedList.get(0);
        assertEquals(labels.removeByKey("host").add("host", "cluster"), unresolved.getLabels());
        assertEquals(MetricType.COUNTER, unresolved.getType());
        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE, unresolved.getWriteType());

        for (int index = 0; index < unresolved.pointsCount(); index++) {
            AggrPoint expectedPoint = new AggrPoint();
            expectedPoint.setTsMillis(ts0 - (ts0 % STEP_MILLIS) + (index * 15_000));
            expectedPoint.setValue(index + 1, ValueColumn.DEFAULT_DENOM);
            expectedPoint.setStepMillis(STEP_MILLIS);
            expectedPoint.setMerge(true);
            expectedPoint.setCount(1);

            AggrPoint delayedPoint = new AggrPoint();
            unresolved.get(index, delayedPoint);
            delayedPoint.setValue(delayedPoint.getValueDivided(), ValueColumn.DEFAULT_DENOM);
            assertEquals(expectedPoint, delayedPoint);
        }
    }

    @Test
    public void strictFailValidation() {
        CoremonMetric metric = makeMetric(
                Labels.of("senso$r", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name()),
                ru.yandex.solomon.model.protobuf.MetricType.COUNTER);
        boolean exception = false;
        try {
            processor.onMetricBegin(metric.getType(), metric.getLabels(), false);
            throw new RuntimeException("should throw exception with strict validation");
        } catch (UrlStatusTypeException e) {
            exception = true;
            Assert.assertEquals(UrlStatusType.PARSE_ERROR, e.urlStatusType());

        }
        Assert.assertTrue(exception);
    }

    @Test
    public void strictFailMemOnlyValidation() {
        this.processor = createProcessor(true, ValidationMode.STRICT_FAIL, STEP_MILLIS, false);
        CoremonMetric metric = makeMetric(
                Labels.of("senso$r", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name()),
                ru.yandex.solomon.model.protobuf.MetricType.COUNTER);
        boolean exception = false;
        try {
            processor.onMetricBegin(    metric.getType(),     metric.getLabels(), true);
            throw new RuntimeException("should throw exception with strict validation");
        } catch (UrlStatusTypeException e) {
            exception = true;
            Assert.assertEquals(UrlStatusType.PARSE_ERROR, e.urlStatusType());

        }
        Assert.assertTrue(exception);
    }

    @Test
    public void testStrictSkipValidation() {
        CoremonMetric invalidMetric = makeMetric(
                Labels.of("senso$r", MetricType.DGAUGE.name()),
                ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        CoremonMetric valid = makeMetric(
                Labels.of("sensor", MetricType.DGAUGE.name()),
                ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        processor = createProcessor(false, ValidationMode.STRICT_SKIP, STEP_MILLIS,false);
        processor.onMetricBegin(MetricType.DGAUGE, invalidMetric.getLabels(), false);
        processor.onPoint(0, 30);
        processor.onMetricBegin(MetricType.DGAUGE, valid.getLabels(), false);
        processor.onPoint(0, 15);
        Assert.assertEquals(1, resolveHelper.getUnresolvedMetricCount());
    }

    @Test
    public void testStrictSkipMemonlyValidation() {
        this.processor = createProcessor(true, ValidationMode.STRICT_SKIP, STEP_MILLIS, false);

        CoremonMetric invalidMetric = makeMetric(
                Labels.of("senso$r", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name()),
                ru.yandex.solomon.model.protobuf.MetricType.COUNTER);
        CoremonMetric valid = makeMetric(
                Labels.of("sensor", ru.yandex.solomon.model.protobuf.MetricType.COUNTER.name()),
                ru.yandex.solomon.model.protobuf.MetricType.COUNTER);

        processor.onMetricBegin(MetricType.RATE, invalidMetric.getLabels(), true);
        processor.onPoint(0, 30);
        processor.onMetricBegin(MetricType.RATE, valid.getLabels(), true);
        processor.onPoint(0, 15);
    }

    @Test
    public void roundGlobalTsMillis() {
        this.processor = createProcessor(false, ValidationMode.STRICT_FAIL, STEP_MILLIS, true);
        var alice = makeMetric("alice");
        fileMetrics.put(alice);

        processor.onMetricBegin(alice.getType(), alice.getLabels(), false);
        processor.onPoint(0, 1);

        var expected = AggrGraphDataArrayList.of(AggrPoint.builder()
                .time("2017-12-02T13:14:15.000Z")
                .doubleValue(1)
                .stepMillis(STEP_MILLIS)
                .build());

        assertEquals(expected, writtenPoints(alice));
    }

    @Test
    public void roundUnknownGlobalTsMillis() {
        this.processor = createProcessor(false, ValidationMode.STRICT_FAIL, STEP_MILLIS, true);
        var alice = makeMetric("alice");

        processor.onMetricBegin(alice.getType(), alice.getLabels(), false);
        processor.onPoint(0, 1);

        var expected = AggrGraphDataArrayList.of(AggrPoint.builder()
                .time("2017-12-02T13:14:15.000Z")
                .doubleValue(1)
                .stepMillis(STEP_MILLIS)
                .build());

        assertEquals(expected, unknownPoints(alice));
    }

    @Test
    public void roundProvidedTsMillis() {
        this.processor = createProcessor(false, ValidationMode.STRICT_FAIL, STEP_MILLIS,true);
        var bob = makeMetric("bob");
        fileMetrics.put(bob);

        processor.onMetricBegin(bob.getType(), bob.getLabels(), false);
        processor.onPoint(Instant.parse("2017-12-02T13:14:18.167Z").toEpochMilli(), 2);
        var expected = AggrGraphDataArrayList.of(AggrPoint.builder()
                .time("2017-12-02T13:14:15.000Z")
                .doubleValue(2)
                .stepMillis(STEP_MILLIS)
                .build());
        assertEquals(expected, writtenPoints(bob));
    }

    @Test
    public void roundUnknownProvidedTsMillis() {
        this.processor = createProcessor(false, ValidationMode.STRICT_FAIL, STEP_MILLIS, true);
        var bob = makeMetric("bob");

        processor.onMetricBegin(bob.getType(), bob.getLabels(), false);
        processor.onPoint(Instant.parse("2017-12-02T13:14:18.167Z").toEpochMilli(), 2);
        var expected = AggrGraphDataArrayList.of(AggrPoint.builder()
                .time("2017-12-02T13:14:15.000Z")
                .doubleValue(2)
                .stepMillis(STEP_MILLIS)
                .build());
        assertEquals(expected, unknownPoints(bob));
    }

    @Test
    public void roundLongTimeseriesMillis() {
        this.processor = createProcessor(false, ValidationMode.STRICT_FAIL, 5_000, true);
        var alice = makeMetric("alice");
        fileMetrics.put(alice);

        processor.onMetricBegin(alice.getType(), alice.getLabels(), false);
        processor.onTimeSeries(TimeSeries.newLong(3)
                .addLong(timeMillis("2020-05-22T10:59:31Z"), 1)
                .addLong(timeMillis("2020-05-22T10:59:36Z"), 2)
                .addLong(timeMillis("2020-05-22T10:59:40Z"), 3));

        var expected = AggrGraphDataArrayList.of(
                AggrPoint.builder()
                        .time("2020-05-22T10:59:30Z")
                        .doubleValue(1)
                        .stepMillis(5_000)
                        .build(),
                AggrPoint.builder()
                        .time("2020-05-22T10:59:35Z")
                        .doubleValue(2)
                        .stepMillis(5_000)
                        .build(),
                AggrPoint.builder()
                        .time("2020-05-22T10:59:40Z")
                        .doubleValue(3)
                        .stepMillis(5_000)
                        .build());

        assertEquals(expected, writtenPoints(alice));
    }

    @Test
    public void roundDoubleTimeseriesMillis() {

        this.processor = createProcessor(false, ValidationMode.STRICT_FAIL, 5_000, true);
        var alice = makeMetric("alice");
        fileMetrics.put(alice);

        processor.onMetricBegin(alice.getType(), alice.getLabels(), false);
        processor.onTimeSeries(TimeSeries.newDouble(3)
                .addDouble(timeMillis("2020-05-22T10:59:31Z"), 1)
                .addDouble(timeMillis("2020-05-22T10:59:36Z"), 2)
                .addDouble(timeMillis("2020-05-22T10:59:40Z"), 3));

        var expected = AggrGraphDataArrayList.of(
                AggrPoint.builder()
                        .time("2020-05-22T10:59:30Z")
                        .doubleValue(1)
                        .stepMillis(5_000)
                        .build(),
                AggrPoint.builder()
                        .time("2020-05-22T10:59:35Z")
                        .doubleValue(2)
                        .stepMillis(5_000)
                        .build(),
                AggrPoint.builder()
                        .time("2020-05-22T10:59:40Z")
                        .doubleValue(3)
                        .stepMillis(5_000)
                        .build());

        assertEquals(expected, writtenPoints(alice));
    }

    @Test
    public void roundUnknownTimeseriesMillis() {
        this.processor = createProcessor(false, ValidationMode.STRICT_FAIL, 5_000, true);
        var alice = makeMetric("alice");

        processor.onMetricBegin(alice.getType(), alice.getLabels(), false);
        processor.onTimeSeries(TimeSeries.newLong(3)
                .addLong(timeMillis("2020-05-22T10:59:31Z"), 1)
                .addLong(timeMillis("2020-05-22T10:59:36Z"), 2)
                .addLong(timeMillis("2020-05-22T10:59:40Z"), 3));

        var expected = AggrGraphDataArrayList.of(
                AggrPoint.builder()
                        .time("2020-05-22T10:59:30Z")
                        .doubleValue(1)
                        .stepMillis(5_000)
                        .build(),
                AggrPoint.builder()
                        .time("2020-05-22T10:59:35Z")
                        .doubleValue(2)
                        .stepMillis(5_000)
                        .build(),
                AggrPoint.builder()
                        .time("2020-05-22T10:59:40Z")
                        .doubleValue(3)
                        .stepMillis(5_000)
                        .build());

        assertEquals(expected, unknownPoints(alice));
    }

    @Test
    public void aggregatesByFetchIntervalNotDistanceBetweenRequest() {
        var tsPrev = timeMillis("2020-05-22T10:00:00Z");
        var tsSkip = timeMillis("2020-05-22T10:00:15Z");
        var tsNow = timeMillis("2020-05-22T10:00:30Z");
        var grid = 15_000L;

        this.processor = new MetricProcessorImpl(
                aggrHelper,
                resolveHelper,
                writer,
                memOnlyMetrics,
                Labels.of("DC", "man"),
                false,
                tsNow,
                tsPrev,
                () -> prevValues,
                grid,
                grid,
                EDecimPolicy.UNDEFINED,
                CoremonShardQuota.DEFAULT,
                false,
                TreeParser.ErrorListenerIgnore.I,
                ValidationMode.STRICT_FAIL,
                false);

        var aggregate = makeMetric(labels("host=cluster, metric=test"));
        fileMetrics.put(aggregate);

        var metric = makeMetric(labels("host=alice, metric=test"));
        fileMetrics.put(metric);

        processor.onMetricBegin(metric.getType(), metric.getLabels(), false);
        processor.onTimeSeries(TimeSeries.newDouble(2)
                .addDouble(tsSkip, 1)
                .addDouble(tsNow, 2));

        var expectByMetric =  AggrGraphDataArrayList.of(
                AggrPoint.builder()
                        .time(tsSkip)
                        .doubleValue(1)
                        .stepMillis(grid)
                        .build(),
                AggrPoint.builder()
                        .time(tsNow)
                        .doubleValue(2)
                        .stepMillis(grid)
                        .build());

        assertEquals(expectByMetric, writtenPoints(metric));

        var expectByAggregate = AggrGraphDataArrayList.of(
                AggrPoint.builder()
                        .time(tsSkip)
                        .doubleValue(1)
                        .stepMillis(grid)
                        .merge(true)
                        .count(1)
                        .build(),
                AggrPoint.builder()
                        .time(tsNow)
                        .doubleValue(2)
                        .stepMillis(grid)
                        .merge(true)
                        .count(1)
                        .build());

        var actualByAggregate = writtenPoints(aggregate);
        actualByAggregate.foldDenomIntoOne();
        assertEquals(expectByAggregate, actualByAggregate);
    }

    @Test
    public void aggregateTimeseriesByLast() {
        var aggregate = makeMetric(labels("host=cluster, name=object_count_last"));
        var metric = makeMetric(labels("host=alice, name=object_count_last"));
        fileMetrics.put(metric);

        var source = TimeSeries.empty()
                .addLong(timeMillis("2020-05-22T00:00:00Z"), 1)
                .addLong(timeMillis("2020-05-22T00:00:05Z"), 1)
                .addLong(timeMillis("2020-05-22T00:00:10Z"), 1)
                .addLong(timeMillis("2020-05-22T00:00:15Z"), 2)
                .addLong(timeMillis("2020-05-22T00:00:20Z"), 2)
                .addLong(timeMillis("2020-05-22T00:00:25Z"), 2);

        var expectedAggregate = AggrGraphDataArrayList.of(
                AggrPoints.dpoint("2020-05-22T00:00:00Z", 1d, false, 1, STEP_MILLIS),
                AggrPoints.dpoint("2020-05-22T00:00:15Z", 2d, false, 1, STEP_MILLIS));

        expectAggregateEqualTo(metric, source, aggregate, expectedAggregate);
    }

    @Test
    public void aggregateTimeseriesBySum() {
        var aggregate = makeMetric(labels("host=cluster, name=rps"));
        var metric = makeMetric(labels("host=alice, name=rps"));
        fileMetrics.put(metric);

        var source = TimeSeries.empty()
                .addLong(timeMillis("2020-05-22T00:00:00Z"), 3)
                .addLong(timeMillis("2020-05-22T00:00:05Z"), 6)
                .addLong(timeMillis("2020-05-22T00:00:10Z"), 9)
                .addLong(timeMillis("2020-05-22T00:00:15Z"), 19)
                .addLong(timeMillis("2020-05-22T00:00:20Z"), 13)
                .addLong(timeMillis("2020-05-22T00:00:25Z"), 16);

        var expectedAggregate = AggrGraphDataArrayList.of(
                AggrPoints.dpoint("2020-05-22T00:00:00Z", 6d, true, 1, STEP_MILLIS),
                AggrPoints.dpoint("2020-05-22T00:00:15Z", 16d, true, 1, STEP_MILLIS));

        expectAggregateEqualTo(metric, source, aggregate, expectedAggregate);
    }

    @Test
    public void aggregateTimeseriesByLastWithGaps() {
        var aggregate = makeMetric(labels("host=cluster, name=object_count_last"));
        var metric = makeMetric(labels("host=alice, name=object_count_last"));
        fileMetrics.put(metric);

        var source = TimeSeries.empty()
                .addLong(timeMillis("2020-05-22T00:00:00Z"), 1)
                //.addLong(timeMillis("2020-05-22T00:00:15Z"), 2) gap
                .addLong(timeMillis("2020-05-22T00:00:30Z"), 3)
                //.addLong(timeMillis("2020-05-22T00:00:45Z"), 4) gap
                //.addLong(timeMillis("2020-05-22T00:00:00Z"), 5) gap
                .addLong(timeMillis("2020-05-22T00:01:15Z"), 6);

        var expectedAggregate = AggrGraphDataArrayList.of(
                AggrPoints.dpoint("2020-05-22T00:00:00Z", 1d, false, 1, STEP_MILLIS),
                AggrPoints.dpoint("2020-05-22T00:00:30Z", 3d, false, 1, STEP_MILLIS),
                AggrPoints.dpoint("2020-05-22T00:01:15Z", 6d, false, 1, STEP_MILLIS));

        expectAggregateEqualTo(metric, source, aggregate, expectedAggregate);
    }

    @Test
    public void aggregateTimeseriesBySumWithGaps() {
        var aggregate = makeMetric(labels("host=cluster, name=rps"));
        var metric = makeMetric(labels("host=alice, name=rps"));
        fileMetrics.put(metric);

        var source = TimeSeries.empty()
                .addLong(timeMillis("2020-05-22T00:00:00Z"), 1)
                //.addLong(timeMillis("2020-05-22T00:00:15Z"), 1) gap
                .addLong(timeMillis("2020-05-22T00:00:30Z"), 2)
                //.addLong(timeMillis("2020-05-22T00:00:45Z"), 2) gap
                //.addLong(timeMillis("2020-05-22T00:00:00Z"), 3) gap
                .addLong(timeMillis("2020-05-22T00:01:15Z"), 3);

        var expectedAggregate = AggrGraphDataArrayList.of(
                AggrPoints.dpoint("2020-05-22T00:00:00Z", 1d, true, 1, STEP_MILLIS),
                AggrPoints.dpoint("2020-05-22T00:00:30Z", 2d, true, 1, STEP_MILLIS),
                AggrPoints.dpoint("2020-05-22T00:01:15Z", 3d, true, 1, STEP_MILLIS));

        expectAggregateEqualTo(metric, source, aggregate, expectedAggregate);
    }

    @Test
    public void timeseriesWithAllNan() {
        var metric = makeMetric(labels("host=alice, name=rps"));
        fileMetrics.put(metric);
        var aggregate = makeMetric(labels("host=cluster, name=rps"));
        fileMetrics.put(aggregate);

        var source = TimeSeries.empty()
                .addDouble(timeMillis("2020-05-22T00:00:00Z"), Double.NaN)
                .addDouble(timeMillis("2020-05-22T00:00:15Z"), Double.NaN)
                .addDouble(timeMillis("2020-05-22T00:00:30Z"), Double.NaN);

        processor = createProcessor(false, ValidationMode.STRICT_FAIL, STEP_MILLIS, false);
        processor.onMetricBegin(metric.getType(), metric.getLabels(), false);
        processor.onTimeSeries(source);

        assertNull(writtenPoints(aggregate));
        assertNull(writtenPoints(metric));
    }

    private void expectAggregateEqualTo(CoremonMetric metric, TimeSeries source, CoremonMetric aggregate, AggrGraphDataArrayList expectedAggregate) {
        // unresolved
        {
            processor = createProcessor(false, ValidationMode.STRICT_FAIL, STEP_MILLIS, false);
            processor.onMetricBegin(metric.getType(), metric.getLabels(), false);
            processor.onTimeSeries(source);

            var actualByAggregate = unknownPoints(aggregate);
            actualByAggregate.foldDenomIntoOne();
            assertEquals(expectedAggregate, actualByAggregate);
        }

        // resolved
        fileMetrics.put(aggregate);
        writer.clear();
        {
            processor = createProcessor(false, ValidationMode.STRICT_FAIL, STEP_MILLIS, false);
            processor.onMetricBegin(metric.getType(), metric.getLabels(), false);
            processor.onTimeSeries(source);

            var actualByAggregate = writtenPoints(aggregate);
            actualByAggregate.foldDenomIntoOne();
            assertEquals(expectedAggregate, actualByAggregate);
        }

        // resolved cached
        writer.clear();
        {
            processor = createProcessor(false, ValidationMode.STRICT_FAIL, STEP_MILLIS, false);
            processor.onMetricBegin(metric.getType(), metric.getLabels(), false);
            processor.onTimeSeries(source);

            var actualByAggregate = writtenPoints(aggregate);
            actualByAggregate.foldDenomIntoOne();
            assertEquals(expectedAggregate, actualByAggregate);
        }
    }

    private MetricProcessorImpl createProcessor(boolean memOnly, ValidationMode validationMode, long gridMillis, boolean gridRounding) {
        return new MetricProcessorImpl(
            aggrHelper,
            resolveHelper,
            writer,
                memOnlyMetrics,
            Labels.of("DC", "man"),
            false,
            RESPONSE_TS_MILLIS,
            RESPONSE_TS_MILLIS - STEP_MILLIS,
            () -> prevValues,
            gridMillis,
            STEP_MILLIS,
            EDecimPolicy.UNDEFINED,
            CoremonShardQuota.DEFAULT,
            memOnly,
            TreeParser.ErrorListenerIgnore.I,
            validationMode,
            gridRounding);
    }

    private AggrPoint getFirstPoint(MetricArchiveMutable archive) {
        AggrPoint point = new AggrPoint();
        assertTrue(archive.iterator().next(point));
        return point;
    }

    private CoremonMetric makeMetric(Labels labels, ru.yandex.solomon.model.protobuf.MetricType type) {
        int shardId = StockpileShardId.random();
        long localId = StockpileLocalId.random();
        return new FileCoremonMetric(shardId, localId, labels, requireNonNull(MetricTypeConverter.fromProto(type)));
    }


    private CoremonMetric makeMetric(Labels labels) {
        return makeMetric(labels, ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
    }

    private CoremonMetric makeMetric(String name) {
        int shardId = StockpileShardId.random();
        long localId = StockpileLocalId.random();
        var labels = Labels.of("name", name);
        return new FileCoremonMetric(shardId, localId, labels, MetricType.DGAUGE);
    }

    private AggrGraphDataArrayList writtenPoints(CoremonMetric metric) {
        return writer.getPoints(metric.getShardId(), metric.getLocalId());
    }

    private AggrGraphDataArrayList unknownPoints(CoremonMetric metric) {
        AggrGraphDataArrayList result = new AggrGraphDataArrayList();
        var point = RecyclableAggrPoint.newInstance();

        var it = Iterators.concat(
                resolveHelper.getUnresolvedMetricData().iterator(),
                resolveHelper.getUnresolvedAggrMetricsDataByHost().values().iterator());

        while (it.hasNext()) {
            var unresolved = it.next();
            if (!metric.getLabels().equals(unresolved.getLabels())) {
                continue;
            }

            for (int index = 0; index < unresolved.pointsCount(); index++) {
                unresolved.get(index, point);
                result.addRecord(point);
            }
        }
        return result;
    }

    private static long timeMillis(String time) {
        return Instant.parse(time).toEpochMilli();
    }

    private static Labels labels(String str) {
        return LabelsFormat.parse(str);
    }
}
