package ru.yandex.solomon.coremon.stockpile;

import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.actor.Tasks;
import ru.yandex.misc.test.ForTestSynchronizationUtility;
import ru.yandex.monlib.metrics.meter.ExpMovingAverage;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author Stepan Koltsov
 */
public class ParsingActorRunnerTest {

    private ExecutorService executor;

    @Before
    public void before() {
        ThreadFactory threadFactory = ThreadUtils.newThreadFactory(ParsingActorRunnerTest.class);
        executor = Executors.newFixedThreadPool(4, threadFactory);
    }

    @After
    public void after() throws Exception {
        executor.shutdown();
        executor.awaitTermination(9999999, TimeUnit.DAYS);
    }


    @Test
    public void withAsyncFuture() {
        ForTestSynchronizationUtility s = new ForTestSynchronizationUtility();
        AtomicInteger invocation = new AtomicInteger();

        var actorRunner = new ParsingActorRunner(1, () -> {
            CompletableFuture<Void> f = new CompletableFuture<>();

            int i = invocation.getAndIncrement();
            s.take(i * 3 + 1);
            executor.execute(() -> {
                s.take(i * 3 + 2);
                f.complete(null);
            });

            return f;
        }, executor, cpuTimeNanos());

        for (int j = 0; j < 1000; ++j) {
            s.take(j * 3);
            actorRunner.schedule();
        }
        s.take(3000);
    }

    @Test
    public void withSyncFuture() {
        ForTestSynchronizationUtility s = new ForTestSynchronizationUtility();
        AtomicInteger invocation = new AtomicInteger();

        var actorRunner = new ParsingActorRunner(1, () -> {
            int i = invocation.getAndIncrement();
            s.take(i * 3 + 1);
            s.take(i * 3 + 2);

            return CompletableFuture.completedFuture(null);
        }, executor, cpuTimeNanos());

        for (int j = 0; j < 1000; ++j) {
            s.take(j * 3);
            actorRunner.schedule();
        }
        s.take(3000);
    }

    @Test
    public void multipleFutures() {
        ForTestSynchronizationUtility s = new ForTestSynchronizationUtility();
        AtomicInteger invocation = new AtomicInteger();

        ArrayList<CompletableFuture<Void>> futures = new ArrayList<>();

        var actorRunner = new ParsingActorRunner(5, () -> {
            s.take(invocation.getAndIncrement() * 2 + 1);
            CompletableFuture<Void> future = new CompletableFuture<>();
            futures.add(future);
            return future;
        }, executor, cpuTimeNanos());

        for (int i = 0; i < 5; ++i) {
            s.take(i * 2);
            actorRunner.schedule();
        }

        s.take(10);

        Assert.assertEquals(Tasks.State.RUNNING_NO_TASKS, actorRunner.tasks.getState());

        actorRunner.schedule();

        Assert.assertEquals(Tasks.State.RUNNING_GOT_TASKS, actorRunner.tasks.getState());

        Assert.assertEquals(5, invocation.get());

        futures.remove(0).complete(null);

        s.take(12);
    }

    private static Meter cpuTimeNanos() {
        return Meter.of(ExpMovingAverage.fiveMinutes());
    }
}
