package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;
import com.google.protobuf.Any;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.coremon.DeleteMetricsConfig;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryDeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardConf;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolverStub;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.locks.dao.memory.InMemoryLocksDao;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.ExecutionContextStub;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Complete;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Fail;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Reschedule;
import ru.yandex.solomon.scheduler.Permit;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static java.util.concurrent.CompletableFuture.delayedExecutor;
import static java.util.concurrent.CompletableFuture.supplyAsync;
import static java.util.concurrent.TimeUnit.HOURS;
import static java.util.concurrent.TimeUnit.MINUTES;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.lessThanOrEqualTo;
import static org.hamcrest.core.AllOf.allOf;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsMoveTask.toLockId;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.matchingMetric;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.metabaseShardConf;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.shardsWithNumIdUpTo;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.solomon.util.time.InstantUtils.currentTimeSeconds;

/**
 * @author Stanislav Kashirin
 */
public class AbstractDeleteMetricsTaskHandlerTest {

    private static final int MAX_NUM_ID = 3;
    private static final List<MetabaseShardConf> SHARDS = shardsWithNumIdUpTo(MAX_NUM_ID);

    @Rule
    public Timeout globalTimeout = Timeout.builder()
        .withTimeout(1, MINUTES)
        .withLookingForStuckThread(true)
        .build();

    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private LocksDao locksDao;
    private SolomonConfHolder confHolder;
    private StockpileClientStub stockpileClient;
    private InMemoryMetricsDaoFactory metricsDaoFactory;
    private MetabaseShardResolverStub shardResolver;
    private InMemoryDeletedMetricsDao deletedMetricsDao;
    private MetricRegistry registry;
    private DeleteMetricsTaskMetrics metrics;

    private AbstractDeleteMetricsTaskHandler<?> handler;

    @Before
    public void setUp() {
        confHolder = new SolomonConfHolder();

        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);

        stockpileClient = new StockpileClientStub(ForkJoinPool.commonPool());

        metricsDaoFactory = new InMemoryMetricsDaoFactory();
        metricsDaoFactory.setSuspendShardInitOnCreate(true);

        shardResolver = new MetabaseShardResolverStub(
            SHARDS,
            metricsDaoFactory,
            stockpileClient
        );

        locksDao = new InMemoryLocksDao(clock);

        deletedMetricsDao = new InMemoryDeletedMetricsDao();

        registry = new MetricRegistry();
        metrics = new DeleteMetricsTaskMetrics(
            registry,
            DeleteMetricsConfig.newBuilder().setReportVerboseMetrics(true).build());

        installRandomHandler();
    }

    @After
    public void tearDown() {
        if (timer != null) {
            timer.shutdownNow();
        }
        close(metricsDaoFactory, shardResolver);
    }

    @Test
    public void absentPermitWhenConfNotLoaded() {
        // arrange
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);
        ensureShardsReady();

        var params = params();

        // act
        var permit = acquire(params);

        // assert
        assertNull(permit);
    }

    @Test
    public void absentPermitWhenStockpileNotAvailable() {
        // arrange
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.UNAVAILABLE);
        shardResolver.setInitCompleted(true);
        ensureShardsReady();

        var params = params();

        // act
        var permit = acquire(params);

        // assert
        assertNull(permit);
    }

    @Test
    public void absentPermitWhenShardResolverNotInit() {
        // arrange
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(false);
        ensureShardsReady();

        var params = params();

        // act
        var permit = acquire(params);

        // assert
        assertNull(permit);
    }

    @Test
    public void absentPermitWhenShardIsNotLocal() {
        // arrange
        var numId = 1337;
        initConfHolder(
            ImmutableList.<MetabaseShardConf>builder()
                .addAll(SHARDS)
                .add(metabaseShardConf(numId))
                .build());
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);
        ensureShardsReady();

        var params = paramsWithNumId(numId);

        // act
        var permit = acquire(params);

        // assert
        assertNull(permit);
    }

    @Test
    public void absentPermitWhenShardIsNotLoaded() {
        // arrange
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var params = params();
        ensureShardsReadyExcept(params.getNumId());

        // act
        var permit = acquire(params);

        // assert
        assertNull(permit);
    }

    @Test
    public void absentPermitWhenShardIsInFlight() {
        // arrange
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var paramsInFlight = params();
        var params = paramsWithNumId(paramsInFlight.getNumId());

        ensureMetricInDao(paramsInFlight.getNumId(), matchingMetric(Selectors.parse(paramsInFlight.getSelectors())));
        ensureShardsReady();

        stockpileClient.beforeSupplier = () -> supplyAsync(() -> null, delayedExecutor(1, TimeUnit.HOURS));
        execute(context(paramsInFlight));

        // act
        var permit = acquire(params);

        // assert
        assertNull(permit);
    }

    @Test
    public void receivePermitWhenShardDoesNotExist() {
        // arrange
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);
        ensureShardsReady();

        var params = paramsWithNumId(666);

        // act
        var permit = acquire(params);

        // assert
        assertNotNull(permit);
    }

    @Test
    public void receivePermitWhenShardIsLocal() {
        // arrange
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var params = params();
        ensureShardReady(params.getNumId());

        // act
        var permit = acquire(params);

        // assert
        assertNotNull(permit);
    }

    @Test
    public void execRescheduleWhenShardIsInFlight() {
        // arrange
        installMoveHandler();
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var paramsInFlight = params();
        var params = paramsWithNumId(paramsInFlight.getNumId());
        var context = context(params);

        ensureMetricInDao(paramsInFlight.getNumId(), matchingMetric(Selectors.parse(paramsInFlight.getSelectors())));
        ensureShardsReady();

        stockpileClient.beforeSupplier = () -> supplyAsync(() -> null, delayedExecutor(1, TimeUnit.HOURS));
        execute(context(paramsInFlight));

        // act
        execute(context);

        // assert
        var reschedule = context.takeDoneEvent(Reschedule.class);
        var delay = reschedule.executeAt() - System.currentTimeMillis();
        assertThat(delay, allOf(
            lessThanOrEqualTo(MINUTES.toMillis(10)),
            greaterThanOrEqualTo(0L)));
    }

    @Test
    public void execCheckCompleteWhenExecuted() {
        // arrange
        installCheckHandler();
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var params = params();
        var context = context(params);

        var metric = matchingMetric(
            Selectors.parse(params.getSelectors()),
            currentTimeSeconds() - (int) HOURS.toSeconds(2));
        ensureMetricInDao(params.getNumId(), metric);
        ensureShardsReady();

        shardResolver.setLastPointSeconds(
            params.getNumId(),
            Map.of(metric.getLabels(), metric.getLastPointSeconds()));

        // act
        execute(context);

        // assert
        var complete = context.takeDoneEvent(Complete.class);

        var result = DeleteMetricsTaskProto.checkResult(complete.result());
        assertEquals(metric.getLastPointSeconds(), result.getMinLastPointSeconds());
    }

    @Test
    public void execMoveCompleteWhenExecuted() {
        // arrange
        installMoveHandler();
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var params = params();
        var context = context(params);

        ensureMetricInDao(params.getNumId(), matchingMetric(Selectors.parse(params.getSelectors())));
        ensureShardsReady();

        // act
        execute(context);

        // assert
        var complete = context.takeDoneEvent(Complete.class);

        var result = DeleteMetricsTaskProto.moveResult(complete.result());
        assertEquals(1, result.getMovedMetrics());
    }

    @Test
    public void execRollbackCompleteWhenExecuted() {
        // arrange
        installRollbackHandler();
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var params = params();
        var context = context(params);

        ensureDeletedMetricInDao(
            params.getOperationId(),
            params.getNumId(),
            matchingMetric(Selectors.parse(params.getSelectors())));
        ensureShardsReady();

        // act
        execute(context);

        // assert
        var complete = context.takeDoneEvent(Complete.class);

        var result = DeleteMetricsTaskProto.rollbackResult(complete.result());
        assertEquals(0, result.getStillDeletedMetrics());
    }

    @Test
    public void execTerminateCompleteWhenExecuted() {
        // arrange
        installTerminateHandler();
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var params = params();
        var context = context(params);

        ensureDeletedMetricInDao(
            params.getOperationId(),
            params.getNumId(),
            matchingMetric(Selectors.parse(params.getSelectors())));
        ensureShardsReady();

        // act
        execute(context);

        // assert
        var complete = context.takeDoneEvent(Complete.class);

        var result = DeleteMetricsTaskProto.terminateResult(complete.result());
        assertEquals(1, result.getDeletedMetrics());
    }

    @Test
    public void execCompleteWhenExecutedWithLockReentrancy() {
        // arrange
        installMoveHandler();
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var params = params();
        var context = context(params);

        ensureMetricInDao(params.getNumId(), matchingMetric(Selectors.parse(params.getSelectors())));
        ensureShardsReady();

        var expiredAt = Instant.now().plus(Duration.ofDays(42));
        locksDao.acquireLock(toLockId(params.getNumId()), params.getOperationId(), expiredAt).join();

        // act
        execute(context);

        // assert
        var complete = context.takeDoneEvent(Complete.class);

        var result = DeleteMetricsTaskProto.moveResult(complete.result());
        assertEquals(1, result.getMovedMetrics());
    }

    @Test
    public void execRescheduleWhenExecutedWithLockHeldByOtherOperation() {
        // arrange
        installMoveHandler();
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var params = params();
        var context = context(params);

        ensureMetricInDao(params.getNumId(), matchingMetric(Selectors.parse(params.getSelectors())));
        ensureShardsReady();

        var expiredAt = Instant.now().plus(Duration.ofDays(42));
        locksDao.acquireLock(toLockId(params.getNumId()), UUID.randomUUID().toString(), expiredAt).join();

        // act
        execute(context);

        // assert
        var reschedule = context.takeDoneEvent(Reschedule.class);
        var delay = reschedule.executeAt() - System.currentTimeMillis();
        assertThat(delay, allOf(
            lessThanOrEqualTo(MINUTES.toMillis(10)),
            greaterThanOrEqualTo(0L)));
    }

    @Test
    public void newTaskCanBeStartedOnShardAfterInFlightTaskComplete() throws Exception {
        // arrange
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);
        ensureShardsReady();

        var paramsInFlight = params();
        var contextInFlight = context(paramsInFlight);

        var newTaskParams = paramsWithNumId(paramsInFlight.getNumId());
        var newTaskContext = context(newTaskParams);

        var permitInFlight = acquire(paramsInFlight);
        execute(contextInFlight);
        contextInFlight.takeDoneEvent(Complete.class);
        permitInFlight.release();

        // act
        do {
            TimeUnit.MILLISECONDS.sleep(1);
        } while (acquire(newTaskParams) == null);
        execute(newTaskContext);

        // assert
        var complete = newTaskContext.takeDoneEvent(Complete.class);
        assertNotNull(complete.result());
    }

    @Test
    public void newTaskCanBeStartedOnShardAfterInFlightTaskFail() throws Exception {
        // arrange
        installCheckHandler();
        initConfHolder();
        stockpileClient.setAvailability(AvailabilityStatus.AVAILABLE);
        shardResolver.setInitCompleted(true);

        var paramsInFlight = params();
        var contextInFlight = context(paramsInFlight);

        var newTaskParams = paramsWithNumId(paramsInFlight.getNumId());
        var newTaskContext = context(newTaskParams);

        var activeMetric = matchingMetric(Selectors.parse(paramsInFlight.getSelectors()));
        ensureMetricInDao(paramsInFlight.getNumId(), activeMetric);
        var oldMetric = matchingMetric(Selectors.parse(newTaskParams.getSelectors()));
        ensureMetricInDao(newTaskParams.getNumId(), oldMetric);
        ensureShardsReady();

        var now = (int) Instant.now().getEpochSecond();
        var oldLastPointSeconds = now - (int) HOURS.toSeconds(3);
        shardResolver.setLastPointSeconds(
            paramsInFlight.getNumId(),
            Map.of(
                activeMetric.getLabels(), now,
                oldMetric.getLabels(), oldLastPointSeconds));

        var permitInFlight = acquire(paramsInFlight);
        execute(contextInFlight);
        contextInFlight.takeDoneEvent(Fail.class);
        permitInFlight.release();

        // act
        do {
            TimeUnit.MILLISECONDS.sleep(1);
        } while (acquire(newTaskParams) == null);
        execute(newTaskContext);

        // assert
        var complete = newTaskContext.takeDoneEvent(Complete.class);
        assertNotNull(complete.result());

        var result = DeleteMetricsTaskProto.checkResult(complete.result());
        assertEquals(oldLastPointSeconds, result.getMinLastPointSeconds());
    }

    private void installRandomHandler() {
        var i = random().nextInt(4);
        switch (i) {
            case 0 -> installCheckHandler();
            case 1 -> installMoveHandler();
            case 2 -> installRollbackHandler();
            case 3 -> installTerminateHandler();
        }
    }

    private void installCheckHandler() {
        handler = new DeleteMetricsCheckTaskHandler(
            metrics,
            confHolder,
            stockpileClient,
            shardResolver,
            ForkJoinPool.commonPool(),
            timer);
    }

    private void installMoveHandler() {
        handler = new DeleteMetricsMoveTaskHandler(
            metrics,
            confHolder,
            stockpileClient,
            shardResolver,
            locksDao,
            deletedMetricsDao,
            metricsDaoFactory,
            ForkJoinPool.commonPool(),
            timer);
    }

    private void installRollbackHandler() {
        handler = new DeleteMetricsRollbackTaskHandler(
            metrics,
            confHolder,
            stockpileClient,
            shardResolver,
            deletedMetricsDao,
            ForkJoinPool.commonPool(),
            timer);
    }

    private void installTerminateHandler() {
        handler = new DeleteMetricsTerminateTaskHandler(
            metrics,
            confHolder,
            stockpileClient,
            shardResolver,
            deletedMetricsDao,
            ForkJoinPool.commonPool(),
            timer);
    }

    private void initConfHolder() {
        initConfHolder(SHARDS);
    }

    private void initConfHolder(Collection<MetabaseShardConf> shardConfs) {
        var projects = new ArrayList<Project>();
        var clusters = new ArrayList<Cluster>();
        var services = new ArrayList<Service>();
        var shards = new ArrayList<Shard>();

        for (var shardConf : shardConfs) {
            var project = shardConf.getShardKey().getProject();
            var cluster = shardConf.getShardKey().getCluster();
            var service = shardConf.getShardKey().getService();

            projects.add(
                Project.newBuilder()
                    .setId(project)
                    .setName(project)
                    .setOwner("any")
                    .build());
            clusters.add(
                Cluster.newBuilder()
                    .setProjectId(project)
                    .setId(cluster)
                    .setName(cluster)
                    .build());
            services.add(
                Service.newBuilder()
                    .setProjectId(project)
                    .setId(service)
                    .setName(service)
                    .build());
            shards.add(
                Shard.newBuilder()
                    .setProjectId(project)
                    .setId(project + "_" + cluster + "_" + service)
                    .setNumId(shardConf.getNumId())
                    .setClusterId(cluster)
                    .setClusterName(cluster)
                    .setServiceId(service)
                    .setServiceName(service)
                    .build());
        }

        var conf = new SolomonRawConf(List.of(), projects, clusters, services, shards);
        confHolder.onConfigurationLoad(SolomonConfWithContext.create(conf));
    }

    private void ensureShardsReady() {
        IntStream.rangeClosed(1, MAX_NUM_ID)
            .forEach(numId -> metricsDaoFactory.resumeShardInit(numId));

        shardResolver.awaitShardsReady();
    }

    private void ensureShardsReadyExcept(int except) {
        IntStream.rangeClosed(1, MAX_NUM_ID)
            .filter(numId -> numId != except)
            .forEach(numId -> metricsDaoFactory.resumeShardInit(numId));

        IntStream.rangeClosed(1, MAX_NUM_ID)
            .filter(numId -> numId != except)
            .forEach(numId -> shardResolver.resolveShard(numId).awaitReady());
    }

    private void ensureShardReady(int numId) {
        metricsDaoFactory.resumeShardInit(numId);
        shardResolver.resolveShard(numId).awaitReady();
    }

    private void ensureMetricInDao(int numId, CoremonMetric metric) {
        var dao = metricsDaoFactory.create(numId, Labels.allocator);
        dao.add(List.of(metric));
    }

    private void ensureDeletedMetricInDao(String operationId, int numId, CoremonMetric metric) {
        deletedMetricsDao.putAll(operationId, numId, List.of(metric));
    }

    @Nullable
    private Permit acquire(DeleteMetricsParams params) {
        return handler.acquire(id(params), Any.pack(params));
    }

    private void execute(ExecutionContext context) {
        handler.execute(context);
    }

    private static DeleteMetricsParams params() {
        return DeleteMetricsRandom.params(MAX_NUM_ID);
    }

    private static DeleteMetricsParams paramsWithNumId(int numId) {
        return DeleteMetricsRandom.params(MAX_NUM_ID).toBuilder()
            .setNumId(numId)
            .build();
    }

    private static String id(DeleteMetricsParams params) {
        return params.getOperationId() + "_" + Integer.toUnsignedString(params.getNumId());
    }

    private static ExecutionContextStub context(DeleteMetricsParams params) {
        var task = Task.newBuilder()
            .setId(id(params))
            .setType("delete_metrics")
            .setExecuteAt(System.currentTimeMillis())
            .setParams(Any.pack(params))
            .build();

        return new ExecutionContextStub(task);
    }

    private static Random random() {
        return ThreadLocalRandom.current();
    }
}
