package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Streams;
import io.grpc.Status;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.DeleteMetricsCheckProgress.CheckNoRecentWritesProgress;
import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardConf;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolverStub;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metrics.client.AbstractStockpileClientStub;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricMeta;
import ru.yandex.stockpile.api.ReadMetricsMetaRequest;
import ru.yandex.stockpile.api.ReadMetricsMetaResponse;

import static java.util.Collections.shuffle;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.delayedExecutor;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static java.util.concurrent.ForkJoinPool.commonPool;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static ru.yandex.solomon.coremon.meta.CoremonMetric.UNKNOWN_LAST_POINT_SECONDS;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.formattedLabels;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.matchingMetric;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.matchingMetricInSpShard;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.matchingMetricWithCreatedAt;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.metric;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.shardsWithNumIdUpTo;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
public class CheckNoRecentWritesTest {

    private static final int TEST_MAX_BATCH_SIZE = 100;

    private static final int MAX_NUM_ID = 3;
    private static final List<MetabaseShardConf> SHARDS = shardsWithNumIdUpTo(MAX_NUM_ID);

    private static final int nowSeconds = (int) Instant.now().getEpochSecond();

    @Rule
    public Timeout globalTimeout = Timeout.builder()
        .withTimeout(1, MINUTES)
        .withLookingForStuckThread(true)
        .build();

    private RetryConfig retryConfig;
    private MetaOnlyStockpileClientStub stockpileClient;
    private InMemoryMetricsDaoFactory metricsDaoFactory;
    private MetabaseShardResolverStub shardResolver;

    @Before
    public void setUp() {
        retryConfig = RetryConfig.DEFAULT
            .withNumRetries(Integer.MAX_VALUE)
            .withMaxDelay(0);

        stockpileClient = new MetaOnlyStockpileClientStub();

        metricsDaoFactory = new InMemoryMetricsDaoFactory();
        metricsDaoFactory.setSuspendShardInitOnCreate(true);

        shardResolver = new MetabaseShardResolverStub(
            SHARDS,
            metricsDaoFactory,
            stockpileClient);
    }

    @After
    public void tearDown() {
        close(metricsDaoFactory, shardResolver);
    }

    @Test
    public void alreadyCompleted() {
        // arrange
        stockpileClient.beforeSupplier = unavailable();

        var ok = random().nextBoolean();
        var progress = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels(ok ? "" : formattedLabels())
            .setMinLastPointSeconds(ok ? 42 : 0)
            .build();
        var proc = checkNoRecentWrites(params(), progress);

        // act
        proc.start().join();

        // assert
        assertEquals(progress, proc.progress());
    }

    @Test
    public void shardIsNotLocalAnymore() {
        // arrange
        stockpileClient.beforeSupplier = unavailable();

        var params = params().toBuilder().setNumId(666).build();
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        // act
        proc.start().join();

        // assert
        assertEquals(CheckNoRecentWritesProgress.getDefaultInstance(), proc.progress());
    }

    @Test
    public void shardIsNotReady() {
        // arrange
        stockpileClient.beforeSupplier = unavailable();

        var params = params();
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        // act
        proc.start().join();

        // assert
        assertEquals(CheckNoRecentWritesProgress.getDefaultInstance(), proc.progress());
    }

    @Test
    public void noMetricsFound() {
        // arrange
        stockpileClient.beforeSupplier = unavailable();

        var params = params();
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels("")
            .setMinLastPointSeconds(Integer.MAX_VALUE)
            .build();
        assertEquals(expected, proc.progress());
    }

    @Test
    public void noMetricsFoundSomeCreatedAfterLaunch() {
        // arrange
        stockpileClient.beforeSupplier = unavailable();

        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var createdAfter = millisecondsToSeconds(params.getCreatedAt()) + 1;
        var irrelevantMetricsCreatedAfter = Stream.generate(() -> matchingMetricWithCreatedAt(selectors, createdAfter))
            .limit(random().nextInt(1500, 2000))
            .collect(toList());

        ensureMetricsInDao(params.getNumId(), irrelevantMetricsCreatedAfter);
        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels("")
            .setMinLastPointSeconds(Integer.MAX_VALUE)
            .build();
        assertEquals(expected, proc.progress());
    }

    @Test
    public void completeWithEmptyAllLastPointsKnownLocally() {
        // arrange
        stockpileClient.beforeSupplier = unavailable();

        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors, aLongTimeAgoSeconds()))
            .limit(random().nextInt(10, 20))
            .collect(toList());

        var irrelevantMetrics = Stream.generate(() -> metric(nowSeconds))
            .limit(random().nextInt(10, 20));

        var metrics = Streams.concat(relevantMetrics.stream(), irrelevantMetrics)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);

        // act
        proc.start().join();

        // assert
        var min = relevantMetrics.stream()
            .mapToInt(CoremonMetric::getLastPointSeconds)
            .min()
            .orElseThrow();

        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels("")
            .setMinLastPointSeconds(min)
            .build();
        assertEquals(expected, proc.progress());
    }

    @Test
    public void completeWithEmptyAllLastPointsKnownLocallyAndSomeCreatedAfterLaunch() {
        // arrange
        stockpileClient.beforeSupplier = unavailable();

        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors, aLongTimeAgoSeconds()))
            .limit(random().nextInt(10, 20))
            .collect(toList());

        var irrelevantMetricsDiffLabels = Stream.generate(() -> metric(nowSeconds))
            .limit(random().nextInt(10, 20));

        var createdAfter = millisecondsToSeconds(params.getCreatedAt()) + 1;
        var irrelevantMetricsCreatedAfter =
            Stream.generate(() -> matchingMetricWithCreatedAt(selectors, createdAfter, nowSeconds))
                .limit(random().nextInt(10, 20));

        var metrics = Streams.concat(
                relevantMetrics.stream(),
                irrelevantMetricsDiffLabels,
                irrelevantMetricsCreatedAfter)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);

        // act
        proc.start().join();

        // assert
        var min = relevantMetrics.stream()
            .mapToInt(CoremonMetric::getLastPointSeconds)
            .min()
            .orElseThrow();

        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels("")
            .setMinLastPointSeconds(min)
            .build();
        assertEquals(expected, proc.progress());
    }

    @Test
    public void completeWithActiveMetricFoundAllLastPointsKnownLocally() {
        // arrange
        stockpileClient.beforeSupplier = unavailable();

        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var notUsedRelevantMetrics = Stream.generate(() -> matchingMetric(selectors, aLongTimeAgoSeconds()))
            .limit(random().nextInt(10, 20));
        var activeRelevantMetric = matchingMetric(selectors, nowSeconds);

        var irrelevantMetrics = Stream.generate(() -> metric(aLongTimeAgoSeconds()))
            .limit(random().nextInt(10, 20));

        var metrics = Streams.concat(Stream.of(activeRelevantMetric), notUsedRelevantMetrics, irrelevantMetrics)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);

        // act
        proc.start().join();

        // assert
        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels(LabelsFormat.format(activeRelevantMetric.getLabels()))
            .setMinLastPointSeconds(0)
            .build();
        assertEquals(expected, proc.progress());
    }

    @Test
    public void completeWithEmptyOnSmallShard() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var relevantMetricsKnown = Stream.generate(() -> matchingMetric(selectors, aLongTimeAgoSeconds()))
            .limit(random().nextInt(10, 20))
            .collect(toList());
        var relevantMetricsUnknown1 =
            Stream.generate(() -> matchingMetricInSpShard(1, selectors, UNKNOWN_LAST_POINT_SECONDS))
                .limit(random().nextInt(10, 20))
                .collect(toList());
        var relevantMetricsUnknown2 =
            Stream.generate(() -> matchingMetricInSpShard(2, selectors, UNKNOWN_LAST_POINT_SECONDS))
                .limit(random().nextInt(10, 20))
                .collect(toList());

        var irrelevantMetrics = Stream.generate(() -> metric(nowSeconds))
            .limit(random().nextInt(10, 20));

        var metrics = Streams.concat(
                relevantMetricsKnown.stream(),
                relevantMetricsUnknown1.stream(),
                relevantMetricsUnknown2.stream(),
                irrelevantMetrics
            )
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);
        ensureMeta(relevantMetricsUnknown1, CheckNoRecentWritesTest::aLongTimeAgoMillis);
        ensureMeta(relevantMetricsUnknown2, CheckNoRecentWritesTest::aLongTimeAgoMillis);

        // act
        proc.start().join();

        // assert
        var minKnown = relevantMetricsKnown.stream()
            .mapToInt(CoremonMetric::getLastPointSeconds)
            .min()
            .orElseThrow();
        var minStockpile = stockpileClient.minLastPointSeconds();

        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels("")
            .setMinLastPointSeconds(Math.min(minKnown, minStockpile))
            .build();
        assertEquals(expected, proc.progress());

        stockpileClient.expectTotalReadCalls(2);
    }

    @Test
    public void completeWithEmptyOnSmallShardWhenSomeCreatedAfterLaunch() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var relevantMetricsKnown = Stream.generate(() -> matchingMetric(selectors, aLongTimeAgoSeconds()))
            .limit(random().nextInt(10, 20))
            .collect(toList());
        var relevantMetricsUnknown1 =
            Stream.generate(() -> matchingMetricInSpShard(1, selectors, UNKNOWN_LAST_POINT_SECONDS))
                .limit(random().nextInt(10, 20))
                .collect(toList());
        var relevantMetricsUnknown2 =
            Stream.generate(() -> matchingMetricInSpShard(2, selectors, UNKNOWN_LAST_POINT_SECONDS))
                .limit(random().nextInt(10, 20))
                .collect(toList());

        var irrelevantMetricsDiffLabels = Stream.generate(() -> metric(nowSeconds))
            .limit(random().nextInt(10, 20));

        var createdAfter = millisecondsToSeconds(params.getCreatedAt()) + 1;
        var irrelevantMetricsCreatedAfter =
            Stream.generate(() -> matchingMetricWithCreatedAt(selectors, createdAfter, nowSeconds))
                .limit(random().nextInt(10, 20));

        var metrics = Streams.concat(
                relevantMetricsKnown.stream(),
                relevantMetricsUnknown1.stream(),
                relevantMetricsUnknown2.stream(),
                irrelevantMetricsDiffLabels,
                irrelevantMetricsCreatedAfter)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);
        ensureMeta(relevantMetricsUnknown1, CheckNoRecentWritesTest::aLongTimeAgoMillis);
        ensureMeta(relevantMetricsUnknown2, CheckNoRecentWritesTest::aLongTimeAgoMillis);

        // act
        proc.start().join();

        // assert
        var minKnown = relevantMetricsKnown.stream()
            .mapToInt(CoremonMetric::getLastPointSeconds)
            .min()
            .orElseThrow();
        var minStockpile = stockpileClient.minLastPointSeconds();

        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels("")
            .setMinLastPointSeconds(Math.min(minKnown, minStockpile))
            .build();
        assertEquals(expected, proc.progress());

        stockpileClient.expectTotalReadCalls(2);
    }

    @Test
    public void completeWithActiveMetricFoundOnSmallShard() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors, aLongTimeAgoSeconds()))
            .limit(random().nextInt(10, 20));
        var relevantMetricsUnknown = Stream.generate(() -> matchingMetric(selectors, UNKNOWN_LAST_POINT_SECONDS))
            .limit(random().nextInt(10, 20))
            .collect(toList());
        var activeRelevantMetricUnknown = matchingMetric(selectors, UNKNOWN_LAST_POINT_SECONDS);

        var irrelevantMetrics = Stream.generate(() -> metric(aLongTimeAgoSeconds()))
            .limit(random().nextInt(10, 20));

        var metrics = Streams.concat(
                relevantMetrics,
                relevantMetricsUnknown.stream(),
                Stream.of(activeRelevantMetricUnknown),
                irrelevantMetrics)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);
        ensureMeta(relevantMetricsUnknown, CheckNoRecentWritesTest::aLongTimeAgoMillis);
        ensureMeta(List.of(activeRelevantMetricUnknown), System::currentTimeMillis);

        // act
        proc.start().join();

        // assert
        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels(LabelsFormat.format(activeRelevantMetricUnknown.getLabels()))
            .setMinLastPointSeconds(0)
            .build();
        assertEquals(expected, proc.progress());
    }

    @Test
    public void completeWithEmptyOnBigShard() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var unknownSize1 = random().nextInt(1000, 2000);
        var unknownSize2 = random().nextInt(1000, 2000);
        var expectedBatches = calculateExpectedBatches(unknownSize1) + calculateExpectedBatches(unknownSize2);

        var relevantMetricsKnown = Stream.generate(() -> matchingMetric(selectors, aLongTimeAgoSeconds()))
            .limit(random().nextInt(1000, 2000))
            .collect(toList());
        var relevantMetricsUnknown1 =
            Stream.generate(() -> matchingMetricInSpShard(1, selectors, UNKNOWN_LAST_POINT_SECONDS))
                .limit(unknownSize1)
                .collect(toList());
        var relevantMetricsUnknown2 =
            Stream.generate(() -> matchingMetricInSpShard(2, selectors, UNKNOWN_LAST_POINT_SECONDS))
                .limit(unknownSize2)
                .collect(toList());

        var irrelevantMetrics = Stream.generate(() -> metric(nowSeconds))
            .limit(random().nextInt(1000, 2000));

        var metrics = Streams.concat(
                relevantMetricsKnown.stream(),
                relevantMetricsUnknown1.stream(),
                relevantMetricsUnknown2.stream(),
                irrelevantMetrics)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);
        ensureMeta(relevantMetricsUnknown1, CheckNoRecentWritesTest::aLongTimeAgoMillis);
        ensureMeta(relevantMetricsUnknown2, CheckNoRecentWritesTest::aLongTimeAgoMillis);

        // act
        proc.start().join();

        // assert
        var minKnown = relevantMetricsKnown.stream()
            .mapToInt(CoremonMetric::getLastPointSeconds)
            .min()
            .orElseThrow();
        var minStockpile = stockpileClient.minLastPointSeconds();

        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels("")
            .setMinLastPointSeconds(Math.min(minKnown, minStockpile))
            .build();
        assertEquals(expected, proc.progress());

        stockpileClient.expectTotalReadCalls(expectedBatches);
    }

    @Test
    public void completeWithEmptyOnBigShardWhenSomeCreatedAfterLaunch() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var unknownSize1 = random().nextInt(1000, 2000);
        var unknownSize2 = random().nextInt(1000, 2000);
        var expectedBatches = calculateExpectedBatches(unknownSize1) + calculateExpectedBatches(unknownSize2);

        var relevantMetricsKnown = Stream.generate(() -> matchingMetric(selectors, aLongTimeAgoSeconds()))
            .limit(random().nextInt(1000, 2000))
            .collect(toList());
        var relevantMetricsUnknown1 =
            Stream.generate(() -> matchingMetricInSpShard(1, selectors, UNKNOWN_LAST_POINT_SECONDS))
                .limit(unknownSize1)
                .collect(toList());
        var relevantMetricsUnknown2 =
            Stream.generate(() -> matchingMetricInSpShard(2, selectors, UNKNOWN_LAST_POINT_SECONDS))
                .limit(unknownSize2)
                .collect(toList());

        var irrelevantMetrics = Stream.generate(() -> metric(nowSeconds))
            .limit(random().nextInt(1000, 2000));

        var createdAfter = millisecondsToSeconds(params.getCreatedAt()) + 1;
        var irrelevantMetricsCreatedAfter =
            Stream.generate(() -> matchingMetricWithCreatedAt(selectors, createdAfter, nowSeconds))
                .limit(random().nextInt(1000, 2000));

        var metrics = Streams.concat(
                relevantMetricsKnown.stream(),
                relevantMetricsUnknown1.stream(),
                relevantMetricsUnknown2.stream(),
                irrelevantMetrics,
                irrelevantMetricsCreatedAfter)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);
        ensureMeta(relevantMetricsUnknown1, CheckNoRecentWritesTest::aLongTimeAgoMillis);
        ensureMeta(relevantMetricsUnknown2, CheckNoRecentWritesTest::aLongTimeAgoMillis);

        // act
        proc.start().join();

        // assert
        var minKnown = relevantMetricsKnown.stream()
            .mapToInt(CoremonMetric::getLastPointSeconds)
            .min()
            .orElseThrow();
        var minStockpile = stockpileClient.minLastPointSeconds();

        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels("")
            .setMinLastPointSeconds(Math.min(minKnown, minStockpile))
            .build();
        assertEquals(expected, proc.progress());

        stockpileClient.expectTotalReadCalls(expectedBatches);
    }

    @Test
    public void completeWithActiveMetricFoundOnBigShard() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors, aLongTimeAgoSeconds()))
            .limit(random().nextInt(1000, 2000));
        var relevantMetricsUnknown = Stream.generate(() -> matchingMetric(selectors, UNKNOWN_LAST_POINT_SECONDS))
            .limit(random().nextInt(1000, 2000))
            .collect(toList());
        var activeRelevantMetricUnknown = matchingMetric(selectors, UNKNOWN_LAST_POINT_SECONDS);

        var irrelevantMetrics = Stream.generate(() -> metric(aLongTimeAgoSeconds()))
            .limit(random().nextInt(1000, 2000));

        var metrics = Streams.concat(
                relevantMetrics,
                relevantMetricsUnknown.stream(),
                Stream.of(activeRelevantMetricUnknown),
                irrelevantMetrics)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);
        ensureMeta(relevantMetricsUnknown, CheckNoRecentWritesTest::aLongTimeAgoMillis);
        ensureMeta(List.of(activeRelevantMetricUnknown), System::currentTimeMillis);

        // act
        proc.start().join();

        // assert
        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels(LabelsFormat.format(activeRelevantMetricUnknown.getLabels()))
            .setMinLastPointSeconds(0)
            .build();
        assertEquals(expected, proc.progress());
    }

    @Test
    public void retryOnStockpileNotOkStatusCodes() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var metrics = Stream.generate(() -> matchingMetric(selectors, UNKNOWN_LAST_POINT_SECONDS))
            .limit(100)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);
        ensureMeta(metrics, CheckNoRecentWritesTest::aLongTimeAgoMillis);

        var a = new AtomicBoolean();
        stockpileClient.beforeSupplier = () -> completedFuture(
            a.getAndSet(true) && random().nextBoolean()
                ? EStockpileStatusCode.OK
                : EStockpileStatusCode.INTERNAL_ERROR);

        // act
        proc.start().join();

        // assert
        var expected = CheckNoRecentWritesProgress.newBuilder()
            .setComplete(true)
            .setRecentWriteLabels("")
            .setMinLastPointSeconds(stockpileClient.minLastPointSeconds())
            .build();
        assertEquals(expected, proc.progress());
    }

    @Test
    public void stockpileUnavailable() {
        // arrange
        retryConfig = retryConfig.withNumRetries(1);
        stockpileClient.beforeSupplier = unavailable();

        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var metrics = Stream.generate(() -> matchingMetric(selectors, UNKNOWN_LAST_POINT_SECONDS))
            .limit(1000)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);

        // act
        var status = proc.start().thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertNotEquals(Status.Code.OK, status.getCode());
        assertEquals(CheckNoRecentWritesProgress.getDefaultInstance(), proc.progress());
    }

    @Test
    public void canceledOnClose() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = checkNoRecentWrites(params, CheckNoRecentWritesProgress.getDefaultInstance());

        var metrics = Stream.generate(() -> matchingMetricInSpShard(1, selectors, UNKNOWN_LAST_POINT_SECONDS))
            .limit(500)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());
        ensureLastPointSeconds(params.getNumId(), metrics);
        ensureMeta(metrics, CheckNoRecentWritesTest::aLongTimeAgoMillis);

        var calls = new AtomicInteger(2);
        stockpileClient.beforeSupplier = () -> {
            if (calls.decrementAndGet() == 0) {
               proc.close();
            }

            return completedFuture(null);
        };

        // act
        var status = proc.start().thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertEquals(Status.Code.CANCELLED, status.getCode());
        assertEquals(CheckNoRecentWritesProgress.getDefaultInstance(), proc.progress());
    }

    private void ensureMetricsInDao(int numId, List<CoremonMetric> metrics) {
        var dao = metricsDaoFactory.create(numId, Labels.allocator);
        dao.add(metrics);
    }

    private void ensureShardReady(int numId) {
        metricsDaoFactory.resumeShardInit(numId);
        shardResolver.resolveShard(numId).awaitReady();
    }

    private void ensureLastPointSeconds(int numId, List<CoremonMetric> metrics) {
        var index = metrics.stream()
            .collect(toMap(
                CoremonMetric::getLabels,
                CoremonMetric::getLastPointSeconds));

        shardResolver.setLastPointSeconds(numId, index);
    }

    private void ensureMeta(List<CoremonMetric> metrics, LongSupplier tsSupplier) {
        for (var metric : metrics) {
            stockpileClient.addMeta(metric.getShardId(), metric.getLocalId(), tsSupplier.getAsLong());
        }
    }

    private CheckNoRecentWrites checkNoRecentWrites(
        DeleteMetricsParams params,
        CheckNoRecentWritesProgress progress)
    {
        return new CheckNoRecentWrites(
            retryConfig,
            stockpileClient,
            shardResolver,
            commonPool(),
            params,
            Selectors.parse(params.getSelectors()),
            progress,
            TEST_MAX_BATCH_SIZE);
    }

    private static DeleteMetricsParams params() {
        return DeleteMetricsRandom.params(MAX_NUM_ID);
    }

    private static int aLongTimeAgoSeconds() {
        return nowSeconds - (int) TimeUnit.DAYS.toSeconds(random().nextInt(13, 37));
    }

    private static long aLongTimeAgoMillis() {
        return System.currentTimeMillis() - TimeUnit.DAYS.toMillis(random().nextInt(13, 37));
    }

    private static ThreadLocalRandom random() {
        return ThreadLocalRandom.current();
    }

    private static Supplier<CompletableFuture<?>> unavailable() {
        return () -> failedFuture(Status.UNAVAILABLE.asRuntimeException());
    }

    private static int calculateExpectedBatches(double size) {
        return (int) Math.ceil(size / TEST_MAX_BATCH_SIZE);
    }

    @ParametersAreNonnullByDefault
    private static class MetaOnlyStockpileClientStub extends AbstractStockpileClientStub {

        private final ConcurrentHashMap<MetricId, MetricMeta> meta = new ConcurrentHashMap<>();

        private final AtomicInteger totalReadCalls = new AtomicInteger();

        @Override
        public CompletableFuture<ReadMetricsMetaResponse> readMetricsMeta(ReadMetricsMetaRequest request) {
            return before().thenApplyAsync(
                before -> {
                    if (before instanceof EStockpileStatusCode code && code != EStockpileStatusCode.OK) {
                        return ReadMetricsMetaResponse.newBuilder()
                            .setStatus(code)
                            .setStatusMessage("meh")
                            .build();
                    }

                    totalReadCalls.incrementAndGet();

                    var metas = request.getLocalIdsList().stream()
                        .map(localId -> metricId(request.getShardId(), localId))
                        .map(id -> meta.getOrDefault(id, meta(id.getLocalId(), UNKNOWN_LAST_POINT_SECONDS)))
                        .collect(toList());

                    return ReadMetricsMetaResponse.newBuilder()
                        .setStatus(EStockpileStatusCode.OK)
                        .addAllMeta(metas)
                        .build();
                },
                delayedExecutor(1, TimeUnit.MILLISECONDS, commonPool()));
        }

        void addMeta(int shardId, long localId, long tsMillis) {
            meta.put(metricId(shardId, localId), meta(localId, tsMillis));
        }

        void expectTotalReadCalls(int expected) {
            assertEquals(expected, this.totalReadCalls.get());
        }

        int minLastPointSeconds() {
            return meta.values().stream()
                .mapToLong(MetricMeta::getLastTsMillis)
                .mapToInt(InstantUtils::millisecondsToSeconds)
                .min()
                .orElseThrow();
        }

        private static MetricMeta meta(long localId, long tsMillis) {
            return MetricMeta.newBuilder()
                .setLocalId(localId)
                .setLastTsMillis(tsMillis)
                .build();
        }

        private static MetricId metricId(int shardId, long localId) {
            return MetricId.newBuilder()
                .setShardId(shardId)
                .setLocalId(localId)
                .build();
        }
    }

}
