package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Stream;

import com.google.common.collect.Streams;
import io.grpc.Status;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.coremon.api.task.DeleteMetricsTerminateProgress.CleanUpNonExistingShardProgress;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryDeletedMetricsDao;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.util.Collections.shuffle;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toUnmodifiableList;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.lessThan;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsAssert.assertMetrics;

/**
 * @author Stanislav Kashirin
 */
public class CleanUpNonExistingShardTest {

    @Rule
    public Timeout globalTimeout = Timeout.builder()
        .withTimeout(1, MINUTES)
        .withLookingForStuckThread(true)
        .build();

    private RetryConfig retryConfig;
    private InMemoryDeletedMetricsDao deletedMetricsDao;

    @Before
    public void setUp() {
        retryConfig = RetryConfig.DEFAULT
            .withNumRetries(Integer.MAX_VALUE)
            .withMaxDelay(0);

        deletedMetricsDao = new InMemoryDeletedMetricsDao();
    }

    @Test
    public void alreadyCompleted() {
        // arrange
        deletedMetricsDao.beforeSupplier = unavailable();

        var progress = CleanUpNonExistingShardProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(12345)
            .setDeletedMetrics(12345)
            .setProgress(1)
            .build();

        var proc = cleanUpNonExistingShard(params(), progress);

        // act
        proc.start().join();

        // assert
        assertEquals(progress, proc.progress());
    }

    @Test
    public void nothingToCleanUp() {
        // arrange
        var params = params();
        var proc = cleanUpNonExistingShard(params, CleanUpNonExistingShardProgress.getDefaultInstance());

        // act
        proc.start().join();

        // assert
        var expectedProgress = CleanUpNonExistingShardProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(0)
            .setDeletedMetrics(0)
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());
    }

    @Test
    public void cleanUpOnSmallShard() {
        // arrange
        var params = params();
        var proc = cleanUpNonExistingShard(params, CleanUpNonExistingShardProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(100, 500))
            .collect(toList());
        shuffle(relevantMetrics);

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsDiffOp.stream(),
                irrelevantMetricsDiffNumId.stream())
            .collect(toUnmodifiableList());

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);

        // act
        proc.start().join();

        // assert
        var expectedProgress = CleanUpNonExistingShardProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(relevantMetrics.size())
            .setDeletedMetrics(relevantMetrics.size())
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(irrelevantMetrics, deletedMetricsDao.metrics());
    }

    @Test
    public void cleanUpOnBigShard() {
        // arrange
        var params = params();
        var proc = cleanUpNonExistingShard(params, CleanUpNonExistingShardProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(2000, 5000))
            .collect(toList());
        shuffle(relevantMetrics);

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(500, 1000))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(500, 1000))
            .collect(toUnmodifiableList());
        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsDiffOp.stream(),
                irrelevantMetricsDiffNumId.stream())
            .collect(toUnmodifiableList());

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);

        // act
        proc.start().join();

        // assert
        var expectedProgress = CleanUpNonExistingShardProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(relevantMetrics.size())
            .setDeletedMetrics(relevantMetrics.size())
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(irrelevantMetrics, deletedMetricsDao.metrics());
    }

    @Test
    public void resumeCleanUp() {
        // arrange
        var params = params();

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsDiffOp.stream(),
                irrelevantMetricsDiffNumId.stream())
            .collect(toUnmodifiableList());

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);

        var progress = CleanUpNonExistingShardProgress.newBuilder()
            .setComplete(false)
            .setTotalMetrics(100)
            .setDeletedMetrics(100 - relevantMetrics.size())
            .setProgress((100 - relevantMetrics.size()) / 100.0)
            .build();
        var proc = cleanUpNonExistingShard(params, progress);

        // act
        proc.start().join();

        // assert
        var expectedProgress = CleanUpNonExistingShardProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(100)
            .setDeletedMetrics(100)
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(irrelevantMetrics, deletedMetricsDao.metrics());
    }

    @Test
    public void resumeCleanUpWhenProgressInaccurate() {
        // arrange
        var params = params();

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsDiffOp.stream(),
                irrelevantMetricsDiffNumId.stream())
            .collect(toUnmodifiableList());

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);

        var progress = CleanUpNonExistingShardProgress.newBuilder()
            .setComplete(false)
            .setTotalMetrics(100)
            .setDeletedMetrics(42)
            .setProgress(0.42)
            .build();
        var proc = cleanUpNonExistingShard(params, progress);

        // act
        proc.start().join();

        // assert
        var expectedProgress = CleanUpNonExistingShardProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(100)
            .setDeletedMetrics(100)
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(irrelevantMetrics, deletedMetricsDao.metrics());
    }

    @Test
    public void onErrorSaveLatestProgress() {
        // arrange
        retryConfig = retryConfig.withNumRetries(3);

        var params = params();
        var proc = cleanUpNonExistingShard(params, CleanUpNonExistingShardProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(1000, 2000))
            .collect(toUnmodifiableList());

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(500, 1000))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(500, 1000))
            .collect(toUnmodifiableList());

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);

        var initialSize = deletedMetricsDao.metrics().size();
        deletedMetricsDao.beforeSupplier = () -> {
            if (deletedMetricsDao.metrics().size() < initialSize) {
                return failedFuture(Status.UNAVAILABLE.asRuntimeException());
            }

            return completedFuture(null);
        };

        // act
        var status = proc.start().thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertNotEquals(Status.Code.OK, status.getCode());

        var progress = proc.progress();
        assertFalse(progress.getComplete());
        assertEquals(relevantMetrics.size(), progress.getTotalMetrics());
        assertThat(progress.getDeletedMetrics(), greaterThan(0));
        assertThat(progress.getProgress(), allOf(greaterThan(0.0), lessThan(1.0)));
    }

    @Test
    public void retryOnDaoNotOkStatusCodes() {
        // arrange
        var params = params();
        var proc = cleanUpNonExistingShard(params, CleanUpNonExistingShardProgress.getDefaultInstance());

        var metrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(1000)
            .collect(toList());
        shuffle(metrics);

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), metrics);

        var a = new AtomicBoolean();
        deletedMetricsDao.beforeSupplier =
            () -> a.getAndSet(true) && random().nextBoolean()
                ? completedFuture(null)
                : unavailable().get();

        // act
        proc.start().join();

        // assert
        var expectedProgress = CleanUpNonExistingShardProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(metrics.size())
            .setDeletedMetrics(metrics.size())
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(List.of(), deletedMetricsDao.metrics());
    }

    @Test
    public void canceledOnClose() {
        // arrange
        var params = params();
        var proc = cleanUpNonExistingShard(params, CleanUpNonExistingShardProgress.getDefaultInstance());

        var metrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(500)
            .collect(toList());
        shuffle(metrics);

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), metrics);

        var calls = new AtomicInteger(2);
        deletedMetricsDao.beforeSupplier = () -> {
            if (calls.decrementAndGet() == 0) {
               proc.close();
            }

            return completedFuture(null);
        };

        // act
        var status = proc.start().thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertEquals(Status.Code.CANCELLED, status.getCode());
        assertFalse(proc.progress().getComplete());
    }

    private void ensureDeletedMetricsInDao(String operationId, int numId, Collection<CoremonMetric> metrics) {
        deletedMetricsDao.putAll(operationId, numId, metrics);
    }

    private CleanUpNonExistingShard cleanUpNonExistingShard(
        DeleteMetricsParams params,
        CleanUpNonExistingShardProgress progress)
    {
        return new CleanUpNonExistingShard(
            retryConfig,
            deletedMetricsDao,
            params,
            progress);
    }

    private static DeleteMetricsParams params() {
        return DeleteMetricsRandom.params(42);
    }

    private static ThreadLocalRandom random() {
        return ThreadLocalRandom.current();
    }

    private static Supplier<CompletableFuture<?>> unavailable() {
        return () -> failedFuture(Status.UNAVAILABLE.asRuntimeException());
    }
}
