package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.List;
import java.util.OptionalInt;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardConf;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.query.GlobSelector;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static java.util.stream.Collectors.toUnmodifiableList;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;
import static ru.yandex.solomon.coremon.meta.CoremonMetric.UNKNOWN_LAST_POINT_SECONDS;
import static ru.yandex.solomon.util.time.InstantUtils.currentTimeSeconds;

/**
 * @author Stanislav Kashirin
 */
final class DeleteMetricsRandom {

    private DeleteMetricsRandom() {
    }

    static List<MetabaseShardConf> shardsWithNumIdUpTo(int maxNumId) {
        return IntStream.rangeClosed(1, maxNumId)
            .mapToObj(DeleteMetricsRandom::metabaseShardConf)
            .collect(toUnmodifiableList());
    }

    static MetabaseShardConf metabaseShardConf(int numId) {
        return MetabaseShardConf.of(
            "project_%s_%d".formatted(randomAlphanumeric(8), Integer.toUnsignedLong(numId)),
            "",
            "cluster_%s_%d".formatted(randomAlphanumeric(8), Integer.toUnsignedLong(numId)),
            "service_%s_%d".formatted(randomAlphanumeric(8), Integer.toUnsignedLong(numId)),
            numId,
4
        );
    }

    static DeleteMetricsParams params(int maxNumId) {
        var createdAt = System.currentTimeMillis() - TimeUnit.SECONDS.toMillis(random().nextInt(100));
        var subTaskCreatedAt = createdAt + TimeUnit.MILLISECONDS.toMillis(300);

        return DeleteMetricsParams.newBuilder()
            .setOperationId(randomAlphanumeric(8))
            .setNumId(random().nextInt(maxNumId) + 1)
            .setSelectors(formattedSelectorsGlob())
            .setCreatedAt(createdAt)
            .setSubTaskCreatedAt(subTaskCreatedAt)
            .build();
    }

    static String formattedLabels() {
        return LabelsFormat.format(labels());
    }

    static CoremonMetric matchingMetricInSpShard(int spShardId, Selectors selectors, int lastPointSeconds) {
        return metric(OptionalInt.of(spShardId), labels(selectors), OptionalInt.empty(), lastPointSeconds);
    }

    static CoremonMetric matchingMetric(Selectors selectors) {
        return metric(OptionalInt.empty(), labels(selectors), OptionalInt.empty(), UNKNOWN_LAST_POINT_SECONDS);
    }

    static CoremonMetric matchingMetric(Selectors selectors, int lastPointSeconds) {
        return metric(OptionalInt.empty(), labels(selectors), OptionalInt.empty(), lastPointSeconds);
    }

    static CoremonMetric matchingMetricWithCreatedAt(Selectors selectors, int createdAtSeconds) {
        return matchingMetricWithCreatedAt(selectors, createdAtSeconds, UNKNOWN_LAST_POINT_SECONDS);
    }

    static CoremonMetric matchingMetricWithCreatedAt(Selectors selectors, int createdAtSeconds, int lastPointSeconds) {
        return metric(
            OptionalInt.empty(),
            labels(selectors),
            OptionalInt.of(createdAtSeconds),
            lastPointSeconds);
    }

    static CoremonMetric metricInSpShard(int spShardId) {
        return metric(OptionalInt.of(spShardId), simpleLabels(), OptionalInt.empty(), UNKNOWN_LAST_POINT_SECONDS);
    }

    static CoremonMetric metric() {
        return metric(UNKNOWN_LAST_POINT_SECONDS);
    }

    static CoremonMetric metric(int lastPointSeconds) {
        return metric(OptionalInt.empty(), simpleLabels(), OptionalInt.empty(), lastPointSeconds);
    }

    private static CoremonMetric metric(
        OptionalInt maybeSpShardId,
        Labels labels,
        OptionalInt createdAtSeconds,
        int lastPointSeconds) {
        var spShardId = maybeSpShardId.orElseGet(() -> StockpileShardId.random(10, random()));
        var spLocalId = StockpileLocalId.random(random());

        var metric = new FileCoremonMetric(
            spShardId,
            spLocalId,
            labels,
            createdAtSeconds.orElseGet(() -> currentTimeSeconds() - random().nextInt(101, 1_000_000)),
            MetricType.DGAUGE);
        metric.setLastPointSeconds(lastPointSeconds);

        return metric;
    }

    private static String formattedSelectorsGlob() {
        var size = random().nextInt(1, 3);
        return SelectorsFormat.format(selectorsGlob(size));
    }

    private static Selectors selectorsGlob(int size) {
        var builder = Selectors.builder(size);
        for (int i = 0; i < size; i++) {
            builder.add(Selector.glob("k-" + randomAlphanumeric(8), "v-" + randomAlphanumeric(8) + "-*"));
        }
        return builder.build();
    }

    private static Labels labels() {
        var size = random().nextInt(1, 4);
        return labels(selectorsGlob(size));
    }

    private static Labels labels(Selectors selectors) {
        var size = selectors.size();
        var builder = new LabelsBuilder(size);
        for (int i = 0; i < size; i++) {
            var selector = selectors.at(i);
            assert selector instanceof GlobSelector;

            builder.add(label((GlobSelector) selector));
        }
        return builder.build();
    }

    private static Label label(GlobSelector selector) {
        var key = selector.getKey();
        var value = selector.getValue()
            .replaceAll("\\?", randomAlphanumeric(1))
            .replaceAll("\\*", randomAlphanumeric(8));
        return Labels.allocator.alloc(key, value);
    }

    private static Labels simpleLabels() {
        return Labels.of("key", "val" + random().nextLong());
    }

    private static ThreadLocalRandom random() {
        return ThreadLocalRandom.current();
    }
}
