package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Stream;

import com.google.common.collect.Streams;
import io.grpc.Status;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.coremon.api.task.DeleteMetricsTerminateProgress.DeletePermanentlyProgress;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryDeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardConf;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolverStub;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.api.EProjectId;

import static java.util.Collections.shuffle;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static java.util.concurrent.ForkJoinPool.commonPool;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toUnmodifiableList;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.lessThan;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsAssert.assertMetrics;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.metricInSpShard;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.shardsWithNumIdUpTo;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Stanislav Kashirin
 */
public class DeletePermanentlyTest {

    private static final int TEST_FIND_METRICS_LIMIT = 1_000;
    private static final int TEST_MAX_BATCH_SIZE = 100;

    private static final int MAX_NUM_ID = 3;
    private static final List<MetabaseShardConf> SHARDS = shardsWithNumIdUpTo(MAX_NUM_ID);

    @Rule
    public Timeout globalTimeout = Timeout.builder()
        .withTimeout(1, MINUTES)
        .withLookingForStuckThread(true)
        .build();

    private RetryConfig retryConfig;
    private StockpileClientStub stockpileClient;
    private InMemoryDeletedMetricsDao deletedMetricsDao;
    private InMemoryMetricsDaoFactory metricsDaoFactory;
    private MetabaseShardResolverStub shardResolver;

    @Before
    public void setUp() {
        retryConfig = RetryConfig.DEFAULT
            .withNumRetries(Integer.MAX_VALUE)
            .withMaxDelay(0);

        stockpileClient = new StockpileClientStub(commonPool());

        deletedMetricsDao = new InMemoryDeletedMetricsDao();

        metricsDaoFactory = new InMemoryMetricsDaoFactory();
        metricsDaoFactory.setSuspendShardInitOnCreate(true);

        shardResolver = new MetabaseShardResolverStub(
            SHARDS,
            metricsDaoFactory,
            stockpileClient);
    }

    @After
    public void tearDown() {
        close(metricsDaoFactory, shardResolver);
    }

    @Test
    public void alreadyCompleted() {
        // arrange
        deletedMetricsDao.beforeSupplier = unavailable();

        var progress = DeletePermanentlyProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(777)
            .setDeletedMetrics(777)
            .setProgress(1)
            .build();

        var proc = deletePermanently(params(), progress);

        // act
        proc.start().join();

        // assert
        assertEquals(progress, proc.progress());
    }

    @Test
    public void shardIsNotLocalAnymore() {
        // arrange
        deletedMetricsDao.beforeSupplier = unavailable();

        var params = params().toBuilder().setNumId(666).build();
        var proc = deletePermanently(params, DeletePermanentlyProgress.getDefaultInstance());

        // act
        proc.start().join();

        // assert
        assertEquals(DeletePermanentlyProgress.getDefaultInstance(), proc.progress());
    }

    @Test
    public void shardIsNotReady() {
        // arrange
        deletedMetricsDao.beforeSupplier = unavailable();

        var params = params();
        var proc = deletePermanently(params, DeletePermanentlyProgress.getDefaultInstance());

        // act
        proc.start().join();

        // assert
        assertEquals(DeletePermanentlyProgress.getDefaultInstance(), proc.progress());
    }

    @Test
    public void nothingToDelete() {
        // arrange
        stockpileClient.beforeSupplier = unavailable();

        var params = params();
        var proc = deletePermanently(params, DeletePermanentlyProgress.getDefaultInstance());

        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expectedProgress = DeletePermanentlyProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(0)
            .setDeletedMetrics(0)
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());
    }

    @Test
    public void deleteOnSmallShard() {
        // arrange
        var params = params();
        var proc = deletePermanently(params, DeletePermanentlyProgress.getDefaultInstance());

        var relevantMetrics1 = Stream.generate(() -> metricInSpShard(1))
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var relevantMetrics2 = Stream.generate(() -> metricInSpShard(2))
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var relevantMetrics = Streams.concat(
                relevantMetrics1.stream(),
                relevantMetrics2.stream())
            .collect(toList());
        shuffle(relevantMetrics);

        var irrelevantMetricsDiffOp = Stream.generate(() -> metricInSpShard(1))
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(() -> metricInSpShard(2))
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsDiffOp.stream(),
                irrelevantMetricsDiffNumId.stream())
            .collect(toUnmodifiableList());

        ensureMetricArchive(params.getNumId(), relevantMetrics);
        ensureMetricArchive(params.getNumId(), irrelevantMetricsDiffOp);
        ensureMetricArchive(params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expectedProgress = DeletePermanentlyProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(relevantMetrics.size())
            .setDeletedMetrics(relevantMetrics.size())
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(irrelevantMetrics, deletedMetricsDao.metrics());
        assertArchiveDeleteBefore(DeleteBeforeField.KEEP, irrelevantMetrics);
        assertArchiveDeleteBefore(DeleteBeforeField.DELETE_ALL, relevantMetrics);
    }

    @Test
    public void deleteOnBigShard() {
        // arrange
        var params = params();
        var proc = deletePermanently(params, DeletePermanentlyProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(1500, 2000))
            .collect(toList());
        shuffle(relevantMetrics);

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(500, 1000))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(500, 1000))
            .collect(toUnmodifiableList());
        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsDiffOp.stream(),
                irrelevantMetricsDiffNumId.stream())
            .collect(toUnmodifiableList());

        ensureMetricArchive(params.getNumId(), relevantMetrics);
        ensureMetricArchive(params.getNumId(), irrelevantMetricsDiffOp);
        ensureMetricArchive(params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expectedProgress = DeletePermanentlyProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(relevantMetrics.size())
            .setDeletedMetrics(relevantMetrics.size())
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(irrelevantMetrics, deletedMetricsDao.metrics());
        assertArchiveDeleteBefore(DeleteBeforeField.KEEP, irrelevantMetrics);
        assertArchiveDeleteBefore(DeleteBeforeField.DELETE_ALL, relevantMetrics);
    }

    @Test
    public void resumeDeletion() {
        // arrange
        var params = params();

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsDiffOp.stream(),
                irrelevantMetricsDiffNumId.stream())
            .collect(toUnmodifiableList());

        ensureMetricArchive(params.getNumId(), relevantMetrics);
        ensureMetricArchive(params.getNumId(), irrelevantMetricsDiffOp);
        ensureMetricArchive(params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureShardReady(params.getNumId());

        var progress = DeletePermanentlyProgress.newBuilder()
            .setComplete(false)
            .setTotalMetrics(100)
            .setDeletedMetrics(100 - relevantMetrics.size())
            .setProgress((100 - relevantMetrics.size()) / 100.0)
            .build();
        var proc = deletePermanently(params, progress);

        // act
        proc.start().join();

        // assert
        var expectedProgress = DeletePermanentlyProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(100)
            .setDeletedMetrics(100)
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(irrelevantMetrics, deletedMetricsDao.metrics());
        assertArchiveDeleteBefore(DeleteBeforeField.KEEP, irrelevantMetrics);
        assertArchiveDeleteBefore(DeleteBeforeField.DELETE_ALL, relevantMetrics);
    }

    @Test
    public void resumeDeletionWhenProgressInaccurate() {
        // arrange
        var params = params();

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsDiffOp.stream(),
                irrelevantMetricsDiffNumId.stream())
            .collect(toUnmodifiableList());

        ensureMetricArchive(params.getNumId(), relevantMetrics);
        ensureMetricArchive(params.getNumId(), irrelevantMetricsDiffOp);
        ensureMetricArchive(params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureShardReady(params.getNumId());

        var progress = DeletePermanentlyProgress.newBuilder()
            .setComplete(false)
            .setTotalMetrics(100)
            .setDeletedMetrics(1)
            .setProgress(0.01)
            .build();
        var proc = deletePermanently(params, progress);

        // act
        proc.start().join();

        // assert
        var expectedProgress = DeletePermanentlyProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(100)
            .setDeletedMetrics(100)
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(irrelevantMetrics, deletedMetricsDao.metrics());
        assertArchiveDeleteBefore(DeleteBeforeField.KEEP, irrelevantMetrics);
        assertArchiveDeleteBefore(DeleteBeforeField.DELETE_ALL, relevantMetrics);
    }

    @Test
    public void onErrorSaveLatestProgress() throws Exception {
        // arrange
        retryConfig = retryConfig.withNumRetries(3);

        var params = params();
        var proc = deletePermanently(params, DeletePermanentlyProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(1000, 2000))
            .collect(toUnmodifiableList());

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(500, 1000))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(500, 1000))
            .collect(toUnmodifiableList());

        ensureMetricArchive(params.getNumId(), relevantMetrics);
        ensureMetricArchive(params.getNumId(), irrelevantMetricsDiffOp);
        ensureMetricArchive(params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureShardReady(params.getNumId());

        var lucky = new AtomicBoolean(true);
        var atLeastOnce = new CompletableFuture<>();
        stockpileClient.beforeSupplier = () -> {
            if (lucky.getAndSet(false)) {
                return completedFuture(null);
            }

            return atLeastOnce.thenRun(() -> {
                throw Status.ABORTED.asRuntimeException();
            });
        };

        // act
        var future = proc.start();

        while (proc.progress().getProgress() == 0) {
            TimeUnit.MILLISECONDS.sleep(1);
        }
        atLeastOnce.completeAsync(() -> null);

        var status = future.thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertNotEquals(Status.Code.OK, status.getCode());

        var progress = proc.progress();
        assertFalse(progress.getComplete());
        assertEquals(relevantMetrics.size(), progress.getTotalMetrics());
        assertThat(progress.getDeletedMetrics(), greaterThan(0));
        assertThat(progress.getProgress(), allOf(greaterThan(0.0), lessThan(1.0)));
    }

    @Test
    public void retryOnDownstreamCallNotOkStatusCodes() {
        // arrange
        var params = params();
        var proc = deletePermanently(params, DeletePermanentlyProgress.getDefaultInstance());

        var metrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(1000)
            .collect(toList());
        shuffle(metrics);

        ensureMetricArchive(params.getNumId(), metrics);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        var a = new AtomicBoolean();
        deletedMetricsDao.beforeSupplier =
            () -> a.getAndSet(true) && random().nextBoolean()
                ? completedFuture(null)
                : unavailable().get();

        var b = new AtomicBoolean();
        stockpileClient.beforeSupplier =
            () -> b.getAndSet(true) && random().nextBoolean()
                ? completedFuture(null)
                : unavailable().get();

        // act
        proc.start().join();

        // assert
        var expectedProgress = DeletePermanentlyProgress.newBuilder()
            .setComplete(true)
            .setTotalMetrics(metrics.size())
            .setDeletedMetrics(metrics.size())
            .setProgress(1)
            .build();
        assertEquals(expectedProgress, proc.progress());

        assertMetrics(List.of(), deletedMetricsDao.metrics());
        assertArchiveDeleteBefore(DeleteBeforeField.DELETE_ALL, metrics);
    }

    @Test
    public void canceledOnClose() {
        // arrange
        var params = params();
        var proc = deletePermanently(params, DeletePermanentlyProgress.getDefaultInstance());

        var metrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(500)
            .collect(toList());
        shuffle(metrics);

        ensureMetricArchive(params.getNumId(), metrics);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        var calls = new AtomicInteger(2);
        deletedMetricsDao.beforeSupplier = () -> {
            if (calls.decrementAndGet() == 0) {
               proc.close();
            }

            return completedFuture(null);
        };

        // act
        var status = proc.start().thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertEquals(Status.Code.CANCELLED, status.getCode());
        assertFalse(proc.progress().getComplete());
    }

    private void ensureDeletedMetricsInDao(String operationId, int numId, Collection<CoremonMetric> metrics) {
        deletedMetricsDao.putAll(operationId, numId, metrics);
    }

    private void ensureMetricArchive(int numId, Collection<CoremonMetric> metrics) {
        metrics.forEach(m -> stockpileClient.addTimeSeries(m.getShardId(), m.getLocalId(), metricArchive(numId)));
    }

    private void ensureShardReady(int numId) {
        metricsDaoFactory.resumeShardInit(numId);
        shardResolver.resolveShard(numId).awaitReady();
    }

    private void assertArchiveDeleteBefore(long expectedDeleteBefore, Collection<CoremonMetric> metrics) {
        for (var m : metrics) {
            var timeSeries = stockpileClient.getTimeSeries(m.getShardId(), m.getLocalId());
            assertEquals(m.toString(), expectedDeleteBefore, timeSeries.getDeleteBefore());
        }
    }

    private DeletePermanently deletePermanently(
        DeleteMetricsParams params,
        DeletePermanentlyProgress progress)
    {
        return new DeletePermanently(
            retryConfig,
            deletedMetricsDao,
            stockpileClient,
            shardResolver,
            commonPool(),
            params,
            progress,
            TEST_FIND_METRICS_LIMIT,
            TEST_MAX_BATCH_SIZE);
    }

    private static DeleteMetricsParams params() {
        return DeleteMetricsRandom.params(MAX_NUM_ID);
    }

    private static MetricArchiveImmutable metricArchive(int numId) {
        try (var archive = new MetricArchiveMutable()) {
            archive.setOwnerProjectIdEnum(EProjectId.SOLOMON);
            archive.setOwnerShardId(numId);
            archive.addRecord(randomPoint());
            return archive.toImmutableNoCopy();
        }
    }

    private static ThreadLocalRandom random() {
        return ThreadLocalRandom.current();
    }

    private static Supplier<CompletableFuture<?>> unavailable() {
        return () -> failedFuture(Status.UNAVAILABLE.asRuntimeException());
    }

}
