package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Stream;

import com.google.common.collect.Streams;
import io.grpc.Status;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.DeleteMetricsMoveProgress.MoveToDeletedMetricsProgress;
import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryDeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDao;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardConf;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolverStub;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.util.Collections.shuffle;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static java.util.concurrent.ForkJoinPool.commonPool;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toUnmodifiableList;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.lessThan;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsAssert.assertMetrics;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.matchingMetric;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.matchingMetricWithCreatedAt;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.shardsWithNumIdUpTo;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
public class MoveToDeletedMetricsTest {

    private static final int TEST_MAX_BATCH_SIZE = 100;

    private static final int MAX_NUM_ID = 3;
    private static final List<MetabaseShardConf> SHARDS = shardsWithNumIdUpTo(MAX_NUM_ID);

    @Rule
    public Timeout globalTimeout = Timeout.builder()
        .withTimeout(1, MINUTES)
        .withLookingForStuckThread(true)
        .build();

    private RetryConfig retryConfig;
    private InMemoryDeletedMetricsDao deletedMetricsDao;
    private InMemoryMetricsDaoFactory metricsDaoFactory;
    private MetabaseShardResolverStub shardResolver;

    @Before
    public void setUp() {
        retryConfig = RetryConfig.DEFAULT
            .withNumRetries(Integer.MAX_VALUE)
            .withMaxDelay(0);

        deletedMetricsDao = new InMemoryDeletedMetricsDao();

        metricsDaoFactory = new InMemoryMetricsDaoFactory();
        metricsDaoFactory.setSuspendShardInitOnCreate(true);

        shardResolver = new MetabaseShardResolverStub(
            SHARDS,
            metricsDaoFactory,
            new StockpileClientStub(commonPool()));
    }

    @After
    public void tearDown() {
        close(metricsDaoFactory, shardResolver);
    }

    @Test
    public void alreadyCompleted() {
        // arrange
        deletedMetricsDao.beforeSupplier = unavailable();

        var progress = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(777)
            .setProcessed(777)
            .setProgress(1)
            .build();

        var proc = moveToDeletedMetrics(params(), progress);

        // act
        proc.start().join();

        // assert
        assertEquals(progress, proc.progress());
    }

    @Test
    public void shardIsNotLocalAnymore() {
        // arrange
        deletedMetricsDao.beforeSupplier = unavailable();

        var params = params().toBuilder().setNumId(666).build();
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        // act
        proc.start().join();

        // assert
        assertEquals(MoveToDeletedMetricsProgress.getDefaultInstance(), proc.progress());
    }

    @Test
    public void shardIsNotReady() {
        // arrange
        deletedMetricsDao.beforeSupplier = unavailable();

        var params = params();
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        // act
        proc.start().join();

        // assert
        assertEquals(MoveToDeletedMetricsProgress.getDefaultInstance(), proc.progress());
    }

    @Test
    public void nothingToMove() {
        // arrange
        var params = params();
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expected = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(0)
            .setProcessed(0)
            .setProgress(1)
            .setExactTotalMetrics(0)
            .build();
        assertEquals(expected, proc.progress());
    }

    @Test
    public void nothingToMoveWhenAllMetricsCreatedAfterLaunch() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        var createdAfter = millisecondsToSeconds(params.getCreatedAt()) + 1;
        var irrelevantMetricsCreatedAfter = Stream.generate(() -> matchingMetricWithCreatedAt(selectors, createdAfter))
            .limit(random().nextInt(1500, 2000))
            .collect(toList());

        ensureMetricsInDao(params.getNumId(), irrelevantMetricsCreatedAfter);
        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expected = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(irrelevantMetricsCreatedAfter.size())
            .setProcessed(irrelevantMetricsCreatedAfter.size())
            .setProgress(1)
            .setExactTotalMetrics(0)
            .build();
        assertEquals(expected, proc.progress());

        assertMetrics(irrelevantMetricsCreatedAfter, getMetricsDao(params.getNumId()).metrics());
        assertMetrics(List.of(), deletedMetricsDao.metrics());
    }

    @Test
    public void deleteOnSmallShardWhenAllMetricsCreatedBeforeLaunch() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors))
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var metrics = Streams.concat(
                relevantMetrics.stream(),
                irrelevantMetrics.stream())
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expected = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(relevantMetrics.size())
            .setProcessed(relevantMetrics.size())
            .setExactTotalMetrics(relevantMetrics.size())
            .setProgress(1)
            .build();
        assertEquals(expected, proc.progress());

        assertMetrics(irrelevantMetrics, getMetricsDao(params.getNumId()).metrics());
        assertMetrics(relevantMetrics, deletedMetricsDao.metrics());
    }

    @Test
    public void deleteOnSmallShardWhenSomeMetricsCreatedAfterLaunch() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors))
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var createdAfter = millisecondsToSeconds(params.getCreatedAt()) + 1;
        var irrelevantMetricsCreatedAfter = Stream.generate(() -> matchingMetricWithCreatedAt(selectors, createdAfter))
            .limit(random().nextInt(10, 20))
            .collect(toList());

        var irrelevantMetricsDiffLabels = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20));

        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsCreatedAfter.stream(),
                irrelevantMetricsDiffLabels)
            .collect(toList());

        var metrics = Streams.concat(
                relevantMetrics.stream(),
                irrelevantMetrics.stream())
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expected = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(irrelevantMetricsCreatedAfter.size() + relevantMetrics.size())
            .setProcessed(irrelevantMetricsCreatedAfter.size() + relevantMetrics.size())
            .setProgress(1)
            .setExactTotalMetrics(relevantMetrics.size())
            .build();
        assertEquals(expected, proc.progress());

        assertMetrics(irrelevantMetrics, getMetricsDao(params.getNumId()).metrics());
        assertMetrics(relevantMetrics, deletedMetricsDao.metrics());
    }

    @Test
    public void deleteOnBigShardWhenAllMetricsCreatedBeforeLaunch() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors))
            .limit(random().nextInt(1500, 2000))
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(1500, 2000))
            .collect(toUnmodifiableList());

        var metrics = Streams.concat(
                relevantMetrics.stream(),
                irrelevantMetrics.stream())
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expected = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(relevantMetrics.size())
            .setProcessed(relevantMetrics.size())
            .setProgress(1)
            .setExactTotalMetrics(relevantMetrics.size())
            .build();
        assertEquals(expected, proc.progress());

        assertMetrics(irrelevantMetrics, getMetricsDao(params.getNumId()).metrics());
        assertMetrics(relevantMetrics, deletedMetricsDao.metrics());
    }

    @Test
    public void deleteOnBigShardWhenSomeMetricsCreatedAfterLaunch() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors))
            .limit(random().nextInt(1500, 2000))
            .collect(toUnmodifiableList());

        var createdAfter = millisecondsToSeconds(params.getCreatedAt()) + 1;
        var irrelevantMetricsCreatedAfter = Stream.generate(() -> matchingMetricWithCreatedAt(selectors, createdAfter))
            .limit(random().nextInt(1500, 2000))
            .collect(toList());

        var irrelevantMetricsDiffLabels = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(1500, 2000));

        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsCreatedAfter.stream(),
                irrelevantMetricsDiffLabels)
            .collect(toList());

        var metrics = Streams.concat(
                relevantMetrics.stream(),
                irrelevantMetrics.stream())
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        // act
        proc.start().join();

        // assert
        var expected = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(irrelevantMetricsCreatedAfter.size() + relevantMetrics.size())
            .setProcessed(irrelevantMetricsCreatedAfter.size() + relevantMetrics.size())
            .setProgress(1)
            .setExactTotalMetrics(relevantMetrics.size())
            .build();
        assertEquals(expected, proc.progress());

        assertMetrics(irrelevantMetrics, getMetricsDao(params.getNumId()).metrics());
        assertMetrics(relevantMetrics, deletedMetricsDao.metrics());
    }

    @Test
    public void resumeDeletion() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors))
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var metrics = Streams.concat(
                relevantMetrics.stream(),
                irrelevantMetrics.stream())
            .collect(toList());
        shuffle(metrics);

        var deletedPreviously = Stream.generate(() -> matchingMetric(selectors))
            .limit(100 - relevantMetrics.size())
            .collect(toUnmodifiableList());

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), deletedPreviously);
        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        var progress = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(false)
            .setEstimatedTotalMetrics(100)
            .setProcessed(100 - relevantMetrics.size())
            .setProgress((100 - relevantMetrics.size()) / 100.0)
            .build();
        var proc = moveToDeletedMetrics(params, progress);

        // act
        proc.start().join();

        // assert
        var expected = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(100)
            .setProcessed(100)
            .setProgress(1)
            .setExactTotalMetrics(100)
            .build();
        assertEquals(expected, proc.progress());

        assertMetrics(irrelevantMetrics, getMetricsDao(params.getNumId()).metrics());

        var expectedDeleted = Stream.concat(
                relevantMetrics.stream(),
                deletedPreviously.stream())
            .collect(toList());
        assertMetrics(expectedDeleted, deletedMetricsDao.metrics());
    }

    @Test
    public void resumeDeletionWhenTotalChanged() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors))
            .limit(111)
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var metrics = Streams.concat(
                relevantMetrics.stream(),
                irrelevantMetrics.stream())
            .collect(toList());
        shuffle(metrics);

        var deletedPreviously = relevantMetrics.subList(0, 99);

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), deletedPreviously);
        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        var progress = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(false)
            .setEstimatedTotalMetrics(100)
            .setProcessed(99)
            .setProgress(0.99)
            .build();
        var proc = moveToDeletedMetrics(params, progress);

        // act
        proc.start().join();

        // assert
        var expected = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(100)
            .setProcessed(210)
            .setProgress(1)
            .setExactTotalMetrics(111)
            .build();
        assertEquals(expected, proc.progress());

        assertMetrics(irrelevantMetrics, getMetricsDao(params.getNumId()).metrics());
        assertMetrics(relevantMetrics, deletedMetricsDao.metrics());
    }

    @Test
    public void onErrorSaveLatestProgress() throws Exception {
        // arrange
        retryConfig = retryConfig.withNumRetries(3);

        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        var relevantMetrics = Stream.generate(() -> matchingMetric(selectors))
            .limit(random().nextInt(1000, 2000))
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(1000, 2000))
            .collect(toUnmodifiableList());

        var metrics = Streams.concat(
                relevantMetrics.stream(),
                irrelevantMetrics.stream())
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        var lucky = new AtomicBoolean(true);
        var atLeastOnce = new CompletableFuture<>();
        deletedMetricsDao.beforeSupplier = () -> {
            if (lucky.getAndSet(false)) {
                return completedFuture(null);
            }

            return atLeastOnce.thenRun(() -> {
                throw Status.ABORTED.asRuntimeException();
            });
        };

        // act
        var future = proc.start();

        while (proc.progress().getProgress() == 0) {
            TimeUnit.MILLISECONDS.sleep(1);
        }
        atLeastOnce.completeAsync(() -> null);

        var status = future.thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertNotEquals(Status.Code.OK, status.getCode());

        var progress = proc.progress();
        assertFalse(progress.getComplete());
        assertEquals(relevantMetrics.size(), progress.getEstimatedTotalMetrics());
        assertThat(progress.getProcessed(), greaterThan(0));
        assertThat(progress.getProgress(), allOf(greaterThan(0.0), lessThan(1.0)));
    }

    @Test
    public void retryOnDaoNotOkStatusCodes() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        var metrics = Stream.generate(() -> matchingMetric(selectors))
            .limit(1000)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        var a = new AtomicBoolean();
        deletedMetricsDao.beforeSupplier =
            () -> a.getAndSet(true) && random().nextBoolean()
                ? completedFuture(null)
                : unavailable().get();

        var b = new AtomicBoolean();
        getMetricsDao(params.getNumId()).beforeSupplier =
            () -> b.getAndSet(true) && random().nextBoolean()
                ? completedFuture(null)
                : unavailable().get();

        // act
        proc.start().join();

        // assert
        var expected = MoveToDeletedMetricsProgress.newBuilder()
            .setComplete(true)
            .setEstimatedTotalMetrics(metrics.size())
            .setProcessed(metrics.size())
            .setProgress(1)
            .setExactTotalMetrics(metrics.size())
            .build();
        assertEquals(expected, proc.progress());

        assertMetrics(List.of(), getMetricsDao(params.getNumId()).metrics());
        assertMetrics(metrics, deletedMetricsDao.metrics());
    }

    @Test
    public void canceledOnClose() {
        // arrange
        var params = params();
        var selectors = Selectors.parse(params.getSelectors());
        var proc = moveToDeletedMetrics(params, MoveToDeletedMetricsProgress.getDefaultInstance());

        var metrics = Stream.generate(() -> matchingMetric(selectors))
            .limit(500)
            .collect(toList());
        shuffle(metrics);

        ensureMetricsInDao(params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        var calls = new AtomicInteger(2);
        deletedMetricsDao.beforeSupplier = () -> {
            if (calls.decrementAndGet() == 0) {
               proc.close();
            }

            return completedFuture(null);
        };

        // act
        var status = proc.start().thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertEquals(Status.Code.CANCELLED, status.getCode());
        assertFalse(proc.progress().getComplete());
    }

    private void ensureDeletedMetricsInDao(String operationId, int numId, Collection<CoremonMetric> metrics) {
        deletedMetricsDao.putAll(operationId, numId, metrics);
    }

    private void ensureMetricsInDao(int numId, List<CoremonMetric> metrics) {
        getMetricsDao(numId).add(metrics);
    }

    private InMemoryMetricsDao getMetricsDao(int numId) {
        return metricsDaoFactory.create(numId, Labels.allocator);
    }

    private void ensureShardReady(int numId) {
        metricsDaoFactory.resumeShardInit(numId);
        shardResolver.resolveShard(numId).awaitReady();
    }

    private MoveToDeletedMetrics moveToDeletedMetrics(
        DeleteMetricsParams params,
        MoveToDeletedMetricsProgress progress)
    {
        return new MoveToDeletedMetrics(
            retryConfig,
            deletedMetricsDao,
            metricsDaoFactory,
            shardResolver,
            commonPool(),
            params,
            Selectors.parse(params.getSelectors()),
            progress,
            TEST_MAX_BATCH_SIZE);
    }

    private static DeleteMetricsParams params() {
        return DeleteMetricsRandom.params(MAX_NUM_ID);
    }

    private static ThreadLocalRandom random() {
        return ThreadLocalRandom.current();
    }

    private static Supplier<CompletableFuture<?>> unavailable() {
        return () -> failedFuture(Status.UNAVAILABLE.asRuntimeException());
    }

}
