package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.DeleteMetricsMoveProgress.ReloadShardProgress;
import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDao;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardConf;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolverStub;
import ru.yandex.solomon.metrics.client.StockpileClientStub;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.ForkJoinPool.commonPool;
import static java.util.concurrent.TimeUnit.MINUTES;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotSame;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.metric;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.shardsWithNumIdUpTo;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Stanislav Kashirin
 */
public class ReloadShardTest {

    private static final int MAX_NUM_ID = 3;
    private static final List<MetabaseShardConf> SHARDS = shardsWithNumIdUpTo(MAX_NUM_ID);

    @Rule
    public Timeout globalTimeout = Timeout.builder()
        .withTimeout(1, MINUTES)
        .withLookingForStuckThread(true)
        .build();

    private InMemoryMetricsDaoFactory metricsDaoFactory;
    private MetabaseShardResolverStub shardResolver;

    @Before
    public void setUp() {
        metricsDaoFactory = new InMemoryMetricsDaoFactory();
        metricsDaoFactory.setSuspendShardInitOnCreate(true);

        shardResolver = new MetabaseShardResolverStub(
            SHARDS,
            metricsDaoFactory,
            new StockpileClientStub(commonPool()));
    }

    @After
    public void tearDown() {
        close(metricsDaoFactory, shardResolver);
    }

    @Test
    public void alreadyCompleted() {
        // arrange
        shardResolver.setInitCompleted(false);

        var progress = ReloadShardProgress.newBuilder()
            .setComplete(true)
            .build();
        var proc = reloadShard(params(), progress);

        // act
        proc.start().join();

        // assert
        assertEquals(progress, proc.progress());
    }

    @Test
    public void shardIsNotLocalAnymore() {
        // arrange
        var params = params().toBuilder().setNumId(666).build();
        var proc = reloadShard(params, ReloadShardProgress.getDefaultInstance());

        // act
        proc.start().join();

        // assert
        assertEquals(ReloadShardProgress.getDefaultInstance(), proc.progress());
    }

    @Test
    public void shardIsNotReady() {
        // arrange
        var params = params();
        var proc = reloadShard(params, ReloadShardProgress.getDefaultInstance());

        var daoCalls = new AtomicInteger();
        getMetricsDao(params.getNumId()).beforeSupplier = () -> {
            daoCalls.incrementAndGet();
            return completedFuture(null);
        };

        // act
        proc.start().join();

        // assert
        assertEquals(ReloadShardProgress.getDefaultInstance(), proc.progress());
        assertEquals(0, daoCalls.get());
    }

    @Test
    public void reload() {
        // arrange
        var params = params();
        var proc = reloadShard(params, ReloadShardProgress.getDefaultInstance());

        ensureMetricsInDao(params.getNumId(), List.of(metric(), metric(), metric()));
        var shard = ensureShardReady(params.getNumId());

        // act
        var fileMetricsBefore = shard.getStorage().getFileMetrics();
        proc.start().join();
        var fileMetricsAfter = shard.getStorage().getFileMetrics();

        // assert
        assertNotSame(fileMetricsBefore, fileMetricsAfter);

        var expected = ReloadShardProgress.newBuilder()
            .setComplete(true)
            .build();
        assertEquals(expected, proc.progress());
    }

    private MetabaseShard ensureShardReady(int numId) {
        metricsDaoFactory.resumeShardInit(numId);
        var metabaseShard = shardResolver.resolveShard(numId);
        metabaseShard.awaitReady();
        return metabaseShard;
    }

    private void ensureMetricsInDao(int numId, List<CoremonMetric> metrics) {
        getMetricsDao(numId).add(metrics);
    }

    private InMemoryMetricsDao getMetricsDao(int numId) {
        return metricsDaoFactory.create(numId, Labels.allocator);
    }

    private ReloadShard reloadShard(DeleteMetricsParams params, ReloadShardProgress progress) {
        return new ReloadShard(
            shardResolver,
            params,
            progress);
    }

    private static DeleteMetricsParams params() {
        return DeleteMetricsRandom.params(MAX_NUM_ID);
    }

}
