package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Stream;

import com.google.common.collect.Streams;
import io.grpc.Status;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryDeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDao;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardConf;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolverStub;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.util.Collections.shuffle;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static java.util.concurrent.ForkJoinPool.commonPool;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toUnmodifiableList;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsAssert.assertMetrics;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.metricInSpShard;
import static ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRandom.shardsWithNumIdUpTo;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Stanislav Kashirin
 */
public class RepairDeletedMetricsTest {

    private static final int TEST_FIND_METRICS_LIMIT = 1000;

    private static final int MAX_NUM_ID = 3;
    private static final List<MetabaseShardConf> SHARDS = shardsWithNumIdUpTo(MAX_NUM_ID);

    @Rule
    public Timeout globalTimeout = Timeout.builder()
        .withTimeout(1, MINUTES)
        .withLookingForStuckThread(true)
        .build();

    private RetryConfig retryConfig;
    private InMemoryDeletedMetricsDao deletedMetricsDao;
    private InMemoryMetricsDaoFactory metricsDaoFactory;
    private MetabaseShardResolverStub shardResolver;

    @Before
    public void setUp() {
        retryConfig = RetryConfig.DEFAULT
            .withNumRetries(Integer.MAX_VALUE)
            .withMaxDelay(0);

        var stockpileClient = new StockpileClientStub(commonPool());

        deletedMetricsDao = new InMemoryDeletedMetricsDao();

        metricsDaoFactory = new InMemoryMetricsDaoFactory();
        metricsDaoFactory.setSuspendShardInitOnCreate(true);

        shardResolver = new MetabaseShardResolverStub(
            SHARDS,
            metricsDaoFactory,
            stockpileClient);
    }

    @After
    public void tearDown() {
        close(metricsDaoFactory, shardResolver);
    }

    @Test
    public void shardIsNotLocalAnymore() {
        // arrange
        deletedMetricsDao.beforeSupplier = unavailable();

        var params = params().toBuilder().setNumId(666).build();
        var proc = repairDeletedMetrics(params);

        // act
        var complete = proc.repair().join();

        // assert
        assertFalse(complete);
    }

    @Test
    public void shardIsNotReady() {
        // arrange
        deletedMetricsDao.beforeSupplier = unavailable();

        var params = params();
        var proc = repairDeletedMetrics(params);

        // act
        var complete = proc.repair().join();

        // assert
        assertFalse(complete);
    }

    @Test
    public void nothingToRepair() {
        // arrange
        var params = params();
        var proc = repairDeletedMetrics(params);

        ensureShardReady(params.getNumId());

        // act
        var complete = proc.repair().join();

        // assert
        assertTrue(complete);
        assertMetrics(List.of(), deletedMetricsDao.metrics());
    }

    @Test
    public void gracefullyRepair() {
        // arrange
        var params = params();

        var relevantMetrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(3000, 4000))
            .collect(toUnmodifiableList());

        var lo = random().nextInt(2500);
        var hi = lo + 5;
        var actuallyNotDeletedMetrics = relevantMetrics.subList(lo, hi);

        var irrelevantMetricsDiffOp = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());
        var irrelevantMetricsDiffNumId = Stream.generate(DeleteMetricsRandom::metric)
            .limit(random().nextInt(10, 20))
            .collect(toUnmodifiableList());

        var irrelevantMetrics = Streams.concat(
                irrelevantMetricsDiffOp.stream(),
                irrelevantMetricsDiffNumId.stream())
            .collect(toUnmodifiableList());

        var otherExistingMetrics = Stream.generate(() -> metricInSpShard(666))
            .limit(random().nextInt(3, 5));
        var alreadyExistingMetrics = Stream.concat(
                actuallyNotDeletedMetrics.stream(),
                otherExistingMetrics)
            .collect(toList());

        ensureMetricsInDao(params.getNumId(), alreadyExistingMetrics);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), relevantMetrics);
        ensureDeletedMetricsInDao(params.getOperationId() + "LOL", params.getNumId(), irrelevantMetricsDiffOp);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId() + 1000, irrelevantMetricsDiffNumId);
        ensureShardReady(params.getNumId());

        var proc = repairDeletedMetrics(params);

        // act
        var complete = proc.repair().join();

        // assert
        assertTrue(complete);

        var expectedDeleted = Stream.of(
                relevantMetrics.subList(0, lo),
                relevantMetrics.subList(hi, relevantMetrics.size()),
                irrelevantMetrics)
            .flatMap(Collection::stream)
            .collect(toList());
        assertMetrics(expectedDeleted, deletedMetricsDao.metrics());

        assertMetrics(alreadyExistingMetrics, getMetricsDao(params.getNumId()).metrics());
    }

    @Test
    public void retryOnDownstreamCallNotOkStatusCodes() {
        // arrange
        var params = params();
        var proc = repairDeletedMetrics(params);

        var metrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(1000)
            .collect(toList());
        shuffle(metrics);

        var lo = random().nextInt(500);
        var hi = lo + 5;
        var actuallyNotDeletedMetrics = metrics.subList(lo, hi);

        ensureMetricsInDao(params.getNumId(), actuallyNotDeletedMetrics);
        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        var a = new AtomicBoolean();
        deletedMetricsDao.beforeSupplier =
            () -> a.getAndSet(true) && random().nextBoolean()
                ? completedFuture(null)
                : unavailable().get();

        // act
        var complete = proc.repair().join();

        // assert
        assertTrue(complete);

        var expectedDeleted = Stream.concat(
                metrics.subList(0, lo).stream(),
                metrics.subList(hi, metrics.size()).stream())
            .collect(toList());
        assertMetrics(expectedDeleted, deletedMetricsDao.metrics());

        assertMetrics(actuallyNotDeletedMetrics, getMetricsDao(params.getNumId()).metrics());
    }

    @Test
    public void canceledOnClose() {
        // arrange
        var params = params();
        var proc = repairDeletedMetrics(params);

        var metrics = Stream.generate(DeleteMetricsRandom::metric)
            .limit(500)
            .collect(toList());
        shuffle(metrics);

        ensureDeletedMetricsInDao(params.getOperationId(), params.getNumId(), metrics);
        ensureShardReady(params.getNumId());

        var calls = new AtomicInteger(2);
        deletedMetricsDao.beforeSupplier = () -> {
            if (calls.decrementAndGet() == 0) {
               proc.close();
            }

            return completedFuture(null);
        };

        // act
        var status = proc.repair().thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertEquals(Status.Code.CANCELLED, status.getCode());
    }

    private void ensureDeletedMetricsInDao(String operationId, int numId, Collection<CoremonMetric> metrics) {
        deletedMetricsDao.putAll(operationId, numId, metrics);
    }

    private void ensureMetricsInDao(int numId, List<CoremonMetric> metrics) {
        getMetricsDao(numId).add(metrics);
    }

    private InMemoryMetricsDao getMetricsDao(int numId) {
        return metricsDaoFactory.create(numId, Labels.allocator);
    }

    private void ensureShardReady(int numId) {
        metricsDaoFactory.resumeShardInit(numId);
        shardResolver.resolveShard(numId).awaitReady();
    }

    private RepairDeletedMetrics repairDeletedMetrics(DeleteMetricsParams params)
    {
        return new RepairDeletedMetrics(
            retryConfig,
            deletedMetricsDao,
            shardResolver,
            commonPool(),
            params,
            TEST_FIND_METRICS_LIMIT);
    }

    private static DeleteMetricsParams params() {
        return DeleteMetricsRandom.params(MAX_NUM_ID);
    }

    private static ThreadLocalRandom random() {
        return ThreadLocalRandom.current();
    }

    private static Supplier<CompletableFuture<?>> unavailable() {
        return () -> failedFuture(Status.UNAVAILABLE.asRuntimeException());
    }

}
