package ru.yandex.solomon.coremon.tasks.removeShard;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import io.grpc.Status;
import io.grpc.Status.Code;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.RemoveShardProgress.RemoveMeta;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDao;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoveShardFromMetabaseTest {

    @Rule
    public Timeout globalTimeout = Timeout.builder()
            .withTimeout(1, TimeUnit.MINUTES)
            .withLookingForStuckThread(true)
            .build();

    private RetryConfig retryConfig;
    private InMemoryMetricsDao dao;

    @Before
    public void setUp() throws Exception {
        dao = new InMemoryMetricsDao();
        retryConfig = RetryConfig.DEFAULT
                .withNumRetries(Integer.MAX_VALUE)
                .withMaxDelay(0);
    }

    @Test
    public void alreadyDone() {
        var progress = RemoveMeta.newBuilder()
                .setComplete(true)
                .setTotalMetrics(42)
                .setRemovedMetrics(42)
                .setProgress(1)
                .build();

        dao.beforeSupplier = () -> {
            throw Status.UNAVAILABLE.asRuntimeException();
        };

        var proc = remove(progress);
        proc.start().join();
        assertEquals(progress, proc.progress());
    }

    @Test
    public void nothingToRemove() {
        var proc = remove(RemoveMeta.getDefaultInstance());
        proc.start().join();

        var expected = RemoveMeta.newBuilder()
                .setProgress(1)
                .setTotalMetrics(0)
                .setRemovedMetrics(0)
                .setComplete(true)
                .build();

        assertEquals(expected, proc.progress());
    }

    @Test
    public void removeMetrics() {
        var metrics= IntStream.range(1, ThreadLocalRandom.current().nextInt(100, 20_000))
                .mapToObj(ignore -> randomMetric())
                .collect(Collectors.toList());

        dao.add(metrics);

        var proc = remove(RemoveMeta.getDefaultInstance());
        proc.start().join();

        var expected = RemoveMeta.newBuilder()
                .setComplete(true)
                .setProgress(1)
                .setTotalMetrics(metrics.size())
                .setRemovedMetrics(metrics.size())
                .build();

        assertEquals(expected, proc.progress());
    }

    @Test
    public void retryError() {
        var metrics= IntStream.range(1, ThreadLocalRandom.current().nextInt(100, 20_000))
                .mapToObj(ignore -> randomMetric())
                .collect(Collectors.toList());

        dao.add(metrics);
        dao.beforeSupplier = () -> {
            if (ThreadLocalRandom.current().nextBoolean()) {
                return CompletableFuture.failedFuture(Status.UNAVAILABLE.asRuntimeException());
            }

            return CompletableFuture.completedFuture(null);
        };

        var proc = remove(RemoveMeta.getDefaultInstance());
        proc.start().join();

        var expected = RemoveMeta.newBuilder()
                .setComplete(true)
                .setProgress(1)
                .setTotalMetrics(metrics.size())
                .setRemovedMetrics(metrics.size())
                .build();

        assertEquals(expected, proc.progress());
    }

    @Test
    public void onErrorSaveLatestProgress() {
        var metrics= IntStream.range(1, ThreadLocalRandom.current().nextInt(10_000, 20_000))
                .mapToObj(ignore -> randomMetric())
                .collect(Collectors.toList());

        dao.add(metrics);
        dao.beforeSupplier = () -> {
            if (dao.metrics().size() < metrics.size()) {
                return CompletableFuture.failedFuture(Status.UNAVAILABLE.asRuntimeException());
            }

            return CompletableFuture.completedFuture(null);
        };

        retryConfig = retryConfig.withNumRetries(3);

        var proc = remove(RemoveMeta.getDefaultInstance());
        var status = proc.start().thenApply(unused -> Status.OK).exceptionally(Status::fromThrowable).join();
        assertNotEquals(Code.OK, status.getCode());

        var progress = proc.progress();
        assertFalse(progress.getComplete());
        assertEquals(metrics.size(), progress.getTotalMetrics());
    }

    @Test
    public void actualMetricCountAfterRestartOnOtherNode() {
        var metrics= IntStream.range(1, ThreadLocalRandom.current().nextInt(100, 1000))
                .mapToObj(ignore -> randomMetric())
                .collect(Collectors.toList());

        dao.add(metrics);

        var proc = remove(RemoveMeta.newBuilder()
                .setTotalMetrics(5_000)
                .setRemovedMetrics(200)
                .setProgress(0.1)
                .build());
        proc.start().join();

        var expected = RemoveMeta.newBuilder()
                .setComplete(true)
                .setProgress(1)
                .setTotalMetrics(5_000)
                .setRemovedMetrics(5_000)
                .build();

        assertEquals(expected, proc.progress());
    }

    @Test
    public void stopRemoveWhenCanceled() {
        var metrics= IntStream.range(1, ThreadLocalRandom.current().nextInt(100_000, 200_000))
                .mapToObj(ignore -> randomMetric())
                .collect(Collectors.toList());

        var proc = remove(RemoveMeta.getDefaultInstance());

        dao.add(metrics);
        dao.beforeSupplier = () -> {
            if (dao.metrics().size() != metrics.size()) {
                proc.close();
            }

            return CompletableFuture.completedFuture(null);
        };

        var status = proc.start().thenApply(unused -> Status.OK).exceptionally(Status::fromThrowable).join();
        assertNotEquals(Code.OK, status.getCode());

        var progress = proc.progress();
        assertFalse(progress.getComplete());
        assertEquals(metrics.size(), progress.getTotalMetrics());
        assertNotEquals(0, dao.metrics().size());
    }

    private CoremonMetric randomMetric() {
        var random = ThreadLocalRandom.current();
        int shardId = StockpileShardId.random(1000, random);
        long localId = StockpileLocalId.random(random);
        var labels = Labels.of("key", "value-"+random.nextLong());
        return new FileCoremonMetric(shardId, localId, labels, MetricType.DGAUGE);
    }

    private RemoveShardFromMetabase remove(RemoveMeta process) {
        return new RemoveShardFromMetabase(retryConfig, dao, process);
    }
}
