package ru.yandex.solomon.coremon.tasks.removeShard;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.protobuf.TextFormat;
import io.grpc.Status;
import io.grpc.Status.Code;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.coremon.api.task.RemoveShardParams;
import ru.yandex.coremon.api.task.RemoveShardProgress.RemoveData;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoveShardFromStockpileTest {
    private RetryConfig retryConfig;
    private StockpileClientStub stockpile;

    @Before
    public void setUp() throws Exception {
        stockpile = new StockpileClientStub(ForkJoinPool.commonPool());
        stockpile.setShardCount(1000);
        retryConfig = RetryConfig.DEFAULT
                .withNumRetries(Integer.MAX_VALUE)
                .withMaxDelay(0);
    }

    @Test
    public void alreadyDone() {
        var params = randomParams();
        var progress = successProgress();

        stockpile.predefineStatusCode(EStockpileStatusCode.NODE_UNAVAILABLE);

        var proc = remove(params, progress);
        proc.start().join();
        assertEquals(progress, proc.progress());
    }

    @Test
    public void nothingToRemove() {
        var params = randomParams();

        var proc = remove(params, RemoveData.getDefaultInstance());
        proc.start().join();

        assertEquals(successProgress(), proc.progress());
    }

    @Test
    public void removeMetricFromEachShard() {
        var params = randomParams();

        List<MetricId> metrics = new ArrayList<>();
        var archive = randomArchive(params.getNumId());
        for (int index = 0; index < 10; index++) {
            var id = stockpile.randomMetricId();
            stockpile.addTimeSeries(id, archive);
            metrics.add(id);
        }

        var proc = remove(params, RemoveData.getDefaultInstance());
        proc.start().join();

        assertEquals(successProgress(), proc.progress());
        for (var metricId : metrics) {
            assertNull(TextFormat.shortDebugString(metricId), stockpile.getTimeSeries(metricId));
        }
    }

    @Test
    public void retryError() {
        stockpile.beforeSupplier = () -> {
            stockpile.predefineStatusCode(ThreadLocalRandom.current().nextBoolean()
                    ? EStockpileStatusCode.OK
                    : EStockpileStatusCode.INTERNAL_ERROR);
            return CompletableFuture.completedFuture(null);
        };

        var params = randomParams();

        List<MetricId> metrics = new ArrayList<>();
        var archive = randomArchive(params.getNumId());
        for (int index = 0; index < 10; index++) {
            var id = stockpile.randomMetricId();
            stockpile.addTimeSeries(id, archive);
            metrics.add(id);
        }

        var proc = remove(params, RemoveData.getDefaultInstance());
        proc.start().join();

        assertEquals(successProgress(), proc.progress());
        for (var metricId : metrics) {
            assertNull(TextFormat.shortDebugString(metricId), stockpile.getTimeSeries(metricId));
        }
    }

    @Test
    public void onErrorSaveLatestProgress() {
        var params = randomParams();

        List<MetricId> metrics = new ArrayList<>();
        var archive = randomArchive(params.getNumId());
        for (int index = 0; index < 100; index++) {
            var id = stockpile.randomMetricId();
            stockpile.addTimeSeries(id, archive);
            metrics.add(id);
        }

        metrics.sort(Comparator.comparingInt(MetricId::getShardId));
        int failedShardId = metrics.get(50).getShardId();

        stockpile.predefineStatusCode(failedShardId, EStockpileStatusCode.INTERNAL_ERROR);
        retryConfig = retryConfig.withNumRetries(3);

        var proc = remove(params, RemoveData.getDefaultInstance());
        var status = proc.start()
                .thenApply(ignore -> Status.OK)
                .exceptionally(Status::fromThrowable)
                .join();

        assertNotEquals(Code.OK, status.getCode());

        var progress = proc.progress();
        assertFalse(progress.getComplete());
        assertNotEquals(failedShardId, progress.getLastShardId());;
    }

    @Test
    public void stopRemoveWhenCanceled() {
        var params = randomParams();

        var archive = randomArchive(params.getNumId());
        for (int index = 0; index < 100; index++) {
            var id = stockpile.randomMetricId();
            stockpile.addTimeSeries(id, archive);
        }

        var proc = remove(params, RemoveData.getDefaultInstance());

        AtomicInteger calls = new AtomicInteger(ThreadLocalRandom.current().nextInt(2, 10));
        stockpile.beforeSupplier = () -> {
            if (calls.get() == 0) {
                proc.close();
            }

            calls.decrementAndGet();
            return CompletableFuture.completedFuture(null);
        };

        var status = proc.start().thenApply(unused -> Status.OK).exceptionally(Status::fromThrowable).join();
        assertNotEquals(Code.OK, status.getCode());

        var progress = proc.progress();
        assertFalse(progress.getComplete());
    }

    private RemoveData successProgress() {
        return RemoveData.newBuilder()
                .setComplete(true)
                .setProgress(1)
                .setLastShardId(stockpile.getTotalShardsCount())
                .build();
    }

    private MetricArchiveImmutable randomArchive(int numId) {
        try (var archive = new MetricArchiveMutable()) {
            archive.setOwnerProjectIdEnum(EProjectId.SOLOMON);
            archive.setOwnerShardId(numId);
            archive.addRecord(randomPoint());
            return archive.toImmutableNoCopy();
        }
    }

    private RemoveShardParams randomParams() {
        return RemoveShardParams.newBuilder()
                .setNumId(ThreadLocalRandom.current().nextInt(1, Integer.MAX_VALUE))
                .setShardId("shard_id_" + ThreadLocalRandom.current().nextLong())
                .setProjectId("project_id_" + ThreadLocalRandom.current().nextLong())
                .build();
    }

    private RemoveShardFromStockpile remove(RemoveShardParams params, RemoveData process) {
        return new RemoveShardFromStockpile(retryConfig, stockpile, ForkJoinPool.commonPool(), params, process);
    }
}
