package ru.yandex.solomon.coremon.tasks.removeShard;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;

import javax.annotation.Nullable;

import com.google.protobuf.Any;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.coremon.api.task.RemoveShardParams;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.ExecutionContextStub;
import ru.yandex.solomon.scheduler.Permit;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoveShardTaskHandlerTest {

    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private StockpileClientStub stockpile;
    private InMemoryMetricsDaoFactory daoFactory;
    private SolomonConfHolder confHolder;
    private RemoveShardTaskHandler handler;

    @Before
    public void setUp() throws Exception {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
        stockpile = new StockpileClientStub(ForkJoinPool.commonPool());
        daoFactory = new InMemoryMetricsDaoFactory();
        confHolder = new SolomonConfHolder();
        handler = new RemoveShardTaskHandler(confHolder, stockpile, daoFactory, ForkJoinPool.commonPool(), timer);
    }

    @After
    public void tearDown() {
        if (timer != null) {
            timer.shutdownNow();
        }
    }

    @Test
    public void absentPermitConfNotLoaded() {
        assertNull(acquire(randomParams()));
    }

    @Test
    public void receivePermit() {
        confHolder.onConfigurationLoad(SolomonConfWithContext.EMPTY);
        assertNotNull(acquire(randomParams()));
    }

    @Test
    public void unableRemoveActiveShard() {
        var params = randomParams();
        initShard(params);
        assertNotNull(acquire(params));

        var context = context(params);
        execute(context);

        var fail = context.takeDoneEvent(ExecutionContextStub.Fail.class);
        assertNotNull(fail.throwable());
    }

    @Test
    public void removeMetrics() {
        confHolder.onConfigurationLoad(SolomonConfWithContext.EMPTY);
        var params = randomParams();
        assertNotNull(acquire(params));

        var metrics = generateMetrics(params.getNumId(), ThreadLocalRandom.current().nextInt(0, 10));
        var context = context(params);
        execute(context);
        var complete = context.takeDoneEvent(ExecutionContextStub.Complete.class);
        var result = RemoveShardTaskProto.result(complete.result());
        assertEquals(metrics.size(), result.getRemovedMetrics());
    }

    private void initShard(RemoveShardParams params) {
        var projects = List.of(Project.newBuilder()
                .setId(params.getProjectId())
                .setName(params.getProjectId())
                .setOwner("any")
                .build());

        var clusters = List.of(Cluster.newBuilder()
                .setProjectId(params.getProjectId())
                        .setId("cluster_id")
                        .setName("cluster_id")
                        .build());

        var services = List.of(Service.newBuilder()
                .setProjectId(params.getProjectId())
                .setId("service_id")
                .setName("service_id")
                .build());

        var shards = List.of(Shard.newBuilder()
                .setProjectId(params.getProjectId())
                .setId(params.getShardId())
                .setNumId(params.getNumId())
                .setClusterId("cluster_id")
                .setClusterName("cluster_id")
                .setServiceId("service_id")
                .setServiceName("service_id")
                .build());

        var conf = new SolomonRawConf(List.of(), projects, clusters, services, shards);
        confHolder.onConfigurationLoad(SolomonConfWithContext.create(conf));
    }

    private RemoveShardParams randomParams() {
        return RemoveShardParams.newBuilder()
                .setNumId(ThreadLocalRandom.current().nextInt(1, Integer.MAX_VALUE))
                .setShardId("shard_id_" + ThreadLocalRandom.current().nextLong())
                .setProjectId("project_id_" + ThreadLocalRandom.current().nextLong())
                .build();
    }

    @Nullable
    private Permit acquire(RemoveShardParams params) {
        return handler.acquire(UUID.randomUUID().toString(), Any.pack(params));
    }

    private List<CoremonMetric> generateMetrics(int numId, int count) {
        List<CoremonMetric> metrics = new ArrayList<>(count);
        var archive = randomArchive(numId);
        for (int index = 0; index < count; index++) {
            var metric = randomMetric();
            stockpile.addTimeSeries(metric.getShardId(), metric.getLocalId(), archive);
            metrics.add(metric);
        }
        daoFactory.create(numId, new InterningLabelAllocator()).add(metrics);
        return metrics;
    }

    private CoremonMetric randomMetric() {
        var random = ThreadLocalRandom.current();
        int shardId = StockpileShardId.random(1000, random);
        long localId = StockpileLocalId.random(random);
        var labels = Labels.of("key", "value-"+random.nextLong());
        return new FileCoremonMetric(shardId, localId, labels, MetricType.DGAUGE);
    }

    private MetricArchiveImmutable randomArchive(int numId) {
        try (var archive = new MetricArchiveMutable()) {
            archive.setOwnerProjectIdEnum(EProjectId.SOLOMON);
            archive.setOwnerShardId(numId);
            archive.addRecord(randomPoint());
            return archive.toImmutableNoCopy();
        }
    }

    private ExecutionContextStub context(RemoveShardParams params) {
        var task = Task.newBuilder()
                .setId(UUID.randomUUID().toString())
                .setType("remove_shard")
                .setExecuteAt(System.currentTimeMillis())
                .setParams(Any.pack(params))
                .build();

        return new ExecutionContextStub(task);
    }

    private void execute(ExecutionContext context) {
        handler.execute(context);
    }
}
