package ru.yandex.solomon.coremon.tasks.removeShard;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.protobuf.Any;
import com.google.protobuf.TextFormat;
import io.grpc.Status;
import io.grpc.Status.Code;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.RemoveShardParams;
import ru.yandex.coremon.api.task.RemoveShardProgress;
import ru.yandex.coremon.api.task.RemoveShardResult;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDao;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.ExecutionContextStub;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Cancel;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Complete;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Progress;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Reschedule;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoveShardTaskTest {

    @Rule
    public Timeout globalTimeout = Timeout.builder()
            .withTimeout(1, TimeUnit.MINUTES)
            .withLookingForStuckThread(true)
            .build();

    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private RetryConfig retryConfig;
    private StockpileClientStub stockpile;
    private InMemoryMetricsDao metricsDao;

    @Before
    public void setUp() throws Exception {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
        stockpile = new StockpileClientStub(ForkJoinPool.commonPool());
        metricsDao = new InMemoryMetricsDao();
        stockpile.setShardCount(1000);
        retryConfig = RetryConfig.DEFAULT
                .withNumRetries(Integer.MAX_VALUE)
                .withMaxDelay(0);
    }

    @Test
    public void nothingToRemove() {
        var context = context(randomParams(), RemoveShardProgress.getDefaultInstance());

        var proc = remove(context);
        proc.start().join();

        var result = RemoveShardTaskProto.result(context.takeDoneEvent(Complete.class).result());
        var expected = RemoveShardResult.newBuilder()
                .setRemovedMetrics(0)
                .build();

        assertEquals(expected, result);
    }

    @Test
    public void removeMetaAndData() {
        var params = randomParams();

        List<CoremonMetric> metrics = generateMetrics(params.getNumId(), 10);

        var context = context(params, RemoveShardProgress.getDefaultInstance());
        var proc = remove(context);
        proc.start().join();

        var result = RemoveShardTaskProto.result(context.takeDoneEvent(Complete.class).result());
        var expected = RemoveShardResult.newBuilder()
                .setRemovedMetrics(metrics.size())
                .build();
        assertEquals(expected, result);

        for (var metric : metrics) {
            var id = StockpileMetricId.toString(metric.getShardId(), metric.getLocalId());
            assertNull(id, stockpile.getTimeSeries(metric.getShardId(), metric.getLocalId()));
        }

        assertEquals(0, metricsDao.metrics().size());
    }

    @Test
    public void rescheduleOnError() {
        var params = randomParams();

        List<CoremonMetric> metrics = generateMetrics(params.getNumId(), 100);

        metrics.sort(Comparator.comparingInt(CoremonMetric::getShardId));
        int failedShardId = metrics.get(50).getShardId();

        stockpile.predefineStatusCode(failedShardId, EStockpileStatusCode.INTERNAL_ERROR);
        retryConfig = retryConfig.withNumRetries(3);

        var context = context(params, RemoveShardProgress.getDefaultInstance());
        var proc = remove(context);
        proc.start().join();

        var rescheduleEvent = context.takeDoneEvent(Reschedule.class);
        assertThat(rescheduleEvent.executeAt(), Matchers.greaterThan(System.currentTimeMillis() + 30_000L));

        var progress = RemoveShardTaskProto.progress(rescheduleEvent.progress());
        assertEquals(1, progress.getAttempt());
        assertNotEquals(Code.OK.value(), progress.getStatus().getCode());
        assertFalse(progress.getRemoveData().getComplete());
        assertNotEquals(failedShardId, progress.getRemoveData().getLastShardId());

        assertFalse(progress.getRemoveMeta().getComplete());
        assertEquals(0, progress.getRemoveMeta().getRemovedMetrics());
        assertEquals(metrics.size(), metricsDao.metrics().size());
    }

    @Test
    public void stopRemoveWhenCanceled() {
        var params = randomParams();

        var metrics = generateMetrics(params.getNumId(), ThreadLocalRandom.current().nextInt(10_000, 20_000));
        var context = context(params, RemoveShardProgress.getDefaultInstance());
        var proc = remove(context);

        metricsDao.beforeSupplier = () -> {
            if (metricsDao.metrics().size() != metrics.size()) {
                proc.close();
            }

            return CompletableFuture.completedFuture(null);
        };

        proc.start().join();
        context.takeEvent(Cancel.class, true);
    }

    @Test
    public void updateProgress() {
        var params = randomParams();

        generateMetrics(params.getNumId(), ThreadLocalRandom.current().nextInt(10_000, 20_000));
        var context = context(params, RemoveShardProgress.getDefaultInstance());
        var proc = remove(context);

        metricsDao.responseDelayMillis = 1000;
        RemoveShardProgress prev = RemoveShardProgress.getDefaultInstance();
        proc.start();
        for (int index = 0; index < 2; index++) {
            var event = awaitProgress(context);
            var progress = RemoveShardTaskProto.progress(event.progress());
            assertNotEquals(prev, progress);
            prev = progress;
            System.out.println(TextFormat.shortDebugString(progress));
        }
    }

    @Test
    public void stopRemoveWhenContextDone() throws InterruptedException {
        var params = randomParams();

        generateMetrics(params.getNumId(), ThreadLocalRandom.current().nextInt(1_000, 2_000));
        var context = context(params, RemoveShardProgress.getDefaultInstance());
        stockpile.beforeSupplier = () -> {
            context.markCompleted();
            return CompletableFuture.failedFuture(new RuntimeException("hi"));
        };
        var removeMeta = new AtomicBoolean(false);
        metricsDao.beforeSupplier = () -> {
            removeMeta.set(true);
            return completedFuture(null);
        };

        var proc = remove(context);
        var future = proc.start().thenApply(unused -> Status.OK).exceptionally(Status::fromThrowable);

        CountDownLatch sync = new CountDownLatch(1);
        future.whenComplete((ignore, e) -> sync.countDown());

        while (!sync.await(1, TimeUnit.MILLISECONDS)) {
            clock.passedTime(10, TimeUnit.SECONDS);
        }

        future.join();
        assertFalse(removeMeta.get());
    }

    private Progress awaitProgress(ExecutionContextStub context) {
        Progress progress;
        while ((progress = context.poolEvent(Progress.class, 1, TimeUnit.NANOSECONDS)) == null) {
            clock.passedTime(30, TimeUnit.SECONDS);
        }
        return progress;
    }

    private List<CoremonMetric> generateMetrics(int numId, int count) {
        List<CoremonMetric> metrics = new ArrayList<>(count);
        var archive = randomArchive(numId);
        for (int index = 0; index < count; index++) {
            var metric = randomMetric();
            stockpile.addTimeSeries(metric.getShardId(), metric.getLocalId(), archive);
            metrics.add(metric);
        }
        metricsDao.add(metrics);
        return metrics;
    }

    private CoremonMetric randomMetric() {
        var random = ThreadLocalRandom.current();
        int shardId = StockpileShardId.random(1000, random);
        long localId = StockpileLocalId.random(random);
        var labels = Labels.of("key", "value-"+random.nextLong());
        return new FileCoremonMetric(shardId, localId, labels, MetricType.DGAUGE);
    }

    private MetricArchiveImmutable randomArchive(int numId) {
        try (var archive = new MetricArchiveMutable()) {
            archive.setOwnerProjectIdEnum(EProjectId.SOLOMON);
            archive.setOwnerShardId(numId);
            archive.addRecord(randomPoint());
            return archive.toImmutableNoCopy();
        }
    }

    private RemoveShardParams randomParams() {
        return RemoveShardParams.newBuilder()
                .setNumId(ThreadLocalRandom.current().nextInt(1, Integer.MAX_VALUE))
                .setShardId("shard_id_" + ThreadLocalRandom.current().nextLong())
                .setProjectId("project_id_" + ThreadLocalRandom.current().nextLong())
                .build();
    }

    private ExecutionContextStub context(RemoveShardParams params, RemoveShardProgress progress) {
        var task = Task.newBuilder()
                .setId(UUID.randomUUID().toString())
                .setType("remove_shard")
                .setExecuteAt(System.currentTimeMillis())
                .setProgress(Any.pack(progress))
                .setParams(Any.pack(params))
                .build();

        return new ExecutionContextStub(task);
    }

    private RemoveShardTask remove(ExecutionContext context) {
        return new RemoveShardTask(
            retryConfig,
            context, RemoveShardTaskProto.params(context.task().params()), RemoveShardTaskProto.progress(context.task().progress()), ForkJoinPool.commonPool(), timer, stockpile,
            metricsDao
        );
    }
}
