package ru.yandex.solomon.coremon.meta;

import java.util.Arrays;
import java.util.Iterator;
import java.util.Spliterator;
import java.util.function.Consumer;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import javax.annotation.concurrent.NotThreadSafe;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.buffer.Unpooled;
import io.netty.util.AbstractReferenceCounted;
import io.netty.util.ReferenceCounted;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;


/**
 * @author Sergey Polovko
 */
@NotThreadSafe
public class CoremonMetricArray extends AbstractReferenceCounted implements AutoCloseable, MemMeasurable {

    private static final Logger logger = LoggerFactory.getLogger(CoremonMetricArray.class);

    private static final Object[] EMPTY_OBJECT_ARRAY = {};
    private static final Labels[] EMPTY_LABELS_ARRAY = {};

    //
    // Metrics struct format
    //
    // +---+---+---+---+---+---+---+---+
    // | t | . | s | s | c | c | c | c |
    // +---+---+---+---+---+---+---+---+
    // | l | l | l | l | l | l | l | l |
    // +---+---+---+---+---+---+---+---+
    // | p | p | p | p |
    // +---+---+---+---+
    //
    // where
    //   t - metric type
    //   s - stockpile shard id
    //   l - stockpile local id
    //   c - created at seconds
    //   p - last point seconds
    //   . - padding
    //

    private static final int SIZE_OF_METRIC = 20;
    private static final int OFFSET_TYPE = 0;
    private static final int OFFSET_SHARD_ID = 2;
    private static final int OFFSET_CREATED_AT = 4;
    private static final int OFFSET_LOCAL_ID = 8;
    private static final int OFFSET_LAST_POINT = 16;

    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(CoremonMetricArray.class);
    private static final MetricType[] TYPES = MetricType.values();
    private static final PooledByteBufAllocator ALLOCATOR = PooledByteBufAllocator.DEFAULT;

    private final int maxBufferCapacity;
    private int size;
    private Labels[] labels;
    private Object[] aggrMetrics;
    private ByteBuf bufferLeft;
    private ByteBuf bufferRight = Unpooled.EMPTY_BUFFER; // used only if we need allocate more than 2 GiB for buffer
                                                         // (more than ~ 107 M metrics)

    public CoremonMetricArray(int capacity) {
        this(capacity, Integer.MAX_VALUE / SIZE_OF_METRIC - 100);
    }

    public CoremonMetricArray(CoremonMetric... metrics) {
        this(metrics.length);
        for (CoremonMetric metric : metrics) {
            add(metric);
        }
    }

    public CoremonMetricArray(int capacity, int maxBufferCapacity) {
        if (capacity < 0 || capacity > 2L * maxBufferCapacity) {
            String msg = String.format("capacity(%d) must be >= 0 and < %d", capacity, 2L * maxBufferCapacity);
            throw new IllegalArgumentException(msg);
        }

        this.maxBufferCapacity = maxBufferCapacity;
        this.size = 0;
        this.labels = new Labels[capacity];
        this.aggrMetrics = new Object[capacity];
        try {
            this.bufferLeft = ALLOCATOR.directBuffer(Math.min(capacity, maxBufferCapacity) * SIZE_OF_METRIC);
            if (capacity > maxBufferCapacity) {
                this.bufferRight = ALLOCATOR.directBuffer((capacity - maxBufferCapacity) * SIZE_OF_METRIC);
            }
        } catch (Throwable e) {
            if (bufferLeft != null) {
                bufferLeft.release();
            }

            if (bufferRight != null) {
                bufferRight.release();
            }

            throw new RuntimeException(e);
        }
    }

    public CoremonMetricArray(CoremonMetricArray metrics) {
        this.maxBufferCapacity = metrics.maxBufferCapacity;
        this.size = metrics.size;
        this.labels = Arrays.copyOf(metrics.labels, metrics.size);
        this.aggrMetrics = Arrays.copyOf(metrics.aggrMetrics, metrics.size);
        try {
            this.bufferLeft = metrics.bufferLeft.copy(0, Math.min(metrics.size, maxBufferCapacity) * SIZE_OF_METRIC);
            if (metrics.size > maxBufferCapacity) {
                this.bufferRight = metrics.bufferRight.copy(0, (metrics.size - maxBufferCapacity) * SIZE_OF_METRIC);
            }
        } catch (Throwable e) {
            if (bufferLeft != null) {
                bufferLeft.release();
            }

            if (bufferRight != null) {
                bufferRight.release();
            }

            throw new RuntimeException(e);
        }
    }

    public void add(int stockpileShardId, long stockpileLocalId, Labels labels, int createdAtSeconds, MetricType type) {
        StockpileShardId.validate(stockpileShardId);
        StockpileLocalId.validate(stockpileLocalId);

        if (size >= capacity()) {
            resize(size + (size >>> 1) + 10);
        }

        final int idx = size++;
        final ByteBuf buffer = (idx < maxBufferCapacity) ? this.bufferLeft : bufferRight;
        final int offset = (idx % maxBufferCapacity) * SIZE_OF_METRIC;

        buffer.setByte(offset + OFFSET_TYPE, typeToByte(type));
        buffer.setShort(offset + OFFSET_SHARD_ID, (short) (stockpileShardId & 0xffff));
        buffer.setInt(offset + OFFSET_CREATED_AT, createdAtSeconds);
        buffer.setLong(offset + OFFSET_LOCAL_ID, stockpileLocalId);
        buffer.setInt(offset + OFFSET_LAST_POINT, CoremonMetric.UNKNOWN_LAST_POINT_SECONDS);
        this.labels[idx] = labels;
        aggrMetrics[idx] = null;
    }

    public void add(CoremonMetric s) {
        StockpileShardId.validate(s.getShardId());
        StockpileLocalId.validate(s.getLocalId());

        if (size >= capacity()) {
            resize(size + (size >>> 1) + 10);
        }

        final int idx = size++;
        final ByteBuf buffer = (idx < maxBufferCapacity) ? this.bufferLeft : bufferRight;
        final int offset = (idx % maxBufferCapacity) * SIZE_OF_METRIC;

        buffer.setByte(offset + OFFSET_TYPE, typeToByte(s.getType()));
        buffer.setShort(offset + OFFSET_SHARD_ID, (short) (s.getShardId() & 0xffff));
        buffer.setInt(offset + OFFSET_CREATED_AT, s.getCreatedAtSeconds());
        buffer.setLong(offset + OFFSET_LOCAL_ID, s.getLocalId());
        buffer.setInt(offset + OFFSET_LAST_POINT, CoremonMetric.UNKNOWN_LAST_POINT_SECONDS);
        labels[idx] = s.getLabels();
        aggrMetrics[idx] = s.getAggrMetrics();
    }

    public void addAll(CoremonMetricArray metrics) {
        if (!metrics.isEmpty()) {
            addAll(metrics, 0, metrics.size());
        }
    }

    public void addAll(CoremonMetricArray metrics, int beginInclusive, int endExclusive) {
        if (beginInclusive >= endExclusive) {
            String msg = String.format("beginInclusive(%d) >= endExclusive(%d)", beginInclusive, endExclusive);
            throw new IllegalArgumentException(msg);
        }

        int addSize = endExclusive - beginInclusive;
        if (capacity() - size() < addSize) {
            resize(size() + addSize);
        }

        int dstIdx = size;
        for (int srcIdx = beginInclusive; srcIdx < endExclusive; ) {
            ByteBuf srcBuf = (srcIdx < metrics.maxBufferCapacity) ? metrics.bufferLeft : metrics.bufferRight;
            ByteBuf dstBuf = (dstIdx < maxBufferCapacity) ? bufferLeft : bufferRight;

            int sizeToCopy = Math.min(Math.min(
                metrics.maxBufferCapacity - (srcIdx % metrics.maxBufferCapacity),
                maxBufferCapacity - (dstIdx % maxBufferCapacity)),
                endExclusive - srcIdx);

            int srcOffset = (srcIdx % metrics.maxBufferCapacity) * SIZE_OF_METRIC;
            int dstOffset = (dstIdx % maxBufferCapacity) * SIZE_OF_METRIC;

            dstBuf.setBytes(dstOffset, srcBuf, srcOffset, sizeToCopy * SIZE_OF_METRIC);

            srcIdx += sizeToCopy;
            dstIdx += sizeToCopy;
        }

        System.arraycopy(metrics.labels, beginInclusive, labels, size, addSize);
        System.arraycopy(metrics.aggrMetrics, beginInclusive, aggrMetrics, size, addSize);

        size += addSize;
    }

    public void shrinkToFit() {
        if (size != capacity()) {
            resize(size);
        }
    }

    public boolean isEmpty() {
        return size == 0;
    }

    public int size() {
        return size;
    }

    public int capacity() {
        return labels.length;
    }

    public void clear() {
        size = 0;
        Arrays.fill(labels, null);
        Arrays.fill(aggrMetrics, null);
        bufferLeft.clear();
        bufferRight.clear();
    }

    public CoremonMetric getWithRetain(int i) {
        if (i < 0 || i >= size) {
            throw new IndexOutOfBoundsException("index(" + i + ") is out of bound [0.." + size + ')');
        }
        return new CoremonMetricProxy(i, true);
    }

    public CoremonMetric get(int i) {
        if (i < 0 || i >= size) {
            throw new IndexOutOfBoundsException("index(" + i + ") is out of bound [0.." + size + ')');
        }
        return new CoremonMetricProxy(i, false);
    }

    public Iterator<Labels> labelsIterator() {
        return new LabelsIterator();
    }

    public Stream<Labels> streamLabels() {
        return Arrays.stream(labels, 0, size);
    }

    public Stream<CoremonMetric> stream() {
        return StreamSupport.stream(new SpliteratorImpl(), false);
    }

    public int getShardId(int i) {
        ByteBuf buffer = this.bufferLeft;
        if (i >= maxBufferCapacity) {
            i -= maxBufferCapacity;
            buffer = this.bufferRight;
        }
        return Short.toUnsignedInt(buffer.getShort(idx(i, OFFSET_SHARD_ID)));
    }

    public long getLocalId(int i) {
        ByteBuf buffer = this.bufferLeft;
        if (i >= maxBufferCapacity) {
            i -= maxBufferCapacity;
            buffer = this.bufferRight;
        }
        return buffer.getLong(idx(i, OFFSET_LOCAL_ID));
    }

    public Labels getLabels(int i) {
        return labels[i];
    }

    public int getCreatedAtSeconds(int i) {
        ByteBuf buffer = this.bufferLeft;
        if (i >= maxBufferCapacity) {
            i -= maxBufferCapacity;
            buffer = this.bufferRight;
        }
        return buffer.getInt(idx(i, OFFSET_CREATED_AT));
    }

    public MetricType getType(int i) {
        ByteBuf buffer = this.bufferLeft;
        if (i >= maxBufferCapacity) {
            i -= maxBufferCapacity;
            buffer = this.bufferRight;
        }
        return byteToType(buffer.getByte(idx(i, OFFSET_TYPE)));
    }

    public void setType(int i, MetricType type) {
        ByteBuf buffer = this.bufferLeft;
        if (i >= maxBufferCapacity) {
            i -= maxBufferCapacity;
            buffer = this.bufferRight;
        }
        buffer.setByte(idx(i, OFFSET_TYPE), typeToByte(type));
    }

    public int getLastPointSeconds(int i) {
        ByteBuf buffer = this.bufferLeft;
        if (i >= maxBufferCapacity) {
            i -= maxBufferCapacity;
            buffer = this.bufferRight;
        }
        return buffer.getInt(idx(i, OFFSET_LAST_POINT));
    }

    public void setLastPointSeconds(int i, int seconds) {
        ByteBuf buffer = this.bufferLeft;
        if (i >= maxBufferCapacity) {
            i -= maxBufferCapacity;
            buffer = this.bufferRight;
        }
        buffer.setInt(idx(i, OFFSET_LAST_POINT), seconds);
    }

    private void resize(int newCapacity) {
        // bufferLeft resizing needs correct values of reader and writer index
        int capacity = capacity();
        if (capacity <= maxBufferCapacity && newCapacity <= maxBufferCapacity) {
            //
            // (1) growing or shrinking left buffer within MAX_CAPACITY limit
            //
            bufferLeft.setIndex(0, capacity * SIZE_OF_METRIC);
            bufferLeft.capacity(ALLOCATOR.calculateNewCapacity(newCapacity * SIZE_OF_METRIC, bufferLeft.maxCapacity()));
        } else if (capacity > maxBufferCapacity && newCapacity <= maxBufferCapacity) {
            //
            // (2) shrinking left buffer and release right buffer
            //
            bufferLeft.setIndex(0, maxBufferCapacity * SIZE_OF_METRIC);
            bufferLeft.capacity(ALLOCATOR.calculateNewCapacity(newCapacity * SIZE_OF_METRIC, bufferLeft.maxCapacity()));
            bufferRight.release();
            bufferRight = Unpooled.EMPTY_BUFFER;
        } else if (capacity <= maxBufferCapacity) {
            //
            // (3) growing left buffer and allocate right buffer
            //
            bufferLeft.setIndex(0, capacity * SIZE_OF_METRIC);
            bufferLeft.capacity(ALLOCATOR.calculateNewCapacity(maxBufferCapacity * SIZE_OF_METRIC, bufferLeft.maxCapacity()));
            bufferRight = ALLOCATOR.directBuffer((newCapacity - maxBufferCapacity) * SIZE_OF_METRIC);
        } else {
            //
            // (4) growing or shrinking right buffer
            //
            bufferRight.setIndex(0, (capacity - maxBufferCapacity) * SIZE_OF_METRIC);
            bufferRight.capacity(ALLOCATOR.calculateNewCapacity((newCapacity - maxBufferCapacity) * SIZE_OF_METRIC, bufferLeft.maxCapacity()));
        }

        labels = Arrays.copyOf(labels, newCapacity);
        aggrMetrics = Arrays.copyOf(aggrMetrics, newCapacity);
    }

    private static byte typeToByte(MetricType type) {
        return (byte) type.ordinal();
    }

    private static MetricType byteToType(byte b) {
        return TYPES[(int) b];
    }

    private static int idx(int index, int fieldOffset) {
        return index * SIZE_OF_METRIC + fieldOffset;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += bufferLeft.capacity();
        size += bufferRight.capacity();
        size += MemoryCounter.arrayObjectSize(labels);
        size += MemoryCounter.arrayObjectSize(aggrMetrics);
        return size;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder()
            .append("{refCnt=").append(refCnt())
            .append(", size=").append(size())
            .append(", capacity=").append(capacity())
            .append(", bufferLeft.capacity=").append(bufferLeft.capacity())
            .append(", bufferLeft.refCnt=").append(bufferLeft.refCnt());

        if (bufferRight == Unpooled.EMPTY_BUFFER) {
            sb.append(", bufferRight=EMPTY");
        } else {
            sb.append(", bufferRight.capacity=").append(bufferRight.capacity());
            sb.append(", bufferRight.refCnt=").append(bufferRight.refCnt());
        }

        return sb.append('}').toString();
    }

    @Override
    protected void deallocate() {
        bufferLeft.release();
        bufferLeft = Unpooled.EMPTY_BUFFER;
        if (bufferRight != null) {
            bufferRight.release();
            bufferRight = Unpooled.EMPTY_BUFFER;
        }
        labels = EMPTY_LABELS_ARRAY;
        aggrMetrics = EMPTY_OBJECT_ARRAY;
        size = 0;
    }

    @Override
    public ReferenceCounted touch(Object hint) {
        return this;
    }

    @Override
    public CoremonMetricArray retain() {
        super.retain();
        return this;
    }

    @Override
    public void close() {
        release();
    }

    public void closeSilent() {
        try {
            close();
        } catch (Throwable t) {
            logger.warn("cannot release {} metrics: {}", size(), t.getMessage());
        }
    }

    /**
     * LABELS ITERATOR
     */
    private class LabelsIterator implements Iterator<Labels> {
        private int i = 0;

        @Override
        public boolean hasNext() {
            return i < CoremonMetricArray.this.size;
        }

        @Override
        public Labels next() {
            return CoremonMetricArray.this.labels[i++];
        }
    }

    private static final long CoremonMetricProxy_SELF_SIZE = MemoryCounter.objectSelfSizeLayout(CoremonMetricProxy.class);

    /**
     * COREMON METRIC PROXY
     */
    private final class CoremonMetricProxy implements CoremonMetric {
        private final int i;
        private final boolean retained;

        CoremonMetricProxy(int i, boolean retained) {
            this.i = i;
            this.retained = retained;
            if (retained) {
                CoremonMetricArray.this.retain();
            }
        }

        @Override
        public void close() {
            if (retained) {
                CoremonMetricArray.this.release();
            }
        }

        @Override
        public Object getAggrMetrics() {
            return CoremonMetricArray.this.aggrMetrics[i];
        }

        @Override
        public void setAggrMetrics(Object aggrMetrics) {
            CoremonMetricArray.this.aggrMetrics[i] = aggrMetrics;
        }

        @Override
        public boolean isMemOnly() {
            return false;
        }

        @Override
        public int getLastPointSeconds() {
            return CoremonMetricArray.this.getLastPointSeconds(i);
        }

        @Override
        public void setLastPointSeconds(int lastPointSeconds) {
            CoremonMetricArray.this.setLastPointSeconds(i, lastPointSeconds);
        }

        @Override
        public int getShardId() {
            return CoremonMetricArray.this.getShardId(i);
        }

        @Override
        public long getLocalId() {
            return CoremonMetricArray.this.getLocalId(i);
        }

        @Override
        public Labels getLabels() {
            return CoremonMetricArray.this.getLabels(i);
        }

        @Override
        public MetricType getType() {
            return CoremonMetricArray.this.getType(i);
        }

        @Override
        public void setType(MetricType type) {
            CoremonMetricArray.this.setType(i, type);
        }

        @Override
        public int getCreatedAtSeconds() {
            return CoremonMetricArray.this.getCreatedAtSeconds(i);
        }

        @Override
        public long memorySizeIncludingSelf() {
            return CoremonMetricProxy_SELF_SIZE;
        }
    }

    /**
     * SPLITERATOR IMPL
     */
    private final class SpliteratorImpl implements Spliterator<CoremonMetric> {
        private int i = 0;

        @Override
        public void forEachRemaining(Consumer<? super CoremonMetric> action) {
            while (i < CoremonMetricArray.this.size) {
                action.accept(CoremonMetricArray.this.get(i++));
            }
        }

        @Override
        public boolean tryAdvance(Consumer<? super CoremonMetric> action) {
            if (i < CoremonMetricArray.this.size) {
                action.accept(CoremonMetricArray.this.get(i++));
                return true;
            }
            return false;
        }

        @Override
        public Spliterator<CoremonMetric> trySplit() {
            return null;
        }

        @Override
        public long estimateSize() {
            return CoremonMetricArray.this.size;
        }

        @Override
        public int characteristics() {
            return Spliterator.ORDERED | Spliterator.IMMUTABLE | Spliterator.SIZED;
        }
    }
}
