package ru.yandex.solomon.coremon.meta;

import java.util.ArrayList;
import java.util.List;

import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.db.StockpileMetricIdProvider;
import ru.yandex.solomon.coremon.meta.file.FileMetricsCollection;
import ru.yandex.solomon.slog.ResolvedLogMetricsBuilderImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaRecord;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class CoremonMetricsResolver {

    public static ResolveResult resolve(ByteString meta, MetricsCollection<CoremonMetric> metrics, StockpileMetricIdProvider metricIdProvider) {
        return resolve(meta, metrics, metricIdProvider, Labels.allocator);
    }

    public static ResolveResult resolve(ByteString meta, MetricsCollection<CoremonMetric> metrics, StockpileMetricIdProvider metricIdProvider, LabelAllocator allocator) {
        var record = new UnresolvedLogMetaRecord();
        var now = InstantUtils.currentTimeSeconds();
        var result = new ResolveResult();
        try (var it = new UnresolvedLogMetaIteratorImpl(ByteStrings.toByteBuf(meta), allocator)) {
            while (it.next(record)) {
                try (var metric = metrics.getOrNull(record.labels)) {
                    if (metric == null) {
                        var metricId = metricIdProvider.metricId(record.labels);
                        result.unresolved.add(metricId.getShardId(), metricId.getLocalId(), record.labels, now, record.type);
                    } else if (metric.getType() != record.type) {
                        result.unresolved.add(metric.getShardId(), metric.getLocalId(), record.labels, metric.getCreatedAtSeconds(), record.type);
                    } else {
                        result.resolved.add(metric.getShardId(), metric.getLocalId(), record.labels, now, record.type);
                    }
                }
            }

            return result;
        } catch (Throwable e) {
            result.close();
            throw new RuntimeException(e);
        }
    }

    public static void resolveUnresolved(CoremonMetricArray resolved, CoremonMetricArray unresolved, FileMetricsCollection metrics) {
        for (int index = 0; index < unresolved.size(); index++) {
            try (var metric = metrics.getOrNull(unresolved.getLabels(index))) {
                if (metric == null) {
                    continue;
                }
                resolved.add(metric.getShardId(), metric.getLocalId(), metric.getLabels(), metric.getCreatedAtSeconds(), unresolved.getType(index));
            }
        }
    }

    public static CoremonMetricArray combineUnresolved(List<ResolveResult> sources, int capacity) {
        CoremonMetricArray result = new CoremonMetricArray(capacity);
        try {
            for (var source : sources) {
                result.addAll(source.unresolved);
            }
            return result;
        } catch (Throwable e) {
            result.close();
            throw new RuntimeException(e);
        }
    }

    public static ByteString pack(int numId, CoremonMetricArray array) {
        try (var builder = new ResolvedLogMetricsBuilderImpl(numId, CompressionAlg.LZ4, ByteBufAllocator.DEFAULT)) {
            for (int index = 0; index < array.size(); index++) {
                builder.onMetric(
                    array.getType(index),
                    array.getLabels(index),
                    array.getShardId(index),
                    array.getLocalId(index));
            }

            var buffer = builder.build();
            try {
                return UnsafeByteOperations.unsafeWrap(ByteBufUtil.getBytes(buffer));
            } finally {
                buffer.release();
            }
        }
    }

    public static List<ByteString> pack(int numId, List<ResolveResult> sources) {
        var result = new ArrayList<ByteString>(sources.size());
        for (var source : sources) {
            result.add(CoremonMetricsResolver.pack(numId, source.resolved));
        }
        return result;
    }

    public static void close(List<ResolveResult> list) {
        for (var r : list) {
            r.close();
        }
    }

    public static class ResolveResult implements AutoCloseable {
        public final CoremonMetricArray resolved = new CoremonMetricArray();
        public final CoremonMetricArray unresolved = new CoremonMetricArray();

        @Override
        public void close() {
            resolved.close();
            unresolved.close();
        }
    }
}
