package ru.yandex.solomon.coremon.meta;

import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class FileCoremonMetric implements CoremonMetric {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(FileCoremonMetric.class);

    private static final AtomicIntegerFieldUpdater<FileCoremonMetric> lastPointSecondsUpdater =
        AtomicIntegerFieldUpdater.newUpdater(FileCoremonMetric.class, "lastPointSeconds");

    private final int stockpileShardId;
    private final long stockpileLocalId;
    private final Labels labels;
    private final int createdAtSeconds; // we have enough time (till 2038-01-19) to migrate from this scheme

    private MetricType type;
    private volatile int lastPointSeconds = CoremonMetric.UNKNOWN_LAST_POINT_SECONDS;
    private volatile Object aggrMetrics;

    public FileCoremonMetric(
        int stockpileShardId, long stockpileLocalId,
        Labels labels,
        MetricType type)
    {
        this(stockpileShardId, stockpileLocalId, labels, InstantUtils.currentTimeSeconds(), type);
    }

    public FileCoremonMetric(CoremonMetric s) {
        this(s.getShardId(), s.getLocalId(), s.getLabels(), s.getCreatedAtSeconds(), s.getType());
        this.lastPointSeconds = s.getLastPointSeconds();
    }

    public FileCoremonMetric(
        int stockpileShardId, long stockpileLocalId,
        Labels labels,
        int createdAtSeconds, MetricType type)
    {
        this.stockpileShardId = stockpileShardId;
        this.stockpileLocalId = stockpileLocalId;
        this.labels = labels;
        this.createdAtSeconds = createdAtSeconds;
        this.type = type;

        if ((stockpileLocalId != 0) != (stockpileShardId != 0)) {
            throw new IllegalArgumentException("invalid stockpile ids for metric: " + labels);
        }

        if (stockpileLocalId != 0) {
            // validate
            StockpileShardId.validate(stockpileShardId);
            StockpileLocalId.validate(stockpileLocalId);
        }
    }

    public static FileCoremonMetric of(CoremonMetric metric) {
        if (metric instanceof FileCoremonMetric) {
            return (FileCoremonMetric) metric;
        } else {
            return new FileCoremonMetric(metric);
        }
    }

    @Override
    public int getCreatedAtSeconds() {
        return createdAtSeconds;
    }

    @Override
    public MetricType getType() {
        return type;
    }

    @Override
    public void setType(MetricType type) {
        this.type = type;
    }

    @Override
    public int getLastPointSeconds() {
        return lastPointSecondsUpdater.get(this);
    }

    @Override
    public void setLastPointSeconds(int lastPointSeconds) {
        int prev, next;
        do {
            prev = lastPointSecondsUpdater.get(this);
            next = Math.max(prev, lastPointSeconds);
        } while (!lastPointSecondsUpdater.compareAndSet(this, prev, next));
    }

    @Override
    public int getShardId() {
        return stockpileShardId;
    }

    @Override
    public long getLocalId() {
        return stockpileLocalId;
    }

    @Override
    public Labels getLabels() {
        return labels;
    }

    @Override
    public boolean isMemOnly() {
        return false;
    }

    @Override
    public Object getAggrMetrics() {
        return aggrMetrics;
    }

    @Override
    public void setAggrMetrics(Object aggrMetrics) {
        this.aggrMetrics = aggrMetrics;
    }

    @Override
    public String toString() {
        return "FileCoremonMetric{" +
            "stockpileShardId=" + stockpileShardId +
            ", stockpileLocalId=" + stockpileLocalId +
            ", labelListSorted=" + labels +
            ", type=" + type +
            ", lastPointSeconds=" + lastPointSeconds +
            '}';
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        FileCoremonMetric metric = (FileCoremonMetric) o;

        if (stockpileShardId != metric.stockpileShardId) return false;
        if (stockpileLocalId != metric.stockpileLocalId) return false;
        if (createdAtSeconds != metric.createdAtSeconds) return false;
        if (!labels.equals(metric.labels)) return false;
        if (type != metric.type) return false;
        // NOTE: lastPointSeconds and aggrMetrics intentionally not used
        return true;
    }

    @Override
    public int hashCode() {
        int result = stockpileShardId;
        result = 31 * result + (int) (stockpileLocalId ^ (stockpileLocalId >>> 32));
        result = 31 * result + labels.hashCode();
        result = 31 * result + createdAtSeconds;
        result = 31 * result + type.hashCode();
        // NOTE: lastPointSeconds and aggrMetrics intentionally not used
        return result;
    }

    @Override
    public void close() {
        // nop
    }
}
