package ru.yandex.solomon.coremon.meta.db;

import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.ints.Int2IntOpenHashMap;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.solomon.staffOnly.html.HtmlWriterWithCommonLibraries;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileMetricIdFactory {
    private static final String UNKNOWN_HOST = "<unknown>";

    /**
     * this constant will allow us to keep first ~ 65M metrics close to each other according to their 'host' label,
     * before we begin randomly distribute all metrics across all stockpile shards.
     */
    private static final int MAX_METRICS_PER_STOCKPILE_SHARD = 16_000;

    // information about how many metrics of this shard we put in each Stockpile shards
    private final ShardIdToAtomicInteger stockpileShardsLoad = new ShardIdToAtomicInteger();
    private final ConcurrentHashMap<String, StockpileMetricIdProvider> byHostStockpileShard = new ConcurrentHashMap<>();

    private final int numId;
    private final StockpileClient stockpileClient;
    private volatile boolean statsLoaded;

    public StockpileMetricIdFactory(int numId, StockpileClient stockpileClient) {
        this.numId = numId;
        this.stockpileClient = stockpileClient;
    }

    public void updateStats(Int2IntOpenHashMap stats) {
        if (statsLoaded) {
            return;
        }

        stockpileShardsLoad.set(stats);
        statsLoaded = true;
    }

    public StockpileMetricIdProvider forHost(@Nullable String host) {
        if (host == null) {
            host = UNKNOWN_HOST;
        }
        return byHostStockpileShard.computeIfAbsent(host, h -> new StockpileMetricIdProviderStatsAware(
                numId, h, MAX_METRICS_PER_STOCKPILE_SHARD, stockpileShardsLoad, stockpileClient));
    }

    public StockpileMetricIdProvider forLabels() {
        return new StockpileMetricIdProviderHash(stockpileShardsLoad, stockpileClient);
    }

    @ExtraContent("Metrics count per Stockpile shard")
    private void shardsPage(ExtraContentParam p) {
        // (count, shardId)
        Tuple2List<Integer, Integer> perShardCount = Tuple2List.arrayList();
        int[] countByShardId = stockpileShardsLoad.array();
        for (int index = 0; index < countByShardId.length; index++) {
            int count = countByShardId[index];
            if (count > 0) {
                perShardCount.add(count, index + 1);
            }
        }

        HtmlWriterWithCommonLibraries hw = p.getHtmlWriter();
        hw.pre(() -> {
            hw.writeRaw(String.format("%15s", "metrics count"));
            hw.writeRaw(" ");
            hw.writeRaw(String.format("%5s", "shard"));
            hw.nl();

            hw.writeRaw(StringUtils.repeat('-', 15));
            hw.writeRaw(" ");
            hw.writeRaw(StringUtils.repeat('-', 5));
            hw.nl();

            for (Tuple2<Integer, Integer> t : perShardCount.sortedBy1Desc()) {
                hw.writeRaw(String.format("%15d", t._1));
                hw.writeRaw(" ");
                hw.writeRaw(String.format("%5d", t._2));
                hw.nl();
            }
        });
    }
}
